/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QVariant>
#include <QtCore>
#include <QtGui>
#ifdef CWRT_BUILDING_TENONE
#include <HbMenu>
#endif
#include "qgraphicswebview.h"
#include "webjsmenu.h"
#include "webjsmenuitem.h"
#include "wrtwidgetcontainer.h"

using namespace WRT;

#ifndef QT_NO_DEBUG
// print UI menus to log

#ifdef CWRT_BUILDING_TENONE
static void printMenu(HbMenu* menu, int level)
{

}
#endif

static void printMenu(QMenu* menu, int level)
{
    QString tab = "";
    QString tabChild = "\t";
    switch (level)
    {
        case 0:
            break;
        case 1:
        default:
            tab = "\t";
            tabChild = "\t\t";
            break;
        case 2:
            tab = "\t\t";
            tabChild = "\t\t\t";
            break;
        case 3:
            tab = "\t\t\t";
            tabChild = "\t\t\t\t";
            break;
        case 4:
            tab = "\t\t\t\t";
            tabChild = "\t\t\t\t\t";
            break;
    }
    QString print = tab + menu->title();
    qDebug() << print;
    QObject* item = NULL;
    QObjectList children = menu->children();
    int count = children.count();
    for (int i=0; i<count; i++)
    {
        item = children.at(i);
        if (item->inherits("QAction"))
        {
            QAction* action = ((QAction*)item);
            QString text = action->text();
            if (text != ((QMenu*)(action->parentWidget()))->title())
            {
                QString printChild = tabChild + text;
                qDebug() << printChild;
            }
        }
        else if (item->inherits("QMenu"))
        {
            printMenu((QMenu*)item, level+1);
        }
    }
}
#endif


// LOCAL FUNCTION PROTOTYPES

// ----------------------------------------------------------------------------
// WebJSMenuPrivate::
//
//
//
// ----------------------------------------------------------------------------
//
WebJSMenuPrivate::WebJSMenuPrivate(WidgetContainer* wrtWidgetContainer) :
                                        m_wrtWidgetContainer(wrtWidgetContainer)
                                       ,m_SKvisible(true)
{
}

WebJSMenuPrivate::~WebJSMenuPrivate()
{
    for (int i=0; i<m_children.count(); i++)
    {
        WebJSMenuItem* menuItem = m_children.at(i);
        delete menuItem;
    }
    m_children.clear();
}

// ----------------------------------------------------------------------------
// WebJSMenu::
//
//
//
// ----------------------------------------------------------------------------
//
WebJSMenu::WebJSMenu(WidgetContainer* wrtWidgetContainer)
{
    d = new WebJSMenuPrivate(wrtWidgetContainer);
}

WebJSMenu::~WebJSMenu()
{
    delete d;
}

bool WebJSMenu::append(double id, const QString& label)
{
    // ensure unique id and label
    if (getMenuItemById(id) != NULL || getMenuItemByName(label) != NULL)
        return false;

    WebJSMenuItem* mi = new WebJSMenuItem(this, id, label);
    d->m_children.append(mi);

    return true;
}

void WebJSMenu::remove(double id)
{
    int count = d->m_children.count();
    for (int i=0; i<count; i++)
    {
        WebJSMenuItem* menuItem = d->m_children.at(i);
        if (menuItem->id() == id)
        {
            d->m_children.removeAt(i);
            delete menuItem;
            break;
        }
    }
}

void WebJSMenu::clear()
{
    int count = d->m_children.count();
    for (int i=0; i<count; i++)
    {
        WebJSMenuItem* menuItem = d->m_children.at(i);
        menuItem->clear();
        delete menuItem;
    }
    d->m_children.clear();
}

void WebJSMenu::setRightSoftkeyLabel(const QString& label)
{
    d->m_RSKlabel = label;
    emit softkeysChanged();
}

void WebJSMenu::setLeftSoftkeyLabel(const QString& label)
{
    d->m_LSKlabel = label;
    emit softkeysChanged();
}

void WebJSMenu::showSoftkeys()
{
    d->m_SKvisible = true;
    emit softkeysChanged();
}

void WebJSMenu::hideSoftkeys()
{
    d->m_SKvisible = false;
    emit softkeysChanged();
}

QString WebJSMenu::rightSoftkeyLabel()
{
    return d->m_RSKlabel;
}

QString WebJSMenu::leftSoftkeyLabel()
{
    return d->m_LSKlabel;
}

bool WebJSMenu::isSoftkeyVisible()
{
    return d->m_SKvisible;
}

bool WebJSMenu::appendToMenuItem(int parentId, double childId, const QString& childLabel)
{
    // ensure unique id and label
    if (getMenuItemById(childId) != NULL || getMenuItemByName(childLabel) != NULL)
        return false;

    WebJSMenuItem* child = new WebJSMenuItem(this, childId, childLabel);
    WebJSMenuItem* parent = getMenuItemById(parentId);
    parent->append(child);

    return true;
}

void WebJSMenu::removeFromMenuItem(int parentId, double childId)
{
    WebJSMenuItem* parent = getMenuItemById(parentId);
    parent->remove(childId);
}

void WebJSMenu::setDimmed(double id, bool dimmed)
{
    WebJSMenuItem* item = getMenuItemById(id);
    if (item)
        item->setDimmed(dimmed);
}

bool WebJSMenu::isExisting(double id, const QString& label)
{
    bool exist = false;
    WebJSMenuItem* itemById = getMenuItemById(id);
    WebJSMenuItem* itemByName = getMenuItemByName(label);
    if (itemById != NULL || itemByName != NULL)
        exist = true;

    return exist;
}

WebJSMenuItem* WebJSMenu::getMenuItemById(double id)
{
    WebJSMenuItem* item = NULL;
    int count = d->m_children.count();
    for (int i=0; i<count; i++)
    {
        item = d->m_children.at(i)->getMenuItemById(id);
        if (item != NULL)
            break;
    }
    return item;
}

WebJSMenuItem* WebJSMenu::getMenuItemByName(const QString& label)
{
    WebJSMenuItem* item = NULL;
    int count = d->m_children.count();
    for (int i=0; i<count; i++)
    {
        item = d->m_children.at(i)->getMenuItemByName(label);
        if (item != NULL)
            break;
    }
    return item;
}

void WebJSMenu::printArray()
{
    qDebug() << "menu";
    WebJSMenuItem* item = NULL;
    int count = d->m_children.count();
    for (int i=0; i<count; i++)
    {
        item = d->m_children.at(i);
        item->printArray(1);
    }
}

void WebJSMenu::printMessage(const QString& message)
{
    qDebug() << message;
}

bool WebJSMenu::eventFilter(QObject *obj, QEvent *event)
{
    bool handled(false);

    if (((d->m_wrtWidgetContainer->webWidget()
      && d->m_wrtWidgetContainer->webWidget()->isVisible())
      ||(d->m_wrtWidgetContainer->graphicsWebView()
      && d->m_wrtWidgetContainer->graphicsWebView()->isVisible()))
      && ((event->type() == QEvent::MouseButtonPress) ||
        (event->type() == QEvent::MouseMove) ||
        (event->type() == QEvent::MouseButtonRelease) ||
        (event->type() == QEvent::MouseButtonDblClick) ||
        (event->type() == QEvent::KeyPress))) {
        // convert first character of object name to lower case
        // if the object has no name, use default action instead
        QString objName(obj->objectName());
        bool showMenu(true);
        if (objName.length() > 0) {
            objName = ((objName.length() > 1) ? objName.at(0).toLower() + objName.mid(1) : objName.at(0).toLower());
            qDebug() << "WebJSMenu::eventFilter new object name " << objName;
            QString jscript("menu.");
            jscript.append(objName);
            jscript.append("Callback == null;");
            qDebug() << "WebJSMenu::eventFilter executing script " << jscript;
            showMenu = d->m_wrtWidgetContainer->evaluateJavaScript(jscript).toBool();
            qDebug() << "WebJSMenu::eventFilter check result " << showMenu;
        }

        if (!showMenu) {
            // just ignore press and move. catch button press on release
            if ((event->type() != QEvent::MouseButtonPress) && (event->type() != QEvent::MouseMove)) {
                QString jscript("menu.");
                jscript.append(objName);
                jscript.append("Callback()");
                qDebug() << "WebJSMenu::eventFilter executing script " << jscript;
                d->m_wrtWidgetContainer->evaluateJavaScript(jscript);
            }
            qDebug() << "WebJSMenu::eventFilter handled!";
            handled = true;
        }
    }
    return (handled ? true : QObject::eventFilter(obj, event));
}

void WebJSMenu::onSelectMenuItem(double id)
{
    // the script looks like
    // var item = menu.getMenuItemById(id);
    // item.onSelect(id);
    QString idStr;
    idStr.setNum(id);
    QString script = "var item = menu.getMenuItemById(";
    script.append(idStr);
    script.append("); if ((typeof(item.onSelect) != 'undefined') && (item.onSelect != null)){item.onSelect(");
    script.append(idStr);
    script.append(");}");
    // qDebug() << script;
    d->m_wrtWidgetContainer->evaluateJavaScript(script);
}

void WebJSMenu::onShowMenu()
{
    d->m_wrtWidgetContainer->evaluateJavaScript("if ((typeof(menu.onShow) != 'undefined') && (menu.onShow != null)){menu.onShow();}");
}

#ifdef CWRT_BUILDING_TENONE
HbMenu *WebJSMenu::constructUIMenu(HbMenu* uiMenu)
{
     onShowMenu();
     int count = d->m_children.count();
     if (count == 0)
         return NULL;

     if (!uiMenu)
         uiMenu = new HbMenu("Menu");

     for (int i=0; i<count; i++)
     {
         WebJSMenuItem* menuItem = d->m_children.at(i);
         menuItem->constructUIMenu(uiMenu);
     }

 #ifndef QT_NO_DEBUG
     ::printMenu(uiMenu, 0);
 #endif
     return uiMenu;

}
#endif

QMenu* WebJSMenu::constructUIMenu(QMenu* uiMenu)
{
    onShowMenu();
    int count = d->m_children.count();
    if (count == 0)
        return NULL;

    if (!uiMenu)
        uiMenu = new QMenu("Menu");

    for (int i=0; i<count; i++)
    {
        WebJSMenuItem* menuItem = d->m_children.at(i);
        menuItem->constructUIMenu(uiMenu);
    }

#ifndef QT_NO_DEBUG
    ::printMenu(uiMenu, 0);
#endif
    return uiMenu;
}

