/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDateTime>
#include <QFileInfo>
#include "wrtwidgetreply.h"
#include "wrtsettings.h"
#include "SuperWidget.h"
#include "widgetmanagerconstants.h"
#include <QFile>
#include <QFileInfo>

#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
#include "storage.h"
#endif

namespace WRT {


WrtWidgetReply::WrtWidgetReply(const QNetworkRequest& request,
                               Storage* storage,
                               const QString& protocol,
                               QString widgetRoot)
    : m_protocol(protocol.toLower()), QNetworkReply(), m_position(0)
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
    , m_storage(storage)
#endif
{
    Q_ASSERT(!request.url().scheme().compare(m_protocol, Qt::CaseInsensitive));
    QUrl url = request.url();
    QUrl newUrl;

    int libRootLength = 0;
    if (!m_protocol.compare("lib")) {
        // Format shared library url to absolute path so that
        // localization and secure storage checks can be done.

#ifndef Q_OS_LINUX
        QString sharedLibPath = widgetRoot+"\\"+SHARED_LIBRARY_FOLDER+"\\";
        QString libRoot("/" + QDir::fromNativeSeparators(sharedLibPath));
#else
        QString libRoot(QDir::fromNativeSeparators(SHARED_LIBRARY_PATH));
#endif
        libRootLength = libRoot.length(); // store root path length for faster conversion back to relative

        // Fix shared lib path if QUrl has parsed first folder of the path into authority field.
        // It does that if there are two slashes after scheme in url, like 'lib://'.
        if (!url.authority().isEmpty()) {
            url.setUrl("lib:" + libRoot + url.authority() + url.path());
        }
        else {
            url.setUrl("lib:" + libRoot + url.path().mid(1)); // remove leading '/' from path
        }
    }

    // Reconstruct the full path to the requested resource if this is a widget:// request
    if (protocol.compare("widget", Qt::CaseInsensitive) == 0)
    {
        // A widget scheme request is of form widget://<widget id>/<a path relative to the widget's private directory>
        // widgetRoot contains the private directory and the relative resource path can be retrieved with QUrl::path()
        // TODO: Do we want to see if the id in the requested url and the id in the widgetRoot matches?
        url = QUrl(widgetRoot + url.path());
    }

    if (getLocalizedUrl(newUrl,url))
    {
      url = newUrl;
    }

#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
    // check if widget files are accessed from secure storage
    QString secFile = url.path();
#ifndef Q_OS_LINUX
    if (secFile.startsWith("/"))
      {
    secFile = secFile.mid(1,-1);
      }
    secFile = QDir::toNativeSeparators(secFile);
#endif
    QFile secureFile(secFile);
    bool secure=false;
    if (m_storage) {
        secure = m_storage->verify(secureFile);
    }
#else
    bool secure = true; // no security/tamper checking
#endif

    m_fileName = toLocalFileName(url);

    if (!m_protocol.compare("lib")) {
        // Format library url back to relative by removing root path
        url.setUrl("lib://" + QDir::fromNativeSeparators(url.path()).mid(libRootLength));
    }

    // FIXME: should all reply URLs be the same as request URLs?
    if (!m_protocol.compare("widget"))
        setUrl(request.url());
    else
        setUrl(url);


    m_file.setFileName(m_fileName);
    bool exists = m_file.open(QIODevice::ReadOnly);
    if (exists) {
        if (!secure) {
            setAttribute(QNetworkRequest::HttpStatusCodeAttribute, 403);
            setAttribute(QNetworkRequest::HttpReasonPhraseAttribute, "Forbidden. Unsecure file access.");
            setError(ContentAccessDenied, "Access Denied");
            QMetaObject::invokeMethod(this, "sendError", Qt::QueuedConnection);
        }
        else {
            m_content = m_file.readAll();
            m_file.close();
            setAttribute(QNetworkRequest::HttpStatusCodeAttribute, 200);
            setAttribute(QNetworkRequest::HttpReasonPhraseAttribute, "ok");
            setHeader(QNetworkRequest::ContentLengthHeader, m_content.size());
            QFileInfo fileInfo(m_fileName);
            setHeader(QNetworkRequest::LastModifiedHeader, fileInfo.lastModified());
            QMetaObject::invokeMethod(this, "sendData", Qt::QueuedConnection);
            setOpenMode(QIODevice::ReadOnly);
        }
    }
    else {
        setAttribute(QNetworkRequest::HttpStatusCodeAttribute, 404);
        setAttribute(QNetworkRequest::HttpReasonPhraseAttribute, "Not found");
        setError(ContentNotFoundError, "Not found");
        QMetaObject::invokeMethod(this, "sendError", Qt::QueuedConnection);
    }
}

WrtWidgetReply::~WrtWidgetReply()
{
}
qint64 WrtWidgetReply::bytesAvailable () const
{
    return m_content.size();
}

QString WrtWidgetReply::toLocalFileName(QUrl& url)
{
    QString requestUrl = url.toString();

    // If the request was made with widget scheme, the url has already been reconstructed in the constructor
    // into a file URL with absolute path and can be treated the same way as a normal file request
    if (!m_protocol.compare("file") ||
        !m_protocol.compare("widget")) {
        return QUrl(requestUrl).toLocalFile();
    }

    if (!m_protocol.compare("lib")) {
#ifndef Q_OS_LINUX
      return QDir::toNativeSeparators(url.path().mid(1)); // remove leading '/'
#else
      return QDir::toNativeSeparators(url.path());
#endif
    }

    // For sandbox scheme
    int idEnd = requestUrl.indexOf("/", m_protocol.length() + 1); // length of "m_protocol:/"
    return QUrl(("file://" + requestUrl.mid(idEnd, -1))).toLocalFile();
}

void WrtWidgetReply::abort()
{
    setError(OperationCanceledError, ("Canceled"));
}

qint64 WrtWidgetReply::readData(char* data, qint64 size)
{
    qint64 len = qMin(size, (qint64)(m_content.size() - m_position));
    memcpy(data, m_content.data() + m_position, len);
    m_position += len;
    return len;
}

void WrtWidgetReply::sendData()
{
    QMetaObject::invokeMethod(this, "sendFinished", Qt::QueuedConnection);
    emit readyRead();
}

void WrtWidgetReply::sendFinished()
{
    emit finished();
}

void WrtWidgetReply::sendError()
{
    emit finished();
}

// function to get localized url from widget
// parameters:
//     newUrl     newUrl that represents localized content
//     reqUrl     requested url
// returns:
//     bool       true if successful
//
// This function is used only in Widget mode.
bool WrtWidgetReply::getLocalizedUrl(QUrl& newUrl, const QUrl& reqUrl)
{
    QString path = reqUrl.path();

    WrtSettings* m_settings = WrtSettings::createWrtSettings();

    QString myLang = m_settings->valueAsString("UserAgentLanguage");

    if (myLang.isEmpty())
    {
        QLocale language;
        myLang = language.name();
        myLang.replace(QString("_"),QString("-"));
    }

    QStringList fileParts;
    if (!SuperWidget::getWidgetPathParts(fileParts,path))return false;


    if (!fileParts.count()>1) return false;

    QString root = fileParts.at(0);
#ifndef Q_OS_LINUX
    if (root.startsWith("/"))
      root = root.mid(1,-1);
#endif

    if (SuperWidget::getWidgetType(root)!=WidgetTypeW3c) return false;

#if defined(Q_OS_LINUX) || defined(Q_OS_MAEMO5) || defined(Q_OS_MAEMO6)
    QString locFile = SuperWidget::getLocalizedFile(path, myLang,true,root);
#else
    QString locFile = SuperWidget::getLocalizedFile(path.right(path.count()-1),myLang,true,root);
#endif

    if ( locFile.length() == 0 ) return false;
#ifdef Q_OS_SYMBIAN
      locFile = "/"+locFile;
#endif

    newUrl.clear();
    newUrl.setPath(locFile);
    newUrl.setAuthority(reqUrl.authority());
    newUrl.setScheme(reqUrl.scheme());
    return true;
}


}
