/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include <QtCore>
#include <QtCore/QCoreApplication>
#include <QDebug>

#include "signatureparser.h"
#include <QProcess>
#include <QDir>
#include <QDirIterator>
#include <QFile>

#include <openssl/bio.h>
#include <openssl/rsa.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/evp.h>
#include <openssl/sha.h>


static const QString WORK_DIR_PATH("/tmp/widget-verifier/");

QString findSignature() {
    QString retval("");
    QDir dir(WORK_DIR_PATH);
    QStringList entries = dir.entryList();
    for (int i =0; i<entries.count(); i++ ) {

        QString name = entries.at(i);
        if ( name.toLower()=="signature1.xml" ) {
            name= WORK_DIR_PATH+name;
            return name;
        }
    }
    return retval;
}

bool recRmDir(QString path) {
    QDir p;
    if (!p.cd(path)) {
        return false;
    }

    QStringList subdirs = p.entryList(QDir::AllDirs|QDir::NoDotAndDotDot);
    //recursively go trhgouth every subdir
    foreach (QString str, subdirs) {

        QString sbdir = QString("%1/%2").arg(p.absolutePath()).arg(str);
        if (!recRmDir(sbdir)) {
            return false;
        }
        p.rmdir(sbdir); //delete what was just cleared
    }

    //kill the files and dirs
    QStringList files = p.entryList(QDir::AllEntries|QDir::NoDotAndDotDot);
    foreach(QString str, files) {
        p.remove(str);
    }

    //check whether we succesfully killed all
    files = p.entryList(QDir::AllEntries|QDir::NoDotAndDotDot);
    if (files.size() != 0) {
        return false;
    }
    return true;
}

int shaDigestTest (int argc, char*argv[]) {
    QByteArray test("Hello world!!!!");
    qDebug() << test.size() << " " << test.data();

    unsigned char hashed[SHA256_DIGEST_LENGTH];
    SHA256((const unsigned char*)test.data(), test.size(), hashed);

    QByteArray qHashed = QByteArray::fromRawData((const char*)hashed, SHA256_DIGEST_LENGTH);
    qDebug() << qHashed.size() << " " << qHashed.data();

    SHA256_CTX sha_ctx;
    unsigned char hashed_sha256[SHA256_DIGEST_LENGTH];
    int rc = SHA256_Init(&sha_ctx);
    if (1 != rc) { qDebug() << "Cannot init SHA256 structure"; return false; }
    rc = SHA256_Update(&sha_ctx, test.data(), test.size());
    if (1 != rc) { qDebug() << "Error hashing digest using SHA256"; return false; }
    rc = SHA256_Final((unsigned char*)hashed_sha256, &sha_ctx);
    if (1 != rc) { qDebug() << "Error producing final SHA256 hash"; return false; }
    QByteArray qHashed_sha256 = QByteArray::fromRawData((const char*)hashed_sha256, SHA256_DIGEST_LENGTH);
    qDebug() << qHashed_sha256.size() << " " << qHashed_sha256.data();


    unsigned char hashed_sha1[SHA_DIGEST_LENGTH];
    SHA1((const unsigned char*)test.data(), test.size(), hashed_sha1);

    QByteArray qHashed_sha1 = QByteArray::fromRawData((const char*)hashed_sha1, SHA_DIGEST_LENGTH);
    qDebug() << qHashed_sha1.size() << " " << qHashed_sha1.data();

    QByteArray qCrypto = QCryptographicHash::hash(test, QCryptographicHash::Sha1);
    qDebug() << qCrypto.size() << " " << qCrypto.data();

    return 0;
}


int rsatest (int argc, char *argv[])
{
    QString pubCert = "/home/mboudrea/certs/rsa2048pub.pem";
    QString privCert = "/home/mboudrea/certs/rsa2048priv.pem";

    QByteArray testData("aabbccddeeffggaabbccddeeffgg1234567890aabbccddeeffggaabbccddeeffaabbccddeeffggaabbccddeeffgg1234567890aabbccddeeffggaabbccddeeffaabbccddeeffggaabbccddeeffgg1234567890aabbccddeeffggaabbccddeeffaabbccddeeffggaabbccddeeffgg1234567890aabbccddeeffgg123456789012");
    // hash data
    QByteArray hashedTestData = QCryptographicHash::hash(testData, QCryptographicHash::Sha1);
    //qDebug() << "size: " << hashedTestData.size() << ", data: " << hashedTestData.toHex();
    qDebug() << "encrypted data: " << hashedTestData.toHex();

    // with PKCS#1 v1.5 padding, the data to encrypt MUST be less than RSA_size() - 11 bytes

    // Encryption phase
    QFile privCertFile(privCert);
    privCertFile.open(QIODevice::ReadOnly);
    QByteArray privCertContent = privCertFile.readAll();

    BIO * vb = BIO_new_mem_buf(privCertContent.data(),privCertContent.size());
    RSA* vrsa = PEM_read_bio_RSAPrivateKey(vb, NULL, 0, 0);
    int rvsize = RSA_size(vrsa);
    unsigned char encryptedPrivate[rvsize];

    int len = RSA_private_encrypt(hashedTestData.size(), (const unsigned char*) hashedTestData.data(), encryptedPrivate, vrsa, RSA_PKCS1_PADDING);

    if (len < 0) {
        char errBuf[120];
        ERR_error_string(ERR_get_error(), errBuf);
        QByteArray errMsg(errBuf);
        qDebug() << errMsg;
        return 1;
    }

//    qDebug() << encryptedPrivate;
    QByteArray encrypted = QByteArray::fromRawData((const char*)encryptedPrivate, len);


    // Decryption phase
    QFile pubCertFile(pubCert);
    pubCertFile.open(QIODevice::ReadOnly);
    QByteArray certificate = pubCertFile.readAll();

    //qDebug() << certificate.data();
    BIO * bb = BIO_new_mem_buf(certificate.data(),certificate.size());
    X509 * bx509 = PEM_read_bio_X509(bb, NULL, 0, NULL);

    EVP_PKEY *bk = X509_get_pubkey(bx509);

    if (!bk) {
        if (NULL != bb) BIO_free(bb);
        if (NULL != bx509) X509_free(bx509);
        return 1; //SignatureValidator::Bad_Certificate;
    }

    if (EVP_PKEY_RSA != EVP_PKEY_type(bk->type)) {
        if (NULL != bk) EVP_PKEY_free(bk);
        if (NULL != bb) BIO_free(bb);
        if (NULL != bx509) X509_free(bx509);
        return 1; //SignatureValidator::Bad_Certificate;
    }

    RSA *brsa = EVP_PKEY_get1_RSA(bk);
    int rbsize = RSA_size(brsa);
    unsigned char decryptedPublic[rbsize];

    len = RSA_public_decrypt(encrypted.size(), (const unsigned char*) encrypted.data(), (unsigned char*) decryptedPublic, brsa, RSA_PKCS1_PADDING);


    if (len < 0) {
        char errBuf[120];
        ERR_error_string(ERR_get_error(), errBuf);
        QByteArray errMsg(errBuf);
        qDebug() << errMsg;
        return 1;
    }

    QByteArray unencrypted = QByteArray::fromRawData((const char*) decryptedPublic, len);
    qDebug() << "decrypted data: " << unencrypted.toHex();

    return 0;
}


int dsaTest (int argc, char *argv[])
{
    QString pubCert = "/home/mboudrea/certs/dsa2048pub.pem";
    QString privCert = "/home/mboudrea/certs/dsa2048priv.pem";

    QByteArray testData("aabbccddeeffggaabbccddeeffgg1234567890aabbccddeeffggaabbccddeeffaabbccddeeffggaabbccddeeffgg1234567890aabbccddeeffggaabbccddeeffaabbccddeeffggaabbccddeeffgg1234567890aabbccddeeffggaabbccddeeffaabbccddeeffggaabbccddeeffgg1234567890aabbccddeeffgg123456789012");
    // hash data
    QByteArray hashedTestData = QCryptographicHash::hash(testData, QCryptographicHash::Sha1);
    qDebug() << "encrypted data: " << hashedTestData.toHex();

    // with PKCS#1 v1.5 padding, the data to encrypt MUST be less than RSA_size() - 11 bytes

    // Encryption phase
    QFile privCertFile(privCert);
    privCertFile.open(QIODevice::ReadOnly);
    QByteArray privCertContent = privCertFile.readAll();

    BIO * vb = BIO_new_mem_buf(privCertContent.data(),privCertContent.size());
    DSA *dsa = PEM_read_bio_DSAPrivateKey(vb, NULL, 0, 0);

    if (dsa==NULL) {
        if (NULL != vb) BIO_free(vb);
        if (NULL != dsa) DSA_free(dsa);
        qDebug() << "Error: Bad DSA private key";
         return -1;
    }

    int dsaSize = DSA_size(dsa);
    unsigned char dsaBuf[dsaSize];
    unsigned int dsaSigLength;

    int rc = DSA_sign(NID_sha1,
                      (const unsigned char*) hashedTestData.data(),
                      hashedTestData.size(),
                      dsaBuf,
                      &dsaSigLength,
                      dsa);

    if (NULL != vb) BIO_free(vb);
    if (NULL != dsa) DSA_free(dsa);

    if (rc != 1) {
        qDebug() << "Error: DSA signature faiulure";
        char errBuf[120];
        ERR_error_string(ERR_get_error(), errBuf);
        QByteArray errMsg(errBuf);
        qDebug() << "openssl DSA_verify error:\n" << errMsg;
        return -1;
    }

    QByteArray encrypted;
    encrypted.resize(dsaSigLength);
    for (unsigned int i = 0; i < dsaSigLength; i++) {
        qDebug() << dsaBuf[i];
        encrypted[i] = dsaBuf[i];
    }

    encrypted = encrypted.toBase64();

    QFile pubCertFile(pubCert);
    pubCertFile.open(QIODevice::ReadOnly);
    QByteArray pubCertContent = pubCertFile.readAll();
    BIO * b = BIO_new_mem_buf(pubCertContent.data(),pubCertContent.size());
    X509 * x509 = PEM_read_bio_X509(b, NULL, 0, NULL);

    EVP_PKEY *k = X509_get_pubkey(x509);

    if (!k) {
        if (NULL != b) BIO_free(b);
        if (NULL != x509) X509_free(x509);
        return -1;
    }

    int keyType = EVP_PKEY_type(k->type);
    if (!(keyType == EVP_PKEY_DSA || keyType == EVP_PKEY_DSA3)) {
        if (NULL != k) EVP_PKEY_free(k);
        if (NULL != b) BIO_free(b);
        if (NULL != x509) X509_free(x509);
        return -1;
    }

    dsa = EVP_PKEY_get1_DSA(k);
    QByteArray sig = QByteArray::fromBase64(encrypted);
    qDebug() << "signature to validate, from base64: " << sig.toHex() << "\nsize: " << sig.size();

    // TODO: verify that DSA_verify is in fact independent of the message digest algorithm used
    // TODO: verify that the decrypted data from sig is NOT base64, otherwise we're passing bad data for verify() to compare against
    rc = DSA_verify(0, (const unsigned char*) hashedTestData.data(), hashedTestData.size(), (const unsigned char*) sig.data(), sig.size(), dsa);

    if (NULL != k) EVP_PKEY_free(k);
    if (NULL != b) BIO_free(b);
    if (NULL != x509) X509_free(x509);
    if (NULL != dsa) DSA_free(dsa);

    if (rc < 0) {
        char errBuf[120];
        ERR_error_string(ERR_get_error(), errBuf);
        QByteArray errMsg(errBuf);
        qDebug() << "OpenSSL DSA_verify error:\n" << errMsg;
        return -1;
    }

    if (rc == 0 ) {
        qDebug() << "DSA verification failed.  Incorrect signature.";
        return -1;
    }

    if (rc == 1) {
        qDebug() << "VERIFICATION SUCCESS!!";
    }

    return 0;
}

int main (int argc, char*argv[]) {
    QCoreApplication a(argc, argv);
    QStringList params;
    if (argc < 3) {
        qDebug() << "Usage: W3CWidgetVerifier <filename> <pubCertPath>";
        return -1;
    }
    QString fileName(argv[1]);
    QString certFile(argv[2]);

    params << "-q" << fileName;

    if ( QProcess::execute( "unzip", params)!=0 ) {
        return -1;
        qDebug() << "Not a valid zip-file";
    }

    params.clear();
    params  <<"-u" << fileName << "-d"+WORK_DIR_PATH;

    if ( QProcess::execute( "unzip", params)!=0 ) {
        qDebug() << "Failed to decompress";
        return -1;
    }

    QString signaturefile;
    //qDebug() << "finding signature";
    signaturefile=findSignature();

    if ( signaturefile== "") {
        recRmDir(WORK_DIR_PATH);

        qDebug() << "Signature.xml not found";
        return -1;
    }

    SignatureParser parser(WORK_DIR_PATH, signaturefile);
    //qDebug() << "opening: " << signaturefile;

    if ( !parser.parse() ) {
        qDebug() << "Parsing failed";
        recRmDir(WORK_DIR_PATH);

        return -1;
    }

    if ( !parser.requiredElementsExist() ) {
        qDebug() << "Error with required elements";
        recRmDir(WORK_DIR_PATH);

        return -1;
    }

    QDir dir("/tmp/widget-verifier/");
    dir.setFilter(QDir::NoDotAndDotDot | QDir::Files);
    QDirIterator dirItr(dir, QDirIterator::Subdirectories);
    QStringList files;
    while (dirItr.hasNext()) {
        dirItr.next();
        int basePathLen = dirItr.path().length();
        QString relativePath = dirItr.filePath();
        relativePath = relativePath.mid(basePathLen + 1);
        files.append(relativePath);
    }

    if ( !parser.checkReferencesExist(files) ) {
        qDebug() << "Reference check failed";
        recRmDir(WORK_DIR_PATH);

        return -1;
    }

    if ( !parser.validateCertificate()) {
        qDebug() << "Certificate validation failed";
        recRmDir(WORK_DIR_PATH);

        return -1;
    }

    if ( !parser.validateSignature(signaturefile) ) {
        qDebug() << "Signature validation failed";
        recRmDir(WORK_DIR_PATH);

        return -1;
    }

    if ( !parser.validateReferences(files) ) {
        qDebug() << "Validating references failed";
        recRmDir(WORK_DIR_PATH);

        return -1;
    }
    qDebug() << "All done, Valid widget";

    recRmDir(WORK_DIR_PATH);

    return 0;
}
