/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef SIGNATUREVALIDATOR_H
#define SIGNATUREVALIDATOR_H
#include <QtCore/QObject>
#include <openssl/x509.h>

namespace W3C {
    // prefixes for signed content when using RSA algorithm
    const QString RSA_digestInfo_SHA1("3021300906052b0e03021a05000414");
    const QString RSA_digestInfo_SHA256("3031300d060960864801650304020105000420");
}

 /** Implements signature validation using multiple algorithms by wrapping OpenSSL functions.
  */

class SignatureValidator : public QObject
{
  Q_OBJECT

  public:

  SignatureValidator();
  ~SignatureValidator();

  /** Verifies that a RSA-SHA{1,256,512} signature matches data and a certifiate.
   * Verifies whether the data corresponds to signature and certificate.
   * The verification is based on RSA-SHA{1,256,512}. Implementation uses OpenSSL
   * functions, the most important of these being RSA_verify.
   *
   * @param data UTF-8 encoded text
   * @param signature RSA-SHA{X} signature that is base 64 encoded
   * @param certificate A certificate in PEM file format
   * @param digestMethod Enum defining the digest method of the certificate
   * @param c14nMode Canonicalization algorithm (1.0, 1.1, or Exclusive 1.0)
   * @param c14nComments Canoicalization with comments (1) or omits comments (0)
   *
   * @return SignatureValidator::Verification_Success (1) if the verification is successfull.
   *         If not successfull returns an int > 1: SignatureValidator::Verification_Failure
   *         (2) or SignatureValidator::Bad_Certificate (3).
   */
  int rsaVerify(QByteArray data, QByteArray signature, QByteArray certificate, int digestMethod, int c14nMode, int c14nComments);
  int dsaVerify(QByteArray data, QByteArray signature, QByteArray certificate, int digestMethod, int c14nMode, int c14nComments);

  QString getSubject(QByteArray certificate);

  static const int Verification_Success = 1;
  static const int Verification_Failure = 2;
  static const int Bad_Certificate = 3;
  static const int Certificate_Ok = 0;

  enum digestMethod {
      sha_1 = 0,
      sha_256,
      sha_512
  };

  bool canonicalize(const QString& inStr, QString& outStr, int c14nMode, int c14nComments);

 private:

     int createDigest(QByteArray* digest, int digestMethod, QByteArray data);
};

#endif // SIGNATUREVALIDATOR_H
