/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

//#include <QDebug>
#include <QCoreApplication>
#include <QDesktopServices>
#include <QtCore/QStringList>
#include <QtGui/QMessageBox>
#include "wrtsettings.h"
#include "qwebsettings.h"
#include "resources/qtnresources.h"
#include "resources/qtnstrings.h"
#include "resources/wrtversions.h"
#include <QDebug>
#include <QDir>

#ifdef Q_OS_SYMBIAN
#include "f32file.h"
#endif

#define VALUEDELIM ";"
#define RANGEDELIM ":"

//#if defined (Q_OS_MAEMO5) || defined (Q_OS_MAEMO6)
#if defined (Q_OS_MAEMO6)
#define WRT_SETTINGS "/etc/cwrt/wrtsettings"
#endif

#if defined (Q_OS_MAEMO5)
#define WRT_SETTINGS "/etc/wrt/wrtsettings"
#endif

namespace WRT {

WrtSetting::WrtSetting(const QString &key):
        m_key(key)
      , m_type(StringType)
      , m_flags(0)
      , m_enum(-1)
      , m_category(0)

{
}

QString WrtSetting::displayValue()
{
    // convert actual value to a string name from the list
    if (m_type == ListType && m_valuesList.count() == m_validList.count()){
        for (int i=0; i < m_validList.count(); i++)
            if (m_value.toString() == m_valuesList[i])
                return m_validList[i];
    }
    // don't display any value for commands
    else if (m_type == CommandType || m_type == NodeType)
        return QString();

    return m_value.toString();
}

QVariant WrtSetting::value()
{
    return m_value;
}

void WrtSetting::setValid(const QString &valid)
{
    if (valid.contains(VALUEDELIM))
        m_validList = valid.split(VALUEDELIM);
    else if (valid.contains(RANGEDELIM))
        m_validList = valid.split(RANGEDELIM);
}
void WrtSetting::setValues(const QString &values)
{
    m_valuesList = values.split(VALUEDELIM);
}

/////////////////////////////////////////

WrtSettings* WrtSettings::g_wrtSettings = 0;
WrtSettings* WrtSettings::createWrtSettings()
{
    if (g_wrtSettings == 0)
        g_wrtSettings = new WrtSettings();

    return g_wrtSettings;
}

WrtSettings::WrtSettings( QObject* parent, QString uid ) :
#if !defined (Q_OS_MAEMO5) && !defined (Q_OS_MAEMO6)
      QSettings(IniFormat, UserScope, WRT_ORGANIZATION_NAME, uid, parent)
#else
      // For Maemo5 and Maemo6 settings file should be accessible only by root.
      QSettings(WRT_SETTINGS, IniFormat, parent)
#endif
{
    setObjectName(QString::fromUtf8("browsersettings"));
    m_appuid = uid;

    init();
    connect(this, SIGNAL(settingActivated(const int)), SLOT(doAction(const int)));
    connect(this, SIGNAL(settingChanged(const QString &)), SLOT(settingTrigger(const QString &)));
}

/*!
    Initialize settings

    Template for adding settings:   TODO: move settings to external file - possibly conf xml file

     // Main setting name and default value.  This is the name used to get and set the setting value
        QSettings::setValue("AutoLoadImages", "1");

     Main setting name is also the 'group' for the meta data.
     Meta data is used by the UI to determine how to show and interact with the setting.

     // desc - non-localized display name
     // descl - localized display name where the value is the index into a localized array of strings. see qtnstrings.h and gtnresources.h
     Note: only one of these should be used.  The last one loaded is the one used and the order is not guaranteed
        QSettings::setValue("AutoLoadImages/desc", "Auto Load Images");
        QSettings::setValue("AutoLoadImages/descl", qtn_wrtsetting_autoloadimages);

     // type - data type used by ui to determine which dialog to use.  Default is String.
        QSettings::setValue("AutoLoadImages/type", WrtSetting::ListType);

     // valid - list of valid values for drop down (delimited by ';') or range of values for spin (delimited by ':')  TODO: localize list of values
        QSettings::setValue("AutoLoadImages/valid", "Enabled;Disabled");

     // values - mapping for the list of values (ex. Enabled = 1 ; Disabled = 0)
        QSettings::setValue("AutoLoadImages/values", "1;0");

     // flags - to control access (ReadOnly | Hidden) or for atribute settings (WrtSetting::WebAttribute)
        QSettings::setValue("AutoLoadImages/flags", WrtSetting::WebAttribute);

     // enum - enumerator for attribute settings
        QSettings::setValue("AutoLoadImages/enum", QWebSettings::AutoLoadImages);

     // cat - to filter settings to different categories in UI.  ex. "General";"Page";"Security"
     //    Default is General.  Not needed if setting is hidden. Root is reserved for the list of Categories in the UI
        QSettings::setValue("AutoLoadImages/cat", WrtSetting::GeneralCategory);

*/
void WrtSettings::init(bool clear)
{
    // check for new version
    QString wrtVersion = QString(WRTSETTINGS_VERSION_FORMAT).arg(WRT_VERSION_MAJOR).arg(WRT_VERSION_MINOR).arg(WRT_VERSION_SVN);

    if (clear || valueAsString(WRTSETTINGS_WRT_VERSION_KEY).compare(wrtVersion)){

        beginGroup(m_appuid);
        if (clear)
            QSettings::clear();

        // Versioning

        QSettings::setValue("Versions", "WRTVersion;BrowserVersion;WidgetsVersion;SettingsVersion");
        QSettings::setValue("Versions/desc", "Delimited list of version settings");
        QSettings::setValue("Versions/flags", WrtSetting::ReadOnly | WrtSetting::Hidden);

        QSettings::setValue("WRTVersion", wrtVersion);
        QSettings::setValue("WRTVersion/descl", qtn_wrtsetting_wrtversion);
        QSettings::setValue("WRTVersion/flags", WrtSetting::Hidden);

        if (!QSettings::contains("SettingsVersion"))
            QSettings::setValue("SettingsVersion", QString());
        QSettings::setValue("SettingsVersion/descl", qtn_wrtsetting_settingsversion);
        QSettings::setValue("SettingsVersion/flags", WrtSetting::Hidden);

        if (!QSettings::contains("BrowserVersion"))
            QSettings::setValue("BrowserVersion", QString());
        QSettings::setValue("BrowserVersion/descl", qtn_wrtsetting_browserversion);
        QSettings::setValue("BrowserVersion/flags", WrtSetting::Hidden);

        if (!QSettings::contains("WidgetsVersion"))
            QSettings::setValue("WidgetsVersion", QString());
        QSettings::setValue("WidgetsVersion/descl", qtn_wrtsetting_widgetsversion);
        QSettings::setValue("WidgetsVersion/flags", WrtSetting::Hidden);

        // Categories

        QSettings::setValue("Category1", "10001");
        QSettings::setValue("Category1/descl", qtn_wrtsetting_category1);
        QSettings::setValue("Category1/type",WrtSetting::CommandType);
        QSettings::setValue("Category1/cat", WrtSetting::RootCategory);

        QSettings::setValue("Category2", "10002");
        QSettings::setValue("Category2/descl", qtn_wrtsetting_category2);
        QSettings::setValue("Category2/type",WrtSetting::CommandType);
        QSettings::setValue("Category2/cat", WrtSetting::RootCategory);

        QSettings::setValue("Category3", "10003");
        QSettings::setValue("Category3/descl", qtn_wrtsetting_category3);
        QSettings::setValue("Category3/type",WrtSetting::CommandType);
        QSettings::setValue("Category3/cat", WrtSetting::RootCategory);

        QSettings::setValue("Category4", "10004");
        QSettings::setValue("Category4/descl", qtn_wrtsetting_category4);
        QSettings::setValue("Category4/type",WrtSetting::CommandType);
        QSettings::setValue("Category4/cat", WrtSetting::RootCategory);

        // Settings

        if (!QSettings::contains("AutoLoadImages"))
            QSettings::setValue("AutoLoadImages", "1");
        QSettings::setValue("AutoLoadImages/descl", qtn_wrtsetting_autoloadimages);
        QSettings::setValue("AutoLoadImages/type", WrtSetting::ListType);
        QSettings::setValue("AutoLoadImages/valid", "Enabled;Disabled");
        QSettings::setValue("AutoLoadImages/values", "1;0");
        QSettings::setValue("AutoLoadImages/flags", WrtSetting::WebAttribute);
        QSettings::setValue("AutoLoadImages/enum", QWebSettings::AutoLoadImages);
        QSettings::setValue("AutoLoadImages/cat", WrtSetting::Category1);

        if (!QSettings::contains("JavaScript"))
            QSettings::setValue("JavaScript", "1");
        QSettings::setValue("JavaScript/descl", qtn_wrtsetting_javascriptenabled);
        QSettings::setValue("JavaScript/type", WrtSetting::ListType);
        QSettings::setValue("JavaScript/valid", "Enabled;Disabled");
        QSettings::setValue("JavaScript/values", "1;0");
        QSettings::setValue("JavaScript/flags", WrtSetting::WebAttribute);
        QSettings::setValue("JavaScript/enum", QWebSettings::JavascriptEnabled);
        QSettings::setValue("JavaScript/cat", WrtSetting::Category4);

        if (!QSettings::contains("Java"))
            QSettings::setValue("Java", "1");
        QSettings::setValue("Java/descl", qtn_wrtsetting_javaenabled);
        QSettings::setValue("Java/type", WrtSetting::ListType);
        QSettings::setValue("Java/valid", "Enabled;Disabled");
        QSettings::setValue("Java/values", "1;0");
        QSettings::setValue("Java/flags", WrtSetting::WebAttribute);
        QSettings::setValue("Java/enum", QWebSettings::JavaEnabled);
        QSettings::setValue("Java/cat", WrtSetting::Category4);

        if (!QSettings::contains("Plugins"))
            QSettings::setValue("Plugins", "1");
        QSettings::setValue("Plugins/descl", qtn_wrtsetting_pluginsenabled);
        QSettings::setValue("Plugins/type", WrtSetting::ListType);
        QSettings::setValue("Plugins/valid", "Enabled;Disabled");
        QSettings::setValue("Plugins/values", "1;0");
        QSettings::setValue("Plugins/flags", WrtSetting::WebAttribute);
        QSettings::setValue("Plugins/enum", QWebSettings::PluginsEnabled);
        QSettings::setValue("Plugins/cat", WrtSetting::Category1);

        if (!QSettings::contains("Cookies"))
            QSettings::setValue("Cookies", "1");
        QSettings::setValue("Cookies/descl", qtn_wrtsetting_cookiesenabled);
        QSettings::setValue("Cookies/type", WrtSetting::ListType);
        QSettings::setValue("Cookies/valid", "Enabled;Disabled");
        QSettings::setValue("Cookies/values", "1;0");
        QSettings::setValue("Cookies/cat", WrtSetting::Category1);

        if (!QSettings::contains("Navigation"))
            QSettings::setValue("Navigation", "1");
        QSettings::setValue("Navigation/descl", qtn_wrtsetting_navigation);
        QSettings::setValue("Navigation/type", WrtSetting::ListType);
        QSettings::setValue("Navigation/valid", "Enabled;Disabled");
        QSettings::setValue("Navigation/values", "1;0");
        QSettings::setValue("Navigation/cat", WrtSetting::Category1);


        if (!QSettings::contains("PrivateBrowsing"))
            QSettings::setValue("PrivateBrowsing", "0");
        QSettings::setValue("PrivateBrowsing/descl", qtn_wrtsetting_privatebrowsingenabled);
        QSettings::setValue("PrivateBrowsing/type", WrtSetting::ListType);
        QSettings::setValue("PrivateBrowsing/valid", "Enabled;Disabled");
        QSettings::setValue("PrivateBrowsing/values", "1;0");
        QSettings::setValue("PrivateBrowsing/flags", WrtSetting::WebAttribute);
        QSettings::setValue("PrivateBrowsing/enum", QWebSettings::PrivateBrowsingEnabled);
        QSettings::setValue("PrivateBrowsing/cat", WrtSetting::Category2);

        if (!QSettings::contains("JavaScriptCanOpenWindows"))
            QSettings::setValue("JavaScriptCanOpenWindows", "1");
        QSettings::setValue("JavaScriptCanOpenWindows/descl", qtn_wrtsetting_jscanopenwindows);
        QSettings::setValue("JavaScriptCanOpenWindows/type", WrtSetting::ListType);
        QSettings::setValue("JavaScriptCanOpenWindows/valid", "Enabled;Disabled");
        QSettings::setValue("JavaScriptCanOpenWindows/values", "1;0");
        QSettings::setValue("JavaScriptCanOpenWindows/flags", WrtSetting::WebAttribute);
        QSettings::setValue("JavaScriptCanOpenWindows/enum", QWebSettings::JavascriptCanOpenWindows);
        QSettings::setValue("JavaScriptCanOpenWindows/cat", WrtSetting::Category4);

        if (!QSettings::contains("JavaScriptCanAccessClipboard"))
            QSettings::setValue("JavaScriptCanAccessClipboard", "1");
        QSettings::setValue("JavaScriptCanAccessClipboard/descl", qtn_wrtsetting_jscanaccessclipboard);
        QSettings::setValue("JavaScriptCanAccessClipboard/type", WrtSetting::ListType);
        QSettings::setValue("JavaScriptCanAccessClipboard/valid", "Enabled;Disabled");
        QSettings::setValue("JavaScriptCanAccessClipboard/values", "1;0");
        QSettings::setValue("JavaScriptCanAccessClipboard/flags", WrtSetting::WebAttribute | WrtSetting::Hidden);
        QSettings::setValue("JavaScriptCanAccessClipboard/enum", QWebSettings::JavascriptCanAccessClipboard);
        QSettings::setValue("JavaScriptCanAccessClipboard/cat", WrtSetting::Category4);

    if (!QSettings::contains("JavaScriptConsoleLog"))
            QSettings::setValue("JavaScriptConsoleLog", "0");
        QSettings::setValue("JavaScriptConsoleLog/descl", qtn_wrtsetting_jsconsolelog);
        QSettings::setValue("JavaScriptConsoleLog/type", WrtSetting::ListType);
        QSettings::setValue("JavaScriptConsoleLog/valid", "Logging Off;Log to file;Show pop-up notes;Log to file and show pop-up notes");
        QSettings::setValue("JavaScriptConsoleLog/values", "0;1;2;3");
        QSettings::setValue("JavaScriptConsoleLog/cat", WrtSetting::Category4);

        if (!QSettings::contains("JavaScriptConsoleLogFilePath"))
            {
                QString jsLogFilePath = QDesktopServices::storageLocation(QDesktopServices::TempLocation);
                jsLogFilePath += "/jsLog_web.log";
                QSettings::setValue("JavaScriptConsoleLogFilePath", jsLogFilePath);
            }
        QSettings::setValue("JavaScriptConsoleLogFilePath/descl", qtn_wrtsetting_jsconsolelogfilepath);
        QSettings::setValue("JavaScriptConsoleLogFilePath/cat", WrtSetting::Category4);

    if (!QSettings::contains("UserAgentLanguage"))
            QSettings::setValue("UserAgentLanguage", "");
        QSettings::setValue("UserAgentLanguage/descl", qtn_wrtsetting_useragentlanguage);
        QSettings::setValue("UserAgentLanguage/type", WrtSetting::ListType);
        QSettings::setValue("UserAgentLanguage/valid", " ;en;en-us;en-gb;es;fi;fr;zh;zh-hans;zh-hans-cn");
        QSettings::setValue("UserAgentLanguage/values", " ;en;en-us;en-gb;es;fi;fr;zh;zh-hans;zh-hans-cn");
        QSettings::setValue("UserAgentLanguage/cat", WrtSetting::Category4);

        if (!QSettings::contains("DeveloperExtras"))
            QSettings::setValue("DeveloperExtras", "0");
        QSettings::setValue("DeveloperExtras/descl", qtn_wrtsetting_developerextrasenabled);
        QSettings::setValue("DeveloperExtras/type", WrtSetting::ListType);
        QSettings::setValue("DeveloperExtras/valid", "Enabled;Disabled");
        QSettings::setValue("DeveloperExtras/values", "1;0");
        QSettings::setValue("DeveloperExtras/flags", WrtSetting::WebAttribute);
        QSettings::setValue("DeveloperExtras/enum", QWebSettings::DeveloperExtrasEnabled);
        QSettings::setValue("DeveloperExtras/cat", WrtSetting::Category4);

        if (!QSettings::contains("LinksIncludedInFocusChain"))
           QSettings::setValue("LinksIncludedInFocusChain", "1");
        QSettings::setValue("LinksIncludedInFocusChain/descl", qtn_wrtsetting_linksincludedinfocaschain);
        QSettings::setValue("LinksIncludedInFocusChain/type", WrtSetting::ListType);
        QSettings::setValue("LinksIncludedInFocusChain/valid", "Yes;No");
        QSettings::setValue("LinksIncludedInFocusChain/values", "1;0");
        QSettings::setValue("LinksIncludedInFocusChain/flags", WrtSetting::WebAttribute | WrtSetting::Hidden);
        QSettings::setValue("LinksIncludedInFocusChain/enum", QWebSettings::LinksIncludedInFocusChain);
        QSettings::setValue("LinksIncludedInFocusChain/cat", WrtSetting::Category4);

        if (!QSettings::contains("ZoomTextOnly"))
            QSettings::setValue("ZoomTextOnly", "0");
        QSettings::setValue("ZoomTextOnly/descl", qtn_wrtsetting_zoomtextonly);
        QSettings::setValue("ZoomTextOnly/type", WrtSetting::ListType);
        QSettings::setValue("ZoomTextOnly/valid", "Yes;No");
        QSettings::setValue("ZoomTextOnly/values", "1;0");
        QSettings::setValue("ZoomTextOnly/flags", WrtSetting::WebAttribute);
        QSettings::setValue("ZoomTextOnly/enum", QWebSettings::ZoomTextOnly);
        QSettings::setValue("ZoomTextOnly/cat", WrtSetting::Category1);

        if (!QSettings::contains("PrintBackgroundElements"))
            QSettings::setValue("PrintBackgroundElements", "0");
        QSettings::setValue("PrintBackgroundElements/descl", qtn_wrtsetting_printbackgroundelements);
        QSettings::setValue("PrintBackgroundElements/type", WrtSetting::ListType);
        QSettings::setValue("PrintBackgroundElements/valid", "Yes;No");
        QSettings::setValue("PrintBackgroundElements/values", "1;0");
        QSettings::setValue("PrintBackgroundElements/flags", WrtSetting::WebAttribute | WrtSetting::Hidden);
        QSettings::setValue("PrintBackgroundElements/enum", QWebSettings::PrintElementBackgrounds);
        QSettings::setValue("PrintBackgroundElements/cat", WrtSetting::Category4);

        if (!QSettings::contains("OfflineStorageDatabase"))
            QSettings::setValue("OfflineStorageDatabase", "1");
        QSettings::setValue("OfflineStorageDatabase/descl", qtn_wrtsetting_offlinestoragedatabase);
        QSettings::setValue("OfflineStorageDatabase/type", WrtSetting::ListType);
        QSettings::setValue("OfflineStorageDatabase/valid", "Enabled;Disabled");
        QSettings::setValue("OfflineStorageDatabase/values", "1;0");
        QSettings::setValue("OfflineStorageDatabase/flags", WrtSetting::WebAttribute | WrtSetting::Hidden);
        QSettings::setValue("OfflineStorageDatabase/enum", QWebSettings::OfflineStorageDatabaseEnabled);
        QSettings::setValue("OfflineStorageDatabase/cat", WrtSetting::Category2);

        if (!QSettings::contains("OfflineWebApplicationCache"))
            QSettings::setValue("OfflineWebApplicationCache", "1");
        QSettings::setValue("OfflineWebApplicationCache/descl", qtn_wrtsetting_offlinewebapplicationcache);
        QSettings::setValue("OfflineWebApplicationCache/type", WrtSetting::ListType);
        QSettings::setValue("OfflineWebApplicationCache/valid", "Enabled;Disabled");
        QSettings::setValue("OfflineWebApplicationCache/values", "1;0");
        QSettings::setValue("OfflineWebApplicationCache/flags", WrtSetting::WebAttribute | WrtSetting::Hidden);
        QSettings::setValue("OfflineWebApplicationCache/enum", QWebSettings::OfflineWebApplicationCacheEnabled);
        QSettings::setValue("OfflineWebApplicationCache/cat", WrtSetting::Category2);

        if (!QSettings::contains("LocalStorageDatabase"))
            QSettings::setValue("LocalStorageDatabase", "1");
        QSettings::setValue("LocalStorageDatabase/descl", qtn_wrtsetting_localstoragedatabase);
        QSettings::setValue("LocalStorageDatabase/type", WrtSetting::ListType);
        QSettings::setValue("LocalStorageDatabase/valid", "Enabled;Disabled");
        QSettings::setValue("LocalStorageDatabase/values", "1;0");
        QSettings::setValue("LocalStorageDatabase/flags", WrtSetting::WebAttribute | WrtSetting::Hidden);
        QSettings::setValue("LocalStorageDatabase/enum", QWebSettings::LocalStorageDatabaseEnabled);
        QSettings::setValue("LocalStorageDatabase/cat", WrtSetting::Category2);

        if (!QSettings::contains("StartPage"))
            QSettings::setValue("StartPage", "");
        QSettings::setValue("StartPage/descl", qtn_wrtsetting_startpage);
        QSettings::setValue("StartPage/cat", WrtSetting::Category1);

        if (!QSettings::contains("DefaultTextSize"))
            QSettings::setValue("DefaultTextSize", "2");
        QSettings::setValue("DefaultTextSize/descl", qtn_wrtsetting_defaulttextsizes);
        QSettings::setValue("DefaultTextSize/type", WrtSetting::ListType);
        QSettings::setValue("DefaultTextSize/valid", "smallest;small;normal;large;largest");
        QSettings::setValue("DefaultTextSize/values", "0;1;2;3;4");
        QSettings::setValue("DefaultTextSize/cat", WrtSetting::Category1);

        if (!QSettings::contains("NetworkProxy")) {
#if (defined(Q_OS_SYMBIAN) && !defined(Q_CC_NOKIAX86)) || defined(Q_OS_MAEMO5)
        // empty proxy for ARMV5 Symbian, Maemo5 targets
        QSettings::setValue("NetworkProxy", QString());
#else
        QSettings::setValue("NetworkProxy", "bswebproxy01.americas.nokia.com");
#endif
        }
        QSettings::setValue("NetworkProxy/descl", qtn_wrtsetting_networkproxy);
        QSettings::setValue("NetworkProxy/cat", WrtSetting::Category3);

        if (!QSettings::contains("SecurityLevel"))
            QSettings::setValue("SecurityLevel", "1");
        QSettings::setValue("SecurityLevel/descl", qtn_wrtsetting_securitylevel);
        QSettings::setValue("SecurityLevel/type", WrtSetting::ListType);
        QSettings::setValue("SecurityLevel/valid", "Low;Medium;High");
        QSettings::setValue("SecurityLevel/values", "0;1;2");
        QSettings::setValue("SecurityLevel/cat", WrtSetting::Category2);

        if (!QSettings::contains("PopupBlocking"))
            QSettings::setValue("PopupBlocking", "0");
        QSettings::setValue("PopupBlocking/descl", qtn_wrtsetting_popupblocking);
        QSettings::setValue("PopupBlocking/type", WrtSetting::ListType);
        QSettings::setValue("PopupBlocking/valid", "On;Off");
        QSettings::setValue("PopupBlocking/values", "1;0");
        QSettings::setValue("PopupBlocking/cat", WrtSetting::Category2);

        if (!QSettings::contains("DefaultEncoding"))
            QSettings::setValue("DefaultEncoding", 0);
        QSettings::setValue("DefaultEncoding/descl", qtn_wrtsetting_defaultencoding);
        QSettings::setValue("DefaultEncoding/type", WrtSetting::ListType);
        QSettings::setValue("DefaultEncoding/valid", "none;automatic");
        QSettings::setValue("DefaultEncoding/values", "0;1");
        QSettings::setValue("DefaultEncoding/cat", WrtSetting::Category1);

        if (!QSettings::contains("MaxRecentUrls"))
            QSettings::setValue("MaxRecentUrls", 10);
        QSettings::setValue("MaxRecentUrls/descl", qtn_wrtsetting_maxrecenturls);
        QSettings::setValue("MaxRecentUrls/type", WrtSetting::IntType);
        QSettings::setValue("MaxRecentUrls/valid", "5:10");
        QSettings::setValue("MaxRecentUrls/flags", WrtSetting::BookmarksSetting);
        QSettings::setValue("MaxRecentUrls/cat", WrtSetting::Category2);

        QSettings::setValue("ClearPrivateData", "10010");
        QSettings::setValue("ClearPrivateData/descl", qtn_wrtsetting_clearallprivatedata);
        QSettings::setValue("ClearPrivateData/type", WrtSetting::CommandType);
        QSettings::setValue("ClearPrivateData/cat", WrtSetting::Category2);

        QSettings::setValue("ResetSettings", "10011");
        QSettings::setValue("ResetSettings/descl", qtn_wrtsetting_resetfactorysettings);
        QSettings::setValue("ResetSettings/type", WrtSetting::CommandType);
        QSettings::setValue("ResetSettings/cat", WrtSetting::Category1);

        QSettings::setValue("WebDatabases", "10013");
        QSettings::setValue("WebDatabases/descl", qtn_wrtsetting_webdatabases);
        QSettings::setValue("WebDatabases/type", WrtSetting::CommandType);
        QSettings::setValue("WebDatabases/cat", WrtSetting::Category2);

        if (!QSettings::contains("MaxSizeOfflineAppCache"))
            QSettings::setValue("MaxSizeOfflineAppCache", 5);
        QSettings::setValue("MaxSizeOfflineAppCache/descl", qtn_wrtsetting_maxsizeofflineappcache);
        QSettings::setValue("MaxSizeOfflineAppCache/type", WrtSetting::ListType);
        QSettings::setValue("MaxSizeOfflineAppCache/valid", "5;10;15;20;25");
        QSettings::setValue("MaxSizeOfflineAppCache/values", "5;10;15;20;25");
        QSettings::setValue("MaxSizeOfflineAppCache/cat", WrtSetting::Category2);

        QSettings::setValue("OfflineAppCache", "10014");
        QSettings::setValue("OfflineAppCache/descl", qtn_wrtsetting_offlineappcache);
        QSettings::setValue("OfflineAppCache/type", WrtSetting::CommandType);
        QSettings::setValue("OfflineAppCache/cat", WrtSetting::Category2);

    if (!QSettings::contains("DiskCacheDirectoryPath"))
            QSettings::setValue("DiskCacheDirectoryPath", "cwrtCache");
        QSettings::setValue("DiskCacheDirectoryPath/descl", qtn_wrtsetting_diskcachedirectorypath);
        QSettings::setValue("DiskCacheDirectoryPath/cat", WrtSetting::Category2);

        if (!QSettings::contains("OfflineMode"))
            QSettings::setValue("OfflineMode", "0");
        QSettings::setValue("OfflineMode/descl", qtn_wrtsetting_offlinemode);
        QSettings::setValue("OfflineMode/type", WrtSetting::ListType);
        QSettings::setValue("OfflineMode/valid", "On;Off");
        QSettings::setValue("OfflineMode/values", "1;0");
        QSettings::setValue("OfflineMode/cat", WrtSetting::Category2);




        // Database

        QSettings::setValue("DatabaseTimeToLive", "14");
        QSettings::setValue("DatabaseTimeToLive/type", WrtSetting::IntType);
        QSettings::setValue("DatabaseTimeToLive/desc", "DatabaseTimeToLive");
        QSettings::setValue("DatabaseTimeToLive/flags", WrtSetting::DatabaseSetting | WrtSetting::ReadOnly | WrtSetting::Hidden);

        QSettings::setValue("DatabaseMaxSize", "52428800"); // 50 MB
        QSettings::setValue("DatabaseMaxSize/type", WrtSetting::IntType);
        QSettings::setValue("DatabaseMaxSize/desc", "DatabaseMaxSize");
        QSettings::setValue("DatabaseMaxSize/flags", WrtSetting::DatabaseSetting | WrtSetting::ReadOnly | WrtSetting::Hidden);

        if (!QSettings::contains("DatabaseQuotaPerOrigin"))
            QSettings::setValue("DatabaseQuotaPerOrigin", "5242880"); // 5 MB
        QSettings::setValue("DatabaseQuotaPerOrigin/type", WrtSetting::IntType);
        QSettings::setValue("DatabaseQuotaPerOrigin/flags", WrtSetting::DatabaseSetting | WrtSetting::Hidden);

        QSettings::setValue("DatabaseQuotaPerOriginMin", "5242880"); // 5 MB
        QSettings::setValue("DatabaseQuotaPerOriginMin/type", WrtSetting::IntType);
        QSettings::setValue("DatabaseQuotaPerOriginMin/flags", WrtSetting::DatabaseSetting | WrtSetting::ReadOnly | WrtSetting::Hidden);

        QSettings::setValue("DatabaseQuotaPerOriginMax", "52428800"); // 50 MB
        QSettings::setValue("DatabaseQuotaPerOriginMax/type", WrtSetting::IntType);
        QSettings::setValue("DatabaseQuotaPerOriginMax/flags", WrtSetting::DatabaseSetting | WrtSetting::ReadOnly | WrtSetting::Hidden);


        // TODO: Once WrtSettings is made a library and it's API is available,
        // the following implementation for storing UserAgent should be removed from WrtSettings.
        // WrtSettings should have an implementation such that the deverloper/tester/build manager can set UserAgent Strings in here.
        if (!QSettings::contains("UserAgent/Platform"))
#ifdef Q_OS_WIN32
            QSettings::setValue("UserAgent/Platform", "Windows");
#elif defined Q_OS_LINUX
            QSettings::setValue("UserAgent/Platform", "Linux");
#elif defined Q_OS_SYMBIAN
            QSettings::setValue("UserAgent/Platform", "SymbianOS/9.2");
#endif

#ifdef Q_OS_SYMBIAN
        QSettings::setValue("UserAgent/Series60Version", "Series60/3.2");
#endif
        if (!QSettings::contains("UserAgent/HardwareType"))
            QSettings::setValue("UserAgent/HardwareType", "");
        if (!QSettings::contains("UserAgent/MIDP-CLDCversion"))
            QSettings::setValue("UserAgent/MIDP-CLDCversion", "Profile/MIDP-2.1 Configuration/CLDC-1.1");
        if (!QSettings::contains("UserAgent/DefaultUserAgentString"))
            QSettings::setValue("UserAgent/DefaultUserAgentString", "");
        QSettings::setValue("UserAgent/flags", WrtSetting::ReadOnly | WrtSetting::Hidden);

        QString path;

#ifdef QT_NO_DESKTOPSERVICES
  #if defined (Q_OS_S40)
        path = "E:\\";
  #else
        path = "C:\\";
  #endif
#else

        QString dataLocation;

#ifdef Q_OS_SYMBIAN
        // private path
        dataLocation = QCoreApplication::applicationDirPath();
        qDebug() << "WrtSettings::init() - QDir::currentPath() : " << dataLocation;
        if (dataLocation.startsWith("\\") || dataLocation.startsWith("/")) {
            // find drive letter
            RProcess me;
            TFileName processFileName = me.FileName();
            TDriveUnit drive(processFileName);
            TDriveName driveName = drive.Name();

            // add drive letter to path
            QString qtDrive = QString::fromUtf16(driveName.Ptr(), driveName.Length());
            dataLocation.prepend(qtDrive);
            qDebug() << "WrtSettings::init() - Path after prepending drive letter" << dataLocation;
        }

        if (dataLocation.startsWith("Z"))
            dataLocation.replace(0, 1, "C");
        QDir privateDir(dataLocation);
        if (!privateDir.exists())
            {
            if (!privateDir.mkdir(privateDir.absolutePath()))
                {
                qDebug() << "Could not create widget UI private dir";
                return ;
                }
            }
#else
        dataLocation = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
#endif

        path = QDir::toNativeSeparators(dataLocation);
        qDebug() << "WrtSettings::init() - Final path" << path;

        // Strip off application name
        if (path.endsWith(QCoreApplication::applicationName())) {
            path = path.left( path.length() - QCoreApplication::applicationName().length() -1 );
            path += "/wrt";
            QDir qd(path);
            path = qd.absolutePath();
        }
#endif
        QSettings::setValue("DatabasesStorageLocation", path);
        QSettings::setValue("DatabasesStorageLocation/type", WrtSetting::StringType);
        QSettings::setValue("DatabasesStorageLocation/flags", WrtSetting::ReadOnly | WrtSetting::Hidden);

        // Security Policy files
#ifdef Q_OS_SYMBIAN
        QString securityStorePath = "c:/private/102829CB/policy/1234567890/";
#else
        QString securityStorePath = path + "/security/";
#endif

        if (!QSettings::contains("SecurityStorePath"))
            QSettings::setValue("SecurityStorePath", securityStorePath);
        QSettings::setValue("SecurityStorePath/desc", "SecurityStorePath");
        QSettings::setValue("SecurityStorePath/flags", WrtSetting::ReadOnly | WrtSetting::Hidden);
        QSettings::setValue("SecurityStorePath/cat", WrtSetting::Category2);

        if (!QSettings::contains("SecurityTrustPolicyFile"))
            QSettings::setValue("SecurityTrustPolicyFile", securityStorePath + "browser_trust_policy.xml");
        QSettings::setValue("SecurityTrustPolicyFile/desc", "SecurityTrustPolicyFile");
        QSettings::setValue("SecurityTrustPolicyFile/flags", WrtSetting::ReadOnly | WrtSetting::Hidden);
        QSettings::setValue("SecurityTrustPolicyFile/cat", WrtSetting::Category2);

        if (!QSettings::contains("SecurityAccessPolicyFile"))
            QSettings::setValue("SecurityAccessPolicyFile", securityStorePath + "browser_access_policy.xml");
        QSettings::setValue("SecurityAccessPolicyFile/desc", "SecurityAccessPolicyFile");
        QSettings::setValue("SecurityAccessPolicyFile/flags", WrtSetting::ReadOnly | WrtSetting::Hidden);
        QSettings::setValue("SecurityAccessPolicyFile/cat", WrtSetting::Category2);

    // Widgets

        if (!QSettings::contains("WidgetPreInstallPath"))
            QSettings::setValue("WidgetPreInstallPath", "/tmp");
        QSettings::setValue("WidgetPreInstallPath/desc", "Widget Pre-Install Path");
        QSettings::setValue("WidgetPreInstallPath/type", WrtSetting::StringType);
        QSettings::setValue("WidgetPreInstallPath/cat", WrtSetting::Category4);

        if (!QSettings::contains("WidgetPreInstall"))
            QSettings::setValue("WidgetPreInstall", "1");
        QSettings::setValue("WidgetPreInstall/desc", "Widget Pre-Install");
        QSettings::setValue("WidgetPreInstall/type", WrtSetting::ListType);
        QSettings::setValue("WidgetPreInstall/valid", "Enabled;Disabled");
        QSettings::setValue("WidgetPreInstall/values", "1;0");
        QSettings::setValue("WidgetPreInstall/cat", WrtSetting::Category4);


        // Wrt Preformance Tracing
        if (!QSettings::contains("WrtPerfTracingEnabled"))
            QSettings::setValue("WrtPerfTracingEnabled", "0");
        QSettings::setValue("WrtPerfTracingEnabled/desc", "Tracing");
        QSettings::setValue("WrtPerfTracingEnabled/type", WrtSetting::ListType);
        QSettings::setValue("WrtPerfTracingEnabled/valid", "Enabled;Disabled");
        QSettings::setValue("WrtPerfTracingEnabled/values", "1;0");
        QSettings::setValue("WrtPerfTracingEnabled/cat", WrtSetting::Category4);

        if (!QSettings::contains("WrtPerfTracingToDebugOut"))
            QSettings::setValue("WrtPerfTracingToDebugOut", "0");
        QSettings::setValue("WrtPerfTracingToDebugOut/desc", "Tracing To Debug Output");
        QSettings::setValue("WrtPerfTracingToDebugOut/type", WrtSetting::ListType);
        QSettings::setValue("WrtPerfTracingToDebugOut/valid", "Enabled;Disabled");
        QSettings::setValue("WrtPerfTracingToDebugOut/values", "1;0");
        QSettings::setValue("WrtPerfTracingToDebugOut/cat", WrtSetting::Category4);

        if (!QSettings::contains("WrtPerfTracingToFile"))
            QSettings::setValue("WrtPerfTracingToFile", "1");
        QSettings::setValue("WrtPerfTracingToFile/desc", "Tracing To File");
        QSettings::setValue("WrtPerfTracingToFile/type", WrtSetting::ListType);
        QSettings::setValue("WrtPerfTracingToFile/valid", "Enabled;Disabled");
        QSettings::setValue("WrtPerfTracingToFile/values", "1;0");
        QSettings::setValue("WrtPerfTracingToFile/cat", WrtSetting::Category4);

        if (!QSettings::contains("WrtPerfTracingFileName"))
            QSettings::setValue("WrtPerfTracingFileName", "c:/data/WrtPerfTrace.csv");
        QSettings::setValue("WrtPerfTracingFileName/desc", "Tracing File Name");
        QSettings::setValue("WrtPerfTracingFileName/type", WrtSetting::StringType);
        QSettings::setValue("WrtPerfTracingFileName/cat", WrtSetting::Category4);

        if (!QSettings::contains("WrtPerfTracingSampleRate"))
            QSettings::setValue("WrtPerfTracingSampleRate", 0);
        QSettings::setValue("WrtPerfTracingSampleRate/desc", "Tracing Sample Rate (ms)");
        QSettings::setValue("WrtPerfTracingSampleRate/type", WrtSetting::IntType);
        QSettings::setValue("WrtPerfTracingSampleRate/valid", "0:3600000"); // From 0 (disabled) to 1 hour
        QSettings::setValue("WrtPerfTracingSampleRate/cat", WrtSetting::Category4);


        //Temp solution. List has to be maintained at the client side.
        // Made it permananet (for now), by adding the right parameters and the item
        // won't show up in the menues (bug #2697)
        if (!QSettings::contains("SelfDownloadableContentType"))
            QSettings::setValue("SelfDownloadableContentType", "audio/x-pn-realaudio-plugin;audio/x-pn-realaudio;text/vnd.sun.j2me.app-descriptor;application/java-archive;application/x-piptext/x-pcs-gcd;application/vnd.oma.drm.ro+xml;application/vnd.oma.drm.roap-trigger+xml;application/vnd.oma.drm.roap-pdu+xml;text/x-opml;text/xml;application/sdp;video/x-ms-wvx;audio/x-ms-wax;video/x-ms-asx;");
            QSettings::setValue("SelfDownloadableContentType/desc", "Self downloadable content types" );
            QSettings::setValue("SelfDownloadableContentType/flags", WrtSetting::Hidden );

#ifdef Q_OS_SYMBIAN
                        const QString secStoragePath("c:/private/102829CB/mapping/"); //secure storage server private path
                        const QString mappingFileName("mapping.xml");
            QSettings::setValue("FeatureMapping",secStoragePath+mappingFileName);
            QSettings::setValue("FeatureMapping/flags", WrtSetting::Hidden );
            QSettings::setValue("FeatureMapping/cat", WrtSetting::Category2);
#endif
        endGroup();
        sync();
    }
}

QList<WrtSetting> WrtSettings::settings(const QString &group)
{
    QList<WrtSetting> list;

    if (group.isEmpty())
        beginGroup(m_appuid);
    else
        beginGroup(group);
    QStringList keys = childGroups();

    for (int i=0; i < keys.count(); i++)
        list.append(setting(keys[i]));

    endGroup();
    return list;
}

WrtSetting WrtSettings::setting(const QString &key)
{
    WrtSetting setting(key);

    bool appMissing( QSettings::group().isEmpty() && !key.startsWith(m_appuid));
    if (appMissing)
        beginGroup(m_appuid);
    setting.m_value = QSettings::value(key);
    QSettings::beginGroup(key);
    QStringList childkeys = QSettings::allKeys();
    setting.m_group = QSettings::group();
    for (int i=0; i < childkeys.count(); i++){
        if (childkeys[i] == "type")
            setting.m_type = QSettings::value(childkeys[i]).toInt();
        else if (childkeys[i] == "desc")
                setting.m_desc = QSettings::value(childkeys[i]).toString();
        else if (childkeys[i] == "descl")// localized
                setting.m_desc = tr(setting_strings[QSettings::value(childkeys[i]).toInt()]);
        else if (childkeys[i] == "valid")
            setting.setValid(QSettings::value(childkeys[i]).toString());
        else if (childkeys[i] == "values")
            setting.setValues(QSettings::value(childkeys[i]).toString());
        else if (childkeys[i] == "flags")
            setting.setFlags(QSettings::value(childkeys[i]).toInt());
        else if (childkeys[i] == "enum")
            setting.m_enum = QSettings::value(childkeys[i]).toInt();
        else if (childkeys[i] == "cat")
            setting.m_category = QSettings::value(childkeys[i]).toInt();
        }
    endGroup();
    if (appMissing)
        endGroup();

/*
    // translate language of list entries
    for (int i = 0; i < setting.valid().count(); i++)
        setting.m_validList[i] = translate(setting.m_validList[i]);
*/

    return setting;
}

QString WrtSettings::valueAsString(const QString &key, const QVariant &defaultValue)
{
    return value(key, defaultValue).toString();
}

int WrtSettings::valueAsInt(const QString &key, const QVariant &defaultValue)
{
    return value(key, defaultValue).toInt();
}

double WrtSettings::valueAsDouble(const QString &key, const QVariant &defaultValue)
{
    return value(key, defaultValue).toDouble();
}

QVariant WrtSettings::value(const QString &key, const QVariant &defaultValue)
{
    bool appMissing (!key.startsWith(m_appuid));
    if (appMissing)
        beginGroup(m_appuid);
    QVariant val = QSettings::value(key, defaultValue);
    if (appMissing)
        endGroup();
    return val;
}

int WrtSettings::setValue(const QString &key, const int value)
{
    return setValue(key, QVariant(value));
}

int WrtSettings::setValue(const QString &key, const double value)
{
    return setValue(key, QVariant(value));
}

int WrtSettings::setValue(const QString &key, const QString &value)
{
    return setValue(key, QVariant(value));
}

int WrtSettings::setValue(const QString &key, const QVariant &value)
{
    int ret(0);
    bool appMissing (!key.startsWith(m_appuid));
    if (appMissing)
        beginGroup(m_appuid);

    if (contains(key)){
        WrtSetting s = setting(key);
        if (s.flags() & WrtSetting::ReadOnly)
            ret = WRTSETTING_ACCESS_DENIED;// access denied
        else{
            QSettings::setValue(key, value);
            sync();
            emit settingChanged(key);
        }
    }
    else
        ret = WRTSETTING_NOT_FOUND;// not found

    if (appMissing)
        endGroup();

    return ret;
}

// check for changes from external source
void WrtSettings::resync()
{
    QStringList list = allKeys();
    SettingsMap map;
    foreach (QString key, list)
        map.insert(key, value(key));

    sync();

    list = allKeys();
    foreach (QString key, list)
        if (value(key) != map.value(key))
            emit settingChanged(key);
}

int WrtSettings::activate(const QString &key)
{
    int ret(0);
    bool appMissing (!key.startsWith(m_appuid));
    if (appMissing)
        beginGroup(m_appuid);

    if (contains(key)){
        WrtSetting s = setting(key);
        if (s.type() == WrtSetting::CommandType){
            if (appMissing)
                endGroup();
            emit settingActivated(s.value().toInt());
        }
        else
            ret = WRTSETTING_BAD_ARGUMENT;// bad argument
    }
    else
        ret = WRTSETTING_NOT_FOUND;// not found

    if (appMissing && ret)
        endGroup();

    return ret;
}

void WrtSettings::doAction(const int action)
{
    //QString message("Command = ");
    switch (action) {
        case (10009):
            //message.append("clear-databases");
           break;
        case (10011):
           init(true);
           //message.append("reset-factory-settings");
           break;
        default:
           //message.append("?");
           break;
        }
    //QMessageBox test(QMessageBox::Information, "Activated", message, QMessageBox::Close, 0);
    //test.exec();
}

// some settings will trigger other settings to change
void WrtSettings::settingTrigger(const QString &key)
{
//qDebug() << key;
    if (key.endsWith("/SecurityLevel"))
        switch (valueAsInt(key)) {
            case (2): // High
                setValue("Java",0);
                setValue("JavaScript",0);
                setValue("Cookies",0);
                setValue("PopupBlocking",1);
                setValue("LocalStorageDatabase",0);
                break;
            case (1): // Medium
                setValue("Java",1);
                setValue("JavaScript",1);
                setValue("Cookies",1); // 0;1
                setValue("PopupBlocking",1);
                setValue("LocalStorageDatabase",1);
                break;
            case (0): // Low
                setValue("Java",1);
                setValue("JavaScript",1);
                setValue("Cookies",1);
                setValue("PopupBlocking",0);
                setValue("LocalStorageDatabase",1);
                break;
            default:
                break;
        }
}

// get wrt version
QString WrtSettings::getWrtVersion()
{
    return QString(WRTSETTINGS_VERSION_FORMAT).arg(WRT_VERSION_MAJOR)\
           .arg(WRT_VERSION_MINOR).arg(WRT_VERSION_SVN);
}

}
