/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QString>
#include <QDesktopServices>
#include <QUrl>
#include <QDebug>
#include <QProcess>

#ifdef Q_OS_MAEMO6
#include <messaging-service/messagingif.h>
#include <call-ui/CallUiServiceApi>
#endif

#include "customschemehandler.h"

#define HTTP_SCHEME "http"
#define HTTPS_SCHEME "https"
#define FILE_SCHEME "file"
#define SMS_SCHEME "sms"
#define TEL_SCHEME "tel"
#define MAILTO_SCHEME "mailto"
#define RTSP_SCHEME "rtsp"

namespace WRT {
namespace Maemo {

CustomSchemeHandler::CustomSchemeHandler(QObject* parent) : QObject(parent) {
   // SetCustomHandlers();
}

CustomSchemeHandler::~CustomSchemeHandler() {
   // UnsetCustomHandlers();
}

void CustomSchemeHandler::SetCustomHandlers() {
    QDesktopServices::setUrlHandler(HTTP_SCHEME, this, "HandleHttpScheme");
    QDesktopServices::setUrlHandler(HTTPS_SCHEME, this, "HandleHttpScheme");
    QDesktopServices::setUrlHandler(SMS_SCHEME, this, "HandleSmsScheme");
    QDesktopServices::setUrlHandler(TEL_SCHEME, this, "HandleTelScheme");
    QDesktopServices::setUrlHandler(MAILTO_SCHEME, this, "HandleMailtoScheme");
    QDesktopServices::setUrlHandler(RTSP_SCHEME, this, "HandleRtspScheme");
}

void CustomSchemeHandler::UnsetCustomHandlers() {
    QDesktopServices::unsetUrlHandler(HTTP_SCHEME);
    QDesktopServices::unsetUrlHandler(HTTPS_SCHEME);
    QDesktopServices::unsetUrlHandler(SMS_SCHEME);
    QDesktopServices::unsetUrlHandler(TEL_SCHEME);
    QDesktopServices::unsetUrlHandler(MAILTO_SCHEME);
    QDesktopServices::unsetUrlHandler(RTSP_SCHEME);
}

int CustomSchemeHandler::HandleHttpScheme(const QUrl &url) {
    qDebug() << Q_FUNC_INFO;
    QStringList params;

#ifdef Q_OS_MAEMO6
    // starts the process and detaches from it
    // Don't use this in future, because of fremantle support!
    qDebug() << "Launching fennec:" << params;
    params << url.toString();

    if (QProcess::startDetached("/usr/lib/fennec/fennec.sh", params)){
        return 0;
    }
#endif
    return -1;
}

int CustomSchemeHandler::HandleSmsScheme(const QUrl &url) {
    qDebug() << Q_FUNC_INFO;

#ifdef Q_OS_MAEMO6
    QString recipient = url.path();
    QString body = url.queryItemValue("body");
    QStringList recipients = recipient.split(",");
    qDebug() << "Sms: recipient =" << recipients;
    qDebug() << "Sms: body =" << body;
    MessagingIf msgIf;
    msgIf.showSmsEditor(recipients, body);
#endif
    return 0;
}

int CustomSchemeHandler::HandleTelScheme(const QUrl &url) {
    qDebug() << Q_FUNC_INFO;
    QString recipient = url.path(); //TODO: how do we handle tel: links to webpage?
    qDebug() << "Tel =" << recipient;
#ifdef Q_OS_MAEMO6
    CallUi::CallUiServiceApi callApi;
    CallUi::PendingCallRequest* pRequest = callApi.RequestCellularCall(recipient);
    if (!pRequest) {
        qCritical(qPrintable(QString("%1 FAIL: Call request returned NULL").arg(Q_FUNC_INFO)));
        return -1;
    }
    if (pRequest->isError()) {
        qCritical(qPrintable(QString("%1 FAIL: %2 %3").arg(Q_FUNC_INFO).arg(pRequest->errorName()).arg(pRequest->errorMessage())));
        return -1;
    }

   connect( pRequest, SIGNAL(finished(CallUi::PendingCallRequest*)), this, SLOT(callRequestFinished(CallUi::PendingCallRequest*)) );
#endif
   return 0;
}

#ifdef Q_OS_MAEMO6
void CustomSchemeHandler::callRequestFinished(CallUi::PendingCallRequest* pRequest) {
    qDebug() << Q_FUNC_INFO;
    if (pRequest) {
        qDebug() << "Finished id =" << pRequest->callId();
        if (pRequest->isError()) {
            qCritical(qPrintable(QString("%1 FAIL: %2 %3").arg(Q_FUNC_INFO).arg(pRequest->errorName()).arg(pRequest->errorMessage())));
            return;
        }
    }
}
#endif

int CustomSchemeHandler::HandleMailtoScheme(const QUrl &url) {
    qDebug() << Q_FUNC_INFO;
    qDebug() << "Url = " << url.toString();
    return QProcess::execute("/usr/bin/xdg-email", QStringList()<<url.toString() );
}

int CustomSchemeHandler::HandleRtspScheme(const QUrl &url) {
    qDebug() <<Q_FUNC_INFO;
    qDebug() << "Rtsp: handling not implemented yet";
    //TODO: implementation
    return -1;
}

/** Gets called by the WebKit whenever a link is clicked.
  * Delegates the opening of URL to QDesktopServices which
  * on Maemo 5 uses hildon_uri_open for launching proper
  * application.
  */
void CustomSchemeHandler::slotLinkClicked(const QUrl& url) const {
    qDebug() << "Delegating" << url << "to QDesktopServices";
    QDesktopServices::openUrl(url);
}

} // namespace Maemo
} // namespace WRT
