/*
* ============================================================================
*  Name        : qsysinfoserviceinterface.h
*  Part of     : serviceproviders / WRT
*  Description : Interface class for sysinfo service
*  Version     : %version: 1 % << Don't touch! Updated by Synergy at check-out.
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef SYSINFOSERVICEINTERFACE_H
#define SYSINFOSERVICEINTERFACE_H

// INCLUDES
#include "servicebase.h"

//Provider id and version
static const char* KISysInfoId = "com.nokia.ISysInfo/1.0";

// ISysInfo interface id
static const ServiceInterfaceInfo KISysInfoInfo =
{
    KISysInfoId, 0
};

// CLASS DECLARATION

/**
*  The class defines the interfaces for SysInfo service
*  @lib sysinfoserviceplugin.qtplugin
*  @since x.y
*/
class ISysInfo : public IServiceBase
{
public:
    //public interfaces

    /**
    * Gets the properties of the specified channel.
    * This is a synchronous method.
    * @since x.y
    * @param aChannel - Name of the channel as specified in the channel
    *                   descriptor.The supported channels are Charging,
    *                   PhoneLanguage,BluetoothOn,ProductType and
    *                   FirmwareVersion
    * @return Returns the properties of the channel (as a map)
    *
    * The data schemas for these property maps(system data) are described
    * below:
    *
    * Charging Data Schema:
    * ---------------------------------------------------
    * | Key name       | Data type  | Range             |
    * ---------------------------------------------------
    * | chargingStatus | bool       | 0 : Not charging  |
    * |                |            | 1 : Charging      |
    * ---------------------------------------------------
    *
    * PhoneLanguage Data Schema:
    * -------------------------------
    * | Key name       | Data type  |
    * -------------------------------
    * | phoneLanguage  | QString    |
    * |                | eg.en-GB   |
    * -------------------------------
    *
    * Bluetooth Data Schema:
    * ---------------------------------------
    * | Key name | Data type  | Range       |
    * ---------------------------------------
    * | btOn     | bool       | 0 : BT off  |
    * |          |            | 1 : BT on   |
    * ---------------------------------------
    *
    * ProductType Data Schema:
    * -------------------------------
    * | Key name       | Data type  |
    * -------------------------------
    * | productType    | QString    |
    * |                | eg.RM-160  |
    * -------------------------------
    *
    * FirmwareVersion Data Schema:
    * -----------------------------------------------------------------
    * | Key name         | Data type                                  |
    * -----------------------------------------------------------------
    * | firmwareVersion  | QString                                    |
    * |                  | eg.V 06.27.1.0_10-07-2006_RM170_NOKIA E50  |
    * -----------------------------------------------------------------
    *
    */
    virtual QVariant getChannelSync( const QString& aChannel ) = 0;

    /**
    * Gets the properties of the specified channel based on paramters passed.
    * This is a synchronous method.
    * @since x.y
    * @param aChannel - Name of the channel as specified in the channel
    *                   descriptor.The supported channel is ListDriveInfo
    * @param aInputData - Map containing input data having the following schema:
        *
    * ------------------------------------------------------
    * | Key name       | Data type  | Range                |
    * ------------------------------------------------------
    * | DriveName            | int        | 1-25 : these specify |
    * |                      |                      | required drive           |
    * ------------------------------------------------------
    * @return Returns the properties of the channel (as a map)
    *
    * The data schemas for these property maps(system data) are described
    * below:
    *
    * ListDriveInfo Data Schema:
    * ---------------------------------------------------
    * | Key name       | Data type  | Range             |
    * ---------------------------------------------------
    * | totalMemory      | int        | NA                        |
    * ---------------------------------------------------
    * | freeMemory       | int        | NA                        |
    * ---------------------------------------------------
    * | driveName            | QString    | NA                        |
    * ---------------------------------------------------
    * | mediaType            | QString    | MediaNotPresent   |
        *   |                                |                      |   MediaUnknown          |
        *   |                                |                      |   MediaFloppyDisk   |
        *   |                                |                      |   MediaHardDisk         |
        *   |                                |                      |   MediaCdRom            |
        *   |                                |                      |   MediaRam                  |
        *   |                                |                      |   MediaFlash            |
        *   |                                |                      |   MediaRom                  |
        *   |                                |                      |   MediaRemote           |
        *   |                                |                      |   MediaNANDFlash    |
        *   |                                |                      |   MediaRotatingMedia|
    * ---------------------------------------------------
        */
    virtual QVariant getChannelSync( const QString& aChannel , const QMap<QString,QVariant>& aInputData ) = 0;


    /**
    * Sets the properties of the specified channel.
    * This is a synchronous method.
    * @since x.y
    * @param aChannel - Name of the channel as specified in the channel
    *                   descriptor.The supported channels are Vibrate,
    *                   Clipboard,WallPaper
    *
    * @param aInputData - Map containing input data having the following schema:
    * Vibrate Data Schema:
    * ----------------------------------------------------------
    * | Key name          | Data type  | Range                 |
    * ----------------------------------------------------------
    * | vibraDurationSecs | int        | Duration in seconds   |
    * ----------------------------------------------------------
    *
    * Clipboard Data Schema:
    * --------------------------------
    * | Key name        | Data type  |
    * --------------------------------
    * | clipboardString | QString    |
    * --------------------------------
    *
    * WallPaper Data Schema:
    * --------------------------------
    * | Key name         | Data type |
    * --------------------------------
    * | wallpaperFileurl | QString   |
    * --------------------------------

    * @return Returns error map as follows:
    * ---------------------------------------------------
    * | Key name       | Data type  | Range             |
    * ---------------------------------------------------
    * | errCode              | int        | 0 : Success           |
    * |                |            | Else : CJSE error |
    * ---------------------------------------------------
    * | errMessage       | QString    | NA                              |
    * ---------------------------------------------------
    *
    */
    virtual QVariant setChannelSync( const QString& aChannel, const QMap<QString,QVariant>& aInputData) = 0;

    /**
    * Gets the properties of the specified channel.
    * This is an asynchronous method.
    * @since x.y
    * @param aChannel - Name of the channel as specified in the channel
    *                   descriptor.Supported channels are BatteryLevel,
    *                   Network and SignalStrength
    * @return Returns a unique transaction id(within a map) which can be used
    *         to cancel this request using cancel().
    *
    * When the async request is complete, the following signal is emitted:
    *     void asyncCallback( int aError,
    *                         int aTransId,
    *                         QVariant aSysbds InfoData );
    *
    * The system data returned (aSysInfoData) is a map whose data schema
    * is described below(depending on the requested channel):
    *
    * BatteryLevel Data Schema:
    * ----------------------------------------
    * | Key name       | Data type           |
    * ----------------------------------------
    * | batteryLevel   | int (a percentage)  |
    * ----------------------------------------
    *
    * SignalStrength Data Schema:
    * ----------------------------------------
    * | Key name       | Data type           |
    * ----------------------------------------
    * | signalStrength | int (a percentage)  |
    * ----------------------------------------
    *
    * Network Data Schema:
    * ----------------------------------------------------------------------
    * | Key name            | Data type           | Range                  |
    * ----------------------------------------------------------------------
    * | networkName         | QString e.g Airtel  | -                      |
    * ----------------------------------------------------------------------
    * | networkStatus       | QString             | "Available",           |
    * |                     |                     | "Current",             |
    * |                     |                     | "Forbidden"            |
    * ----------------------------------------------------------------------
    * | networkMode         | QString             | "ModeGsm",             |
    * |                     |                     | "ModeCdma",            |
    * |                     |                     | "ModeWcdma"            |
    * ----------------------------------------------------------------------
    * | mobileCountryCode   | QString e.g 404     | -                      |
    * ----------------------------------------------------------------------
    * | networkCode         | QString e.g 45      | -                      |
    * ----------------------------------------------------------------------
    * | locationStatus      | bool                | 0 : Location area code,|
    * |                     |                     |     CellId are invalid |
    * |                     |                     | 1: Location area code, |
    * |                     |                     |    CellId are valid    |
    * ----------------------------------------------------------------------
    * | areaCode            | int e.g 1627        | -                      |
    * ----------------------------------------------------------------------
    * | cellID              | int e.g 40331       | -                      |
    * ----------------------------------------------------------------------
    *
    * @emits void asyncCallback( int aError, int aTransId, QVariant aSysInfoData );
    */
    virtual QVariant getChannelAsync( const QString& aChannel ) = 0;

    /**
    * Cancels the request made by the call to getChannelAsync.
    * This is a synchronous method.
    * @since x.y
    * @param aTransId - Transaction id returned by getChannelAsync.
    * @return Success code (as a map)
    */
    virtual QVariant cancel( const int aTransId ) = 0;

    /**
    * Registers for notifications from a particular channel.
    * This is an asynchronous method. If the notification for a particular
    * channel has already been registered, and subsequently a call made again,
    * then the new handler will replace the older call(s).
    * @since x.y
    * @param aChannel - Name of the channel as specified in the channel
    *                   descriptor.The supported channels are Charging,
    *                   BatteryLevel,Network,SignalStrength and BluetoothOn.
    * @param aTrigger - An optional parameter indicating a threshold value
    *                   at which the user wishes to receive notification.
    *                   This parameter is applicable only for the
    *                   following channels: BatteryLevel, SignalStrength.
    *                   The default value for BatteryLevel is 50.In other cases,
    *                   if the user does not provide the trigger value,then user
    *                   receives notification whenever a change is detected.
    *                   If trigger is specified then user will be notified only
    *                   when detected change value goes below the trigger value.
    *                   e.g. when user specifies the aTrigger as 75, the
    *                   notification will only be sent when the battery level
    *                   falls below 75 percent.
    *                   The trigger values are in terms of percentages(0-100)
    * @return Dummy zero(0) transaction id and error code.
    *
    * When the async request is complete, the following signal is emitted:
    *     void asyncCallback( int aError,
    *                         int aTransId,
    *                         QVariant aSysInfoData );
    *
    * A dummy zero(0) transaction ID is returned in the callback signal.
    *
    * The system data returned (aSysInfoData) is a map whose data schema
    * is described below (depending on the requested channel):
    *
    * Charging Data Schema:
    * ---------------------------------------------------
    * | Key name       | Data type  | Range             |
    * ---------------------------------------------------
    * | chargingStatus | bool       | 0 : Not charging  |
    * |                |            | 1 : Charging      |
    * ---------------------------------------------------
    *
    * BatteryLevel Data Schema:
    * ----------------------------------------
    * | Key name       | Data type           |
    * ----------------------------------------
    * | batteryLevel   | int (a percentage)  |
    * ----------------------------------------
    *
    * SignalStrength Data Schema:
    * ----------------------------------------
    * | Key name       | Data type           |
    * ----------------------------------------
    * | signalStrength | int (a percentage)  |
    * ----------------------------------------
    *
    * Network Data Schema:
    * ----------------------------------------------------------------------
    * | Key name            | Data type           | Range                  |
    * ----------------------------------------------------------------------
    * | networkName         | QString e.g Airtel  | -                      |
    * ----------------------------------------------------------------------
    * | networkStatus       | QString             | "Available",           |
    * |                     |                     | "Current",             |
    * |                     |                     | "Forbidden"            |
    * ----------------------------------------------------------------------
    * | networkMode         | QString             | "ModeGsm",             |
    * |                     |                     | "ModeCdma",            |
    * |                     |                     | "ModeWcdma"            |
    * ----------------------------------------------------------------------
    * | mobileCountryCode   | QString e.g 404     | -                      |
    * ----------------------------------------------------------------------
    * | networkCode         | QString e.g 45      | -                      |
    * ----------------------------------------------------------------------
    * | locationStatus      | bool                | 0 : Location area code,|
    * |                     |                     |     CellId are invalid |
    * |                     |                     | 1: Location area code, |
    * |                     |                     |    CellId are valid    |
    * ----------------------------------------------------------------------
    * | areaCode            | int e.g 1627        | -                      |
    * ----------------------------------------------------------------------
    * | cellID              | int e.g 40331       | -                      |
    * ----------------------------------------------------------------------
    *
    * Bluetooth Data Schema:
    * ---------------------------------------
    * | Key name | Data type  | Range       |
    * ---------------------------------------
    * | btOn     | bool       | 0 : BT off  |
    * |          |            | 1 : BT on   |
    * ---------------------------------------
    */
    virtual QVariant startChannel( const QString& aChannel, int aTrigger ) = 0;

    /**
    * Registers for notifications from a particular channel.
    * This is an asynchronous method. If the notification for a particular
    * channel has already been registered, and subsequently a call made again,
    * then the new handler will replace the older call(s).
    * @since x.y
    * @param aChannel - Name of the channel as specified in the channel
    *                   descriptor.The supported channels are CriticalMemory,
    * @param aInputData - The parameter used to carry a input to the channel.
    * Input data for critical memory:
    * -----------------------------------------------------------------------
    * | Key name            | Data type           | Range                   |
    * -----------------------------------------------------------------------
    * | driveName           | QString                     | -                       |
    * -----------------------------------------------------------------------
    * | criticalMemory      | int                 | Trigger beyond which      |
    * |                     |                     | notification is sent to |
    * |                     |                     | consumer                |
    * -----------------------------------------------------------------------

    * @return Dummy zero(0) transaction id and error code.
    *
    * Please refer the earlier startChannel help.
    */
    virtual QVariant startChannel( const QString& aChannel, const QMap<QString,QVariant>& aInputData) = 0;

    /**
    * Stops the notifications (started by startChannel) for the channel specified.
    * This is a synchronous method.
    * @since x.y
    * @param aChannel - Name of the channel as specified in the channel descriptor.
    * @return Success code(as a map)
    */
    virtual QVariant stopChannel( const QString& aChannel ) = 0;
};

// MACROS

Q_DECLARE_INTERFACE(ISysInfo, KISysInfoId);

#endif //SYSINFOSERVICEINTERFACE_H
