/*
* ============================================================================
*  Name        : qsysinfoservice.cpp
*  Part of     : serviceproviders / WRT
*  Description : Qt class for sysinfo service
*  Version     : %version: 1 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

//Project includes
#include "qsysinfoservice.h"
#include "qsysinfokeys.h"
#include "listdriveinfo.h"

#include <QLocale>
#include <QApplication>
#include <QClipboard>
#include <QX11Info>
#include <qnetworksession.h>
#include <qnetworkconfigmanager.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <gst/gst.h>
#include <gst/interfaces/propertyprobe.h>

//---------------------------------------
//Default constructor
//---------------------------------------
SysInfoService::SysInfoService():m_transIdCount(0),m_Trigger(-1)
{
    // Maps
    m_asyncMap.clear();
    m_startMap.clear();

    // List
    m_ChannelList.append(KChargingInp);
    m_ChannelList.append(KBatteryLevelInp);
    m_ChannelList.append(KNetworkInp);
    m_ChannelList.append(KSignalStrengthInp);
    m_ChannelList.append(KPhoneLanguageInp);
    m_ChannelList.append(KBluetoothOnInp);
    m_ChannelList.append(KProductTypeInp);
    m_ChannelList.append(KFirmwareVersionInp);
    m_ChannelList.append(KHomeNetworkInp);
    m_ChannelList.append(KIMEIInp);
    m_ChannelList.append(KDisplayOrientationInp);
    m_ChannelList.append(KDeviceInputMethodInp);
    m_ChannelList.append(KCountryNameInp);
    m_ChannelList.append(KIMSIInp);
    m_ChannelList.append(KCellularNetworkStatusInp);
    m_ChannelList.append(KSupportedVideoDecodersInp);
    m_ChannelList.append(KWirelessConnSupportInp);
    m_ChannelList.append(KActiveDataConnectionsInp);
    //m_ChannelList.append(KVibrateInp);
    m_ChannelList.append(KFlipOpenInp);
    m_ChannelList.append(KDisplayWallpaperInp);
    m_ChannelList.append(KListDrivesInp);
    m_ChannelList.append(KListDriveInfoInp);
    m_ChannelList.append(KClipboardInp);
    m_ChannelList.append(KPhoneManufacturerInp);
    m_ChannelList.append(KPhoneModelInp);
    m_ChannelList.append(KPlatformNameInp);
    m_ChannelList.append(KPhoneColorDepthInp);
    m_ChannelList.append(KDisplayResolutionInp);
    m_ChannelList.append(KBacklightStateInp);
    //m_ChannelList.append(KKeypadLightStateInp);
    m_ChannelList.append(KActiveProfileInp);
    //m_ChannelList.append(KMicrophoneVolumeInp);
    //m_ChannelList.append(KSpeakerVolumeInp);
    m_ChannelList.append(KPublicIDInp);

    desktop = QApplication::desktop();
    height = desktop->height();
    width = desktop->width();
    connect( desktop,SIGNAL( resized(int) ),this,SLOT( displayResizedCb() ));

}

//---------------------------------------
//Destructor
//---------------------------------------
SysInfoService::~SysInfoService()
{
    disconnect( desktop,SIGNAL( resized(int) ),this, SLOT( displayResizedCb() ));
}

//---------------------------------------
//Implemented from IServiceSecurity.Needed to handle capabilities.
//---------------------------------------
void SysInfoService::setSecuritySession( WRT::SecSession *aSecSession )
{
    _sec_session = aSecSession;
}

//---------------------------------------
//This method is used to get the list of channels supported by the device.
//This is a Synchronous API
//---------------------------------------
QVariant SysInfoService::getChannelList()
{
    return m_ChannelList;
}

//---------------------------------------
//Set the properties of the specified channel.
//---------------------------------------
QVariant SysInfoService::setChannelSync( const QString& aChannel, const QMap<QString,QVariant>& aInputData)
{
    ErrorMap error = CheckingOfErrors( aChannel ,aInputData, m_setChannelSync);
    if ( FALSE == error.empty() )
    {
        return error;
    }

    if ( KVibrateInp == aChannel )
    {
        return NULL;
    }
    else if ( KDisplayWallpaperInp == aChannel )
    {
        const QString display_name = getenv( "DISPLAY" );
        Display *display = XOpenDisplay( display_name.toAscii().data() );
        Window   rootWindow = RootWindow( display, DefaultScreen(display) );
        Atom     atom = XInternAtom( display, "_NET_CURRENT_DESKTOP", FALSE );

        Atom realType;
        int format,status;
        unsigned long n, extra;
        unsigned char* data = NULL;
        status = XGetWindowProperty(display,
                                    rootWindow,
                                    atom,
                                    0L, 16L, 0,
                                    XA_CARDINAL,
                                    &realType,
                                    &format,
                                    &n,
                                    &extra,
                                    &data );
        qint32 num = -1;
        if ( status == Success &&
             realType == XA_CARDINAL &&
             format == 32 &&
             n == 1 &&
             data )
        {
            num = ((qint32*)data)[0];
        }


        if ( NULL != data )
        {
            XFree(data);
        }
        XCloseDisplay( display );

        const QString uri = aInputData.value(KWallpaperFileurl).toString();
        GConfClient *client = gconf_client_get_default();
        if ( NULL != client )
        {
            const QString path = g_strdup_printf("/apps/osso/hildon-desktop/views/%i/bg-image",1 + num);
            if ( !gconf_client_set_string(client, path.toAscii().data(), uri.toAscii().data(), NULL) )
            {
                ErrorMap error = CheckingOfErrors( aChannel ,NULL, m_setWallpaper);
                if ( FALSE == error.empty() )
                {
                    return error;
                }
            }
            g_object_unref(client);
        }
    }
    else if ( KClipboardInp == aChannel )
    {
         QClipboard *clipboard = QApplication::clipboard();
         clipboard->setText( aInputData.value(KClipboardString).toString() , QClipboard::Clipboard );
    }
    else
    {
        error.clear();
        error.insert(KErrCode, NOT_SUPPORTED_ERR);
        error.insert(KErrMessage, "");
        return error;
    }
    error.clear();
    error.insert(KErrCode, 0);
    error.insert(KErrMessage, "");
    return error;
}

//---------------------------------------
//Get the properties of the specified channel.
//This is an synchronous method.
//---------------------------------------
QVariant SysInfoService::getChannelSync( const QString& aChannel , const QMap<QString,QVariant>& aInputData )
{
    ErrorMap error = CheckingOfErrors( aChannel ,NULL, m_getChannelSync);
    if ( FALSE == error.empty() )
    {
        return error;
    }

    if ( KChargingInp == aChannel )
    {
        QDBusInterface interface( HAL_BME_SERVICE,
                                  HAL_BME_PATH,
                                  HAL_BME_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KChargingInp + KErrDBusIf,m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( HAL_BME_GET_PROPERTY,
                                           HAL_BME_CHARGING );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 )
        {
            bool charging = msg.arguments().at(0).toBool();
            OutputDataMap m_outputDataMap;
            m_outputDataMap.insert( KChargingStatus, charging );
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KChargingInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KBatteryLevelInp == aChannel )
    {
        QDBusInterface interface( HAL_BME_SERVICE,
                                  HAL_BME_PATH,
                                  HAL_BME_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KBatteryLevelInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( HAL_BME_GET_PROPERTY,
                                           HAL_BME_CHARGE_LEVEL );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 &&
             msg.arguments().at(0).type() == QVariant::Int )
        {
            qint32 percentage = msg.arguments().at(0).toInt();
            OutputDataMap m_outputDataMap;
            m_outputDataMap.insert( KBatteryLevel, percentage );
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KBatteryLevelInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KNetworkInp == aChannel )
    {
        QDBusInterface interface( PHONENET_SERVICE,
                                  PHONENET_PATH,
                                  PHONENET_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KNetworkInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

        QString net_name;
        QString net_status = KUnknown;
        QString net_mode = KUnknown;
        QString net_ncode;
        QString net_ccode;
        bool net_locstatus = false;
        qint32 net_lac     = 0;
        qint32 net_cellid  = 0;

        QDBusMessage msg_reg = interface.call( PHONENET_GET_REGISTR_STATUS);
        if ( msg_reg.type() == QDBusMessage::ReplyMessage &&
             msg_reg.arguments().count() >= 6 )
        {
            net_ncode.setNum(msg_reg.arguments().at(3).toInt()); //Network code
            net_ccode.setNum(msg_reg.arguments().at(4).toInt()); //Country code

            if ( net_ncode.toInt() > 0 && net_ccode.toInt() > 0 )
            {
                QVariant mode;
                mode.setValue((uchar)NETWORK_HARDCODED_LATIN_OPER_NAME);

                QDBusMessage msg_name = interface.call( PHONENET_GET_OPERATOR_NAME, mode, net_ncode.toUInt(),  net_ccode.toUInt());
                if ( msg_name.type() == QDBusMessage::ReplyMessage &&
                     msg_name.arguments().count() >= 1 &&
                     msg_name.arguments().at(0).type() == QVariant::String )
                {
                    net_name = msg_name.arguments().at(0).toString();
                }
                else
                {
                    ErrorMap error = CheckingOfErrors( aChannel , KNetworkInp + KErrDBusMsg, m_dbusErrors);
                    if ( FALSE == error.empty() )
                    {
                        return error;
                    }
                }

                qint32  net_type = msg_reg.arguments().at(5).toInt(); //Network type
                if ( NETWORK_TYPE_HOME == net_type )
                {
                    net_status = KCurrent;
                }
                else if ( NETWORK_TYPE_PREFERRED == net_type ||
                          NETWORK_TYPE_NOPREF_NOFORBIDDEN == net_type )
                {
                    net_status = KAvailable;
                }
                else if ( NETWORK_TYPE_FORBIDDEN == net_type )
                {
                    net_status = KForbidden;
                }
                else
                {
                    net_status = KUnknown;
                }

                QDBusMessage msg_mode = interface.call( PHONENET_GET_MODE );
                if ( msg_mode.type() == QDBusMessage::ReplyMessage &&
                     msg_mode.arguments().count() >= 1 )
                {
                    qint32  net_type = msg_mode.arguments().at(0).toInt();  //Network mode
                    if ( NETWORK_MODE_GSM == net_type )
                    {
                        net_mode = KModeGsm;
                    }
                    else if ( NETWORK_MODE_3G == net_type )
                    {
                        net_mode = KMode3G;
                    }
                    else
                    {
                        net_mode = KUnknown;
                    }
                }
                else
                {
                    ErrorMap error = CheckingOfErrors( aChannel , KNetworkInp + KErrDBusMsg, m_dbusErrors);
                    if ( FALSE == error.empty() )
                    {
                        return error;
                    }
                }

                net_lac    = msg_reg.arguments().at(1).toInt();                  //Area Code
                net_cellid = msg_reg.arguments().at(2).toInt();                  //Cell Id
                net_locstatus = (net_lac == 0 && net_cellid == 0)? false: true;  //Loc status
            }
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KNetworkInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

        OutputDataMap m_outputDataMap;
        m_outputDataMap.insert(KNetworkName, net_name );
        m_outputDataMap.insert(KNetworkStatus, net_status );
        m_outputDataMap.insert(KNetworkMode, net_mode );
        m_outputDataMap.insert(KMobileCountryCode,  net_ccode);
        m_outputDataMap.insert(KNetworkCode,  net_ncode);
        m_outputDataMap.insert(KLocationStatus, net_locstatus);
        m_outputDataMap.insert(KAreaCode, net_lac );
        m_outputDataMap.insert(KCellID, net_cellid );
        return m_outputDataMap;
    }
    else if ( KSignalStrengthInp == aChannel )
    {
        QDBusInterface interface( PHONENET_SERVICE,
                                  PHONENET_PATH,
                                  PHONENET_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KSignalStrengthInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( PHONENET_GET_SIGNAL_STRENGTH );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 )
        {
             qint32 signal_strength = msg.arguments().at(0).toInt();
             OutputDataMap m_outputDataMap;
             m_outputDataMap.insert(KSignalStrength, signal_strength);
             return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KSignalStrengthInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KPhoneLanguageInp == aChannel )
    {
        QLocale locale = QLocale::system();
        QString lName = locale.name().replace( QChar('_'), "-" );

        OutputDataMap m_outputDataMap;
        m_outputDataMap.insert( KPhoneLanguage, lName );
        return m_outputDataMap;
    }
    else if ( KBluetoothOnInp == aChannel )
    {
        QDBusInterface interface( BLUEZ_SERVICE,
                                  BLUEZ_PATH,
                                  BLUEZ_IF_MANAGER,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KBluetoothOnInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( BLUEZ_GET_PATH );
        if ( msg.type() == QDBusMessage::ReplyMessage )
        {
            QString path = msg.arguments().at(0).value<QDBusObjectPath>().path();
            QDBusInterface interface( BLUEZ_SERVICE,
                                      path,
                                      BLUEZ_IF_ADAPTER,
                                      QDBusConnection::systemBus());
            if ( FALSE == interface.isValid() )
            {
                ErrorMap error = CheckingOfErrors( aChannel ,KBluetoothOnInp + KErrDBusMsg, m_dbusErrors);
                if ( FALSE == error.empty() )
                {
                    return error;
                }
            }
            QDBusReply< QMap<QString,QVariant> > reply = interface.call( BLUEZ_GET_PROPERTY );
            if ( reply.isValid() )
            {
                bool powered = reply.value().value( BLUEZ_POWERED ).toBool();
                OutputDataMap m_outputDataMap;
                m_outputDataMap.insert(KBtOn, powered);
                return m_outputDataMap;
            }
            else
            {
                ErrorMap error = CheckingOfErrors( aChannel ,KBluetoothOnInp + KErrDBusMsg, m_dbusErrors);
                if ( FALSE == error.empty() )
                {
                    return error;
                }
            }
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KBluetoothOnInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KProductTypeInp == aChannel )
    {

        QDBusInterface interface( SYSINFO_SERVICE,
                                  SYSINFO_PATH,
                                  SYSINFO_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KProductTypeInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( SYSINFO_GET_CONFIG,
                                           SYSINFO_PRODUCT_TYPE );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 &&
             msg.arguments().at(0).type() == QVariant::ByteArray )
        {
            QString product_type = msg.arguments().at(0).toByteArray();
            OutputDataMap m_outputDataMap;
            m_outputDataMap.insert(KProductType,product_type);
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KProductTypeInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

    }
    else if ( KFirmwareVersionInp == aChannel )
    {
        QDBusInterface interface( SYSINFO_SERVICE,
                                  SYSINFO_PATH,
                                  SYSINFO_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KFirmwareVersionInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( SYSINFO_GET_CONFIG,
                                           SYSINFO_FIRMWARE );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 &&
             msg.arguments().at(0).type() == QVariant::ByteArray )
        {
             QString firmware_version = msg.arguments().at(0).toByteArray();
             OutputDataMap m_outputDataMap;
             m_outputDataMap.insert(KFirmwareVersion,firmware_version);
             return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KFirmwareVersionInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KIMEIInp == aChannel )
    {
        QDBusInterface interface( SIM_SERVICE,
                                  SIM_PATH_IMEI,
                                  SIM_IF_IMEI,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KIMEIInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( SIM_GET_IMEI );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 &&
             msg.arguments().at(0).type() == QVariant::String )
        {
            QString imei = msg.arguments().at(0).toString();
            OutputDataMap m_outputDataMap;
            m_outputDataMap.insert(KIMEI,imei);
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KIMEIInp + KErrDBusMsg ,m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KDisplayOrientationInp == aChannel )
    {
        OutputDataMap m_outputDataMap;
        if ( width > height )
        {
            m_outputDataMap.insert( KDisplayOrientation, KLandscape );
        }
        else
        {
            m_outputDataMap.insert( KDisplayOrientation, KPortrait );
        }
        return m_outputDataMap;
    }
    else if ( KDeviceInputMethodInp == aChannel )
    {
        bool keypad = FALSE;
        bool touchpad = FALSE;

        QDBusInterface interface( HAL_IM_SERVICE,
                                  HAL_IM_PATH,
                                  HAL_IM_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KDeviceInputMethodInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

        QDBusMessage msg_key = interface.call( QDBus::Block,
                                               HAL_IM_FIND_DEVICE,
                                               HAL_IM_KEYPAD );
        if ( msg_key.type() == QDBusMessage::ReplyMessage &&
             msg_key.arguments().count() >= 1 &&
             msg_key.arguments().at(0).type() == QVariant::StringList )
        {
            QStringList list_key = msg_key.arguments().at(0).toStringList();
            if ( FALSE == list_key.isEmpty())
            {
                keypad = TRUE;
            }
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KDeviceInputMethodInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

        QDBusMessage msg_touch = interface.call( QDBus::Block,
                                                 HAL_IM_FIND_DEVICE,
                                                 HAL_IM_TOUCHPAD );
        if ( msg_touch.type() == QDBusMessage::ReplyMessage &&
             msg_touch.arguments().count() >= 1 &&
             msg_touch.arguments().at(0).type() == QVariant::StringList )
        {
            QStringList list_touch = msg_touch.arguments().at(0).toStringList();
            if ( FALSE == list_touch.isEmpty())
            {
                touchpad = TRUE;
            }
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KDeviceInputMethodInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

        OutputDataMap m_outputDataMap;
        if ( TRUE == touchpad && TRUE == keypad  )
        {
            m_outputDataMap.insert( KDeviceInputMethod, KDual );
        }
        if ( TRUE == touchpad && FALSE == keypad  )
        {
            m_outputDataMap.insert( KDeviceInputMethod, KTouch );
        }
        if ( FALSE == touchpad && TRUE == keypad  )
        {
            m_outputDataMap.insert( KDeviceInputMethod, KNonTouch );
        }
        return m_outputDataMap;
    }
    else if ( KCountryNameInp == aChannel )
    {
        QDBusInterface interface( CLOCKD_SERVICE,
                                  CLOCKD_PATH,
                                  CLOCKD_INTERFACE,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KCountryNameInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

        QDBusMessage msg = interface.call( GET_TZ );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 &&
             msg.arguments().at(0).type() == QVariant::String )
        {
            QString timezone = msg.arguments().at(0).toString().remove(QChar(':'));
            QFile zone("/usr/share/zoneinfo/zone.tab");
            QMap<QString,QString> zoneMap;
            if ( TRUE == zone.open(QFile::ReadOnly) )
            {
                while ( FALSE == zone.atEnd() )
                {
                    QByteArray line = zone.readLine();
                    if (line.at(0)=='#')
                    {
                        continue;
                    }

                    QList<QByteArray> vars = line.split('\t');
                    if ( vars.count() >= 3 )
                    {
                       zoneMap.insert(vars.at(2).trimmed(),vars.at(0).trimmed());//time zone ,country
                    }
                }
                zone.close();

                OutputDataMap m_outputDataMap;
                m_outputDataMap.insert(KCountryName,zoneMap.value(timezone));
                return m_outputDataMap;
            }
            else
            {
                OutputDataMap m_outputDataMap;
                m_outputDataMap.insert(KCountryName,QString("GB"));
                return m_outputDataMap;
            }
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KCountryNameInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KIMSIInp == aChannel )
    {
        QDBusInterface interface( SIM_SERVICE,
                                  SIM_PATH_IMSI,
                                  SIM_IF_IMSI,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,  KIMSIInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( SIM_GET_IMSI );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 &&
             msg.arguments().at(0).type() == QVariant::String )
        {
            QString imsi = msg.arguments().at(0).toString();
            OutputDataMap m_outputDataMap;
            m_outputDataMap.insert(KIMSI,imsi);
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KIMSIInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KCellularNetworkStatusInp == aChannel )
    {
        QDBusInterface interface( PHONENET_SERVICE,
                                  PHONENET_PATH,
                                  PHONENET_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KCellularNetworkStatusInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( PHONENET_GET_REGISTR_STATUS);
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 )
        {
            qint32 reg_status = msg.arguments().at(0).toInt();
            OutputDataMap m_outputDataMap;
            if ( NETWORK_REG_STATUS_HOME == reg_status )
            {
                m_outputDataMap.insert(  KCellularNetworkStatus, KRegisteredOnHomeNetwork );
            }
            else if ( NETWORK_REG_STATUS_ROAM == reg_status ||
                     NETWORK_REG_STATUS_ROAM_BLINK == reg_status)
            {
                m_outputDataMap.insert(  KCellularNetworkStatus, KRegisteredRoaming );
            }
            else if ( NETWORK_REG_STATUS_NOSERV == reg_status ||
                     NETWORK_REG_STATUS_NOSERV_NOTSEARCHING == reg_status ||
                     NETWORK_REG_STATUS_NOSERV_SIM_REJECTED_BY_NW == reg_status)
            {
                m_outputDataMap.insert(  KCellularNetworkStatus, KNotRegisteredNoService);
            }
            else if ( NETWORK_REG_STATUS_NOSERV_SEARCHING == reg_status )
            {
                m_outputDataMap.insert(  KCellularNetworkStatus, KNotRegisteredSearching);
            }
            else if ( NETWORK_REG_STATUS_NOSERV_NOSIM == reg_status )
            {
                m_outputDataMap.insert(  KCellularNetworkStatus, KNotRegisteredEmergencyOnly);
            }
            else if ( NETWORK_REG_STATUS_POWER_OFF == reg_status ||
                     NETWORK_REG_STATUS_NSPS == reg_status ||
                     NETWORK_REG_STATUS_NSPS_NO_COVERAGE == reg_status )
            {
                m_outputDataMap.insert(  KCellularNetworkStatus, KRegistrationDenied);
            }
            else
            {
                m_outputDataMap.insert(  KCellularNetworkStatus, KRegistrationUnknown);
            }
            //KRegisteredBusy
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KCellularNetworkStatusInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if (  KSupportedVideoDecodersInp == aChannel )
    {
        //VideoCodecList availableMimeTypes;
        QList< QVariant > availableMimeTypesForSP;

        if ( TRUE == gst_init_check(NULL,NULL,NULL) )
        {
            // Iterate over all audio and video decoders and extract mime types from sink caps
            GList* factoryList = gst_registry_get_feature_list(gst_registry_get_default (), GST_TYPE_ELEMENT_FACTORY);

            for ( GList* iter = (factoryList) ; iter != NULL ; iter = iter->next )
            {
                GstPluginFeature *feature = GST_PLUGIN_FEATURE(iter->data);
                if ( NULL != feature)
                {
                    QString klass = gst_element_factory_get_klass(GST_ELEMENT_FACTORY(feature));

                    if ( klass == QLatin1String(GST_CODEC_TYPE_VIDEO) )
                    {
                        const GList *static_templates = NULL;
                        GstElementFactory *factory = GST_ELEMENT_FACTORY(feature);
                        if ( NULL != factory )
                        {
                            static_templates = gst_element_factory_get_static_pad_templates(factory);

                            for (; static_templates != NULL ; static_templates = static_templates->next )
                            {
                                GstStaticPadTemplate *padTemplate = (GstStaticPadTemplate *) static_templates->data;
                                if ( padTemplate &&
                                     padTemplate->direction == GST_PAD_SINK )
                                {
                                    GstCaps *caps = gst_static_pad_template_get_caps (padTemplate);

                                    if ( caps )
                                    {
                                        QMap <QString, QVariant> mimeDetails;
                                        mimeDetails.insert( Kmanufacturer,gst_element_factory_get_author (GST_ELEMENT_FACTORY(feature)) );
                                        mimeDetails.insert( Kidentifier,QString::fromUtf8(gst_structure_get_name (gst_caps_get_structure (caps, 0))) );
                                        mimeDetails.insert( KmaxBitrate,"" );
                                        mimeDetails.insert( KisHWAccelerationEnabled,"" );

                                        if ((GST_PLUGIN_FEATURE (iter->data))->plugin_name)
                                        {
                                            GstPlugin *plugin = gst_registry_find_plugin (gst_registry_get_default (),
                                                                                          GST_PLUGIN_FEATURE (iter->data)->plugin_name);
                                            mimeDetails.insert( Kversion, (plugin)?(plugin->desc.version):("N/A") );
                                        }
                                        availableMimeTypesForSP.append(static_cast<QVariant>(mimeDetails));
                                    }
                                }
                            }
                        }
                    }
                }
            }
            if ( NULL != factoryList )
                g_list_free( factoryList );
        }
        return static_cast<QVariant>(availableMimeTypesForSP);
    }
    else if (  KWirelessConnSupportInp == aChannel )
    {
        QStringList AvailableWIInterfaces;

        QDBusInterface interface_hal( HAL_WCS_SERVICE,
                                      HAL_WCS_PATH,
                                      HAL_WCS_IF,
                                      QDBusConnection::systemBus());

        if ( FALSE == interface_hal.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KWirelessConnSupportInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

        QDBusMessage wi_wlan = interface_hal.call( QDBus::Block,
                                                   HAL_WCS_FIND_DEVICE,
                                                   HAL_WI_WLAN );
        if ( wi_wlan.type() == QDBusMessage::ReplyMessage &&
             wi_wlan.arguments().count() >= 1 &&
             wi_wlan.arguments().at(0).type() == QVariant::StringList )
        {
            QStringList list_key = wi_wlan.arguments().at(0).toStringList();
            if ( FALSE == list_key.isEmpty())
            {
                AvailableWIInterfaces.append(WI_NAME_WLAN);
            }
        }

        QDBusInterface interface_bluez( BLUEZ_SERVICE,
                                        BLUEZ_PATH,
                                        BLUEZ_IF_MANAGER,
                                        QDBusConnection::systemBus());
        if ( FALSE == interface_bluez.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KWirelessConnSupportInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage wi_bt = interface_bluez.call( BLUEZ_GET_PATH );
        if ( wi_bt.type() == QDBusMessage::ReplyMessage )
        {
            QString path = wi_bt.arguments().at(0).value<QDBusObjectPath>().path();
            if (FALSE == path.isEmpty())
            {
                AvailableWIInterfaces.append(WI_NAME_BLUETOOTH);
            }

        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KWirelessConnSupportInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

        //N900 supported following connection(temporary hardcode):
        AvailableWIInterfaces.append(WI_NAME_IRDA);
        AvailableWIInterfaces.append(WI_NAME_GSM);
        AvailableWIInterfaces.append(WI_NAME_WCDMA);

        OutputDataMap m_outputDataMap;
        m_outputDataMap.insert(KWirelessConnSupport,AvailableWIInterfaces);
        return m_outputDataMap;
    }
    else if (  KActiveDataConnectionsInp == aChannel )
    {
        QVariantList activeConnections;
        QtMobility::QNetworkConfigurationManager manager;
        QList<QtMobility::QNetworkConfiguration> activeConfigurations = manager.allConfigurations (QtMobility::QNetworkConfiguration::Discovered);
        for (qint32 i=0; i<activeConfigurations.count(); i++)
        {
            OutputDataMap connectionRetInf;
            connectionRetInf.insert( KConnectionType ,activeConfigurations.at(i).bearerName() );
            connectionRetInf.insert( KConnectionName ,activeConfigurations.at(i).name() );
            activeConnections.append( connectionRetInf );
        }
        return activeConnections;
    }
    else if (  KListDrivesInp == aChannel )
    {
        OutputDataMap m_outputDataMap;
        ListDriveInfo m_list_drive;
        m_list_drive.init();
        QStringList infRes = m_list_drive.getNames();
        m_outputDataMap.insert(KListOfDrives, infRes);
        return m_outputDataMap;
    }
    else if (  KListDriveInfoInp == aChannel )
    {
        OutputDataMap m_outputDataMap;
        ListDriveInfo m_list_drive;
        m_list_drive.init();
        if ( TRUE == aInputData.contains(KInfoDriveName) )
        {
            QString driveName = aInputData.value(KInfoDriveName).toString();
            m_outputDataMap = m_list_drive.getInfo(driveName);
        }
        return m_outputDataMap;
    }
    else if ( KClipboardInp == aChannel )
    {
         QClipboard *clipboard = QApplication::clipboard();
         QString text = clipboard->text(QClipboard::Clipboard);
         OutputDataMap m_outputDataMap;
         if (!text.isNull())
         {
            m_outputDataMap.insert( KClipboard,text );
         }
         else
         {
            m_outputDataMap.insert( KClipboard,NULL );
         }
         return m_outputDataMap;
    }
    else if ( KPhoneManufacturerInp == aChannel )
    {
        QDBusInterface interface( HAL_HARD_SERVICE,
                                  HAL_HARD_PATH,
                                  HAL_HARD_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KPhoneManufacturerInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( HAL_HARD_GET_PROPERTY,
                                           HAL_HARD_MANUFACTURER );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 &&
             msg.arguments().at(0).type() == QVariant::String )
        {
            QString manufacturer = msg.arguments().at(0).toString();
            OutputDataMap m_outputDataMap;
            m_outputDataMap.insert( KPhoneManufacturer, manufacturer );
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,  KPhoneManufacturerInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KPhoneModelInp == aChannel )
    {
        QString model = "N900";
        OutputDataMap m_outputDataMap;
        m_outputDataMap.insert( KPhoneModel, model );
        return m_outputDataMap;
    }
    else if ( KPlatformNameInp == aChannel )
    {
        QString platform_name = "Maemo5";
        OutputDataMap m_outputDataMap;
        m_outputDataMap.insert( KPlatformName, platform_name );
        return m_outputDataMap;
    }
    else if ( KPhoneColorDepthInp == aChannel )
    {
        QDesktopWidget widget;
        qint32 depth = widget.screen(desktop->primaryScreen())->x11Info().depth();
        quint64 color_depth = 1;

        for (qint32 i = 0; i < depth; i++)
            color_depth *= 2;

        OutputDataMap m_outputDataMap;
        m_outputDataMap.insert( KPhoneColorDepth, color_depth );
        return m_outputDataMap;
    }
    else if ( KDisplayResolutionInp == aChannel )
    {
        OutputDataMap m_outputDataMap;
        if ( width > height )
        {
            m_outputDataMap.insert( KHeightDisplay, height );
            m_outputDataMap.insert( KWidthDisplay, width );
        }
        else
        {
            m_outputDataMap.insert( KHeightDisplay, width  );
            m_outputDataMap.insert( KWidthDisplay, height );
        }
        return m_outputDataMap;
    }
    else if ( KBacklightStateInp == aChannel )
    {
        QDBusInterface interface( MCE_SERVICE,
                                  MCE_REQUEST_PATH,
                                  MCE_REQUEST_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KBacklightStateInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( MCE_DISPLAY_GET_STATUS );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 &&
             msg.arguments().at(0).type() == QVariant::String )
        {
            QString status = msg.arguments().at(0).toString();
            OutputDataMap m_outputDataMap;
            if ( "off" == status )
            {
                m_outputDataMap.insert(KBacklightState,false);
            }
            else
            {
                m_outputDataMap.insert(KBacklightState,true);
            }
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KBacklightStateInp + KErrDBusMsg , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KKeypadLightStateInp == aChannel )
    {
        return NULL;
    }
    else if ( KActiveProfileInp == aChannel )
    {
        bool vibration = FALSE;
        qint32 r_vol = 0;
        qint32 m_vol = 0;

        QDBusInterface interface( PROFILED_SERVICE,
                                  PROFILED_PATH,
                                  PROFILED_INTERFACE,
                                  QDBusConnection::sessionBus() );
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KActiveProfileInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

        QDBusReply<QString> profileNameReply = interface.call( GET_PROFILE );
        if ( TRUE == profileNameReply.isValid() )
        {
            const QString profileName = profileNameReply.value();

            QDBusReply<QString> vibratingAlertEnabledReply = interface.call( GET_VALUE , profileName, GET_VIBRATION );
            if ( TRUE == vibratingAlertEnabledReply.isValid() )
            {
                vibration = vibratingAlertEnabledReply.value() == "On";
            }
            else
            {
                ErrorMap error = CheckingOfErrors( aChannel , KActiveProfileInp + KErrDBusMsg , m_dbusErrors);
                if ( FALSE == error.empty() )
                {
                    return error;
                }
            }

            QDBusReply<QString> ringingAlertVolumeReply = interface.call( GET_VALUE, profileName, GET_R_VOLUME );
            if ( TRUE == ringingAlertVolumeReply.isValid() )
            {
                r_vol = ringingAlertVolumeReply.value().toInt();
            }
            else
            {
                ErrorMap error = CheckingOfErrors( aChannel , KActiveProfileInp + KErrDBusMsg , m_dbusErrors);
                if ( FALSE == error.empty() )
                {
                    return error;
                }
            }

            QDBusReply<QString> msgAlertVolumeReply = interface.call( GET_VALUE, profileName, GET_M_VOLUME );
            if ( TRUE == msgAlertVolumeReply.isValid() )
            {
                m_vol = msgAlertVolumeReply.value().toInt();
            }
            else
            {
                ErrorMap error = CheckingOfErrors( aChannel , KActiveProfileInp + KErrDBusMsg , m_dbusErrors);
                if ( FALSE == error.empty() )
                {
                    return error;
                }
            }
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KActiveProfileInp + KErrDBusMsg , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        OutputDataMap m_outputDataMap;
        m_outputDataMap.insert( KMsgRingtoneVolume, m_vol );
        m_outputDataMap.insert( KRingtoneVolume, r_vol );
        m_outputDataMap.insert( KVibrationActive, vibration );
        return m_outputDataMap;
    }
    else if ( KMicrophoneVolumeInp == aChannel )
    {
        return NULL;
    }
    else if ( KSpeakerVolumeInp == aChannel )
    {
        return NULL;
    }
    else if ( KPublicIDInp == aChannel )
    {
        QDBusInterface interface( INFO_SERVICE,
                                  INFO_PATH,
                                  INFO_INTERFACE,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KPublicIDInp  + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( INFO_GETPUBLICID );
        if ( msg.type() == QDBusMessage::ReplyMessage &&
             msg.arguments().count() >= 1 &&
             msg.arguments().at(0).type() == QVariant::ByteArray )
        {
            QString publicId;
            QByteArray var = msg.arguments().at(0).toByteArray();
            for ( int i=0; i < var.size(); i++ )
            {
                publicId.append(QString("%1").arg(var[i], 2, 16, QChar('0')));
            }

            OutputDataMap m_outputDataMap;
            m_outputDataMap.insert(KPublicID,publicId);
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KPublicIDInp  + KErrDBusMsg , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else
    {
        return NULL;
    }
    return NULL;
}

//---------------------------------------
//Get the properties of the specified channel.
//This is a synchronous method.
//---------------------------------------
QVariant SysInfoService::getChannelSync( const QString& aChannel )
{
    QMap<QString,QVariant> aInputData;
    return getChannelSync( aChannel, aInputData );
}

//---------------------------------------
//Get the properties of the specified channel.
//This is an asynchronous method.
//---------------------------------------
QVariant SysInfoService::getChannelAsync( const QString& aChannel )
{
    ErrorMap error = CheckingOfErrors( aChannel , NULL, m_getChannelASync);
    if ( FALSE == error.empty() )
    {
        return error;
    }


    if ( TRUE == m_asyncMap.contains( aChannel ) )
    {
        m_asyncMap.remove( aChannel );
    }

    if ( KBatteryLevelInp == aChannel )
    {
        QList<QVariant> xArg;
        xArg.append( HAL_BME_CHARGE_LEVEL );
        QDBusInterface interface( HAL_BME_SERVICE,
                                  HAL_BME_PATH,
                                  HAL_BME_IF,
                                  QDBusConnection::systemBus());

        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KBatteryLevelInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        if ( FALSE == interface.callWithCallback ( HAL_BME_GET_PROPERTY,
                                                  xArg,
                                                  this,
                                                  SLOT(asyncBatteryLevelChannelCb(const QDBusMessage&)) ))
        {
            ErrorMap error = CheckingOfErrors( aChannel , KBatteryLevelInp + KErrDBusCb, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KNetworkInp == aChannel )
    {
        QDBusInterface interface( PHONENET_SERVICE,
                                  PHONENET_PATH,
                                  PHONENET_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KNetworkInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        if ( FALSE == interface.callWithCallback( PHONENET_GET_REGISTR_STATUS,
                                                 QList<QVariant>(),
                                                 this,
                                                 SLOT(asyncNetworkChannelCb(const QDBusMessage&)) ))
        {
            ErrorMap error = CheckingOfErrors( aChannel ,  KNetworkInp + KErrDBusCb, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KSignalStrengthInp == aChannel )
    {
        QDBusInterface interface( PHONENET_SERVICE,
                                  PHONENET_PATH,
                                  PHONENET_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KSignalStrengthInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        if ( FALSE == interface.callWithCallback( PHONENET_GET_SIGNAL_STRENGTH,
                                                 QList<QVariant>(),
                                                 this,
                                                 SLOT(asyncSignalStrengthChannelCb(const QDBusMessage&)) ))
        {
            ErrorMap error = CheckingOfErrors( aChannel , KSignalStrengthInp + KErrDBusCb, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KHomeNetworkInp == aChannel )
    {
        QDBusInterface interface( HN_SERVICE,
                                  HN_PATH,
                                  HN_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KHomeNetworkInp + KErrDBusIf , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        if ( FALSE == interface.callWithCallback( HN_GET_OPERATOR_NAME,
                                                  QList<QVariant>(),
                                                  this,
                                                  SLOT(asyncHomeNetworkChannelCb(const QDBusMessage&)) ))
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KHomeNetworkInp + KErrDBusCb, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }

    qint32 transId = ++m_transIdCount;
    m_asyncMap.insert( aChannel, transId );
    return transId;
}

//---------------------------------------
//Stops the notifications (started by getChannelAsync) for the channel specified.
//This is a synchronous method.
//---------------------------------------
QVariant SysInfoService::cancel( const int aTransId )
{
    QString aChannel = m_asyncMap.key( aTransId,"not_found" );
    ErrorMap error = CheckingOfErrors( aChannel , NULL, m_cancel);
    if ( FALSE == error.empty() )
    {
        return error;
    }

    m_asyncMap.remove( aChannel );
    return NULL;
}

//---------------------------------------
//Registers for notifications from a particular channel.
//This is an asynchronous method.
//If the notification for a particular channel has already been registered,
//and subsequently a call made again, api will return an error.
//---------------------------------------
QVariant SysInfoService::startChannel( const QString& aChannel, int aTrigger )
{
    ErrorMap error = CheckingOfErrors( aChannel , QVariant(aTrigger), m_startChannel);
    if ( FALSE == error.empty() )
    {
        return error;
    }


    if ( TRUE == m_startMap.contains( aChannel ) )
    {
        m_startMap.remove( aChannel );
        m_triggerMap.remove( aChannel );
    }

    if ( aTrigger != -1 && ( KBatteryLevelInp == aChannel || KSignalStrengthInp == aChannel ) )
    {
        m_triggerMap.insert( aChannel,aTrigger );
    }

    if ( KChargingInp == aChannel )
    {
        if ( FALSE == QDBusConnection::systemBus().connect( HAL_BME_SERVICE,
                                                           HAL_BME_PATH,
                                                           HAL_BME_IF,
                                                           HAL_BME_CHANGE_PROPERTY,
                                                           this,
                                                           SLOT(startChargingChannelCb(const QDBusMessage& ))) )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KChargingInp + KErrDBusCon, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KBatteryLevelInp == aChannel )
    {
        if ( FALSE == QDBusConnection::systemBus().connect( HAL_BME_SERVICE,
                                                           HAL_BME_PATH,
                                                           HAL_BME_IF,
                                                           HAL_BME_CHANGE_PROPERTY,
                                                           this,
                                                           SLOT(startBatteryLevelChannelCb(const QDBusMessage& ))) )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KBatteryLevelInp + KErrDBusCon, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KNetworkInp == aChannel )
    {
        if ( FALSE == QDBusConnection::systemBus().connect( PHONENET_SERVICE,
                                                           PHONENET_PATH,
                                                           PHONENET_IF,
                                                           PHONENET_REGISTR_STATUS_CHANGED,
                                                           this,
                                                           SLOT(startNetworkChannelCb(const QDBusMessage& ))) )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KNetworkInp + KErrDBusCon, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KSignalStrengthInp == aChannel )
    {
        if ( FALSE == QDBusConnection::systemBus().connect( PHONENET_SERVICE,
                                                           PHONENET_PATH,
                                                           PHONENET_IF,
                                                           PHONENET_CHANGE_SIGNAL_STRENGTH,
                                                           this,
                                                           SLOT(startSignalStrengthChannelCb(const QDBusMessage& ))) )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KSignalStrengthInp + KErrDBusCon, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KBluetoothOnInp == aChannel )
    {
        QDBusInterface interface( BLUEZ_SERVICE,
                                  BLUEZ_PATH,
                                  BLUEZ_IF_MANAGER,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KBluetoothOnInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( BLUEZ_GET_PATH );
        if ( msg.type() == QDBusMessage::ReplyMessage )
        {
            QString path = msg.arguments().at(0).value<QDBusObjectPath>().path();
            if ( FALSE ==  QDBusConnection::systemBus().connect( BLUEZ_SERVICE ,
                                                                path,
                                                                BLUEZ_IF_ADAPTER,
                                                                BLUEZ_CHANGE_PROPERTY,
                                                                this,
                                                                SLOT(startBluetoothOnChannelCb(const QDBusMessage& ))) )
            {
                ErrorMap error = CheckingOfErrors( aChannel ,KBluetoothOnInp + KErrDBusCon, m_dbusErrors);
                if ( FALSE == error.empty() )
                {
                    return error;
                }
            }
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,KBluetoothOnInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }

    }
    else if ( KFlipOpenInp == aChannel )
    {
        //if( FALSE ==  QDBusConnection::systemBus().connect( HAL_FLIP_SERVICE,
        //                                                    HAL_FLIP_PATH,
        //                                                    HAL_FLIP_IF,
        //                                                    HAL_FLIP_CHANGE_PROPERTY,
        //                                                    this,
        //                                                    SLOT(startFlipOpenChannelCb(const QDBusMessage& ))) )
        //{
        //    ErrorMap error = CheckingOfErrors( aChannel , KFlipOpenInp  + KErrDBusCon, m_dbusErrors);
        //    if ( FALSE == error.empty() )
        //    {
        //        return error;
        //    }
        //}
    }
    else
    {
        return NULL;
    }

    qint32 transId = ++m_transIdCount;
    m_startMap.insert( aChannel, transId );
    return (qint32)transId;
}

//---------------------------------------
//Registers for notifications from a particular channel.
//This is an asynchronous method.
//If the notification for a particular channel has already been registered,
//and subsequently a call made again, api will return an error.
//---------------------------------------
QVariant SysInfoService::startChannel(const QString& aChannel , const QMap<QString, QVariant>& aInputData)
{
    Q_UNUSED(aChannel);
    Q_UNUSED(aInputData);
    return NULL;
}

//---------------------------------------
//Stops the notifications (started by startChannel) for the channel specified.
//This is a synchronous method.
//---------------------------------------
QVariant SysInfoService::stopChannel( const QString& aChannel )
{
    ErrorMap error = CheckingOfErrors( aChannel , NULL, m_stopChannel);
    if ( FALSE == error.empty() )
    {
        return error;
    }

    if ( KChargingInp == aChannel )
    {
        QDBusConnection::systemBus().disconnect( HAL_BME_SERVICE,
                                                 HAL_BME_PATH,
                                                 HAL_BME_IF,
                                                 HAL_BME_CHANGE_PROPERTY ,
                                                 this,
                                                 SLOT(startChargingChannelCb(const QDBusMessage& )));
    }
    else if ( KBatteryLevelInp == aChannel )
    {
        QDBusConnection::systemBus().disconnect( HAL_BME_SERVICE,
                                                 HAL_BME_PATH,
                                                 HAL_BME_IF,
                                                 HAL_BME_CHANGE_PROPERTY,
                                                 this,
                                                 SLOT(startBatteryLevelChannelCb(const QDBusMessage& )));
    }
    else if ( KNetworkInp == aChannel )
    {
        QDBusConnection::systemBus().disconnect( PHONENET_SERVICE,
                                                 PHONENET_PATH,
                                                 PHONENET_IF,
                                                 PHONENET_REGISTR_STATUS_CHANGED,
                                                 this,
                                                 SLOT(startNetworkChannelCb(const QDBusMessage& )));

    }
    else if ( KSignalStrengthInp == aChannel )
    {
        QDBusConnection::systemBus().disconnect( PHONENET_SERVICE,
                                                 PHONENET_PATH,
                                                 PHONENET_IF,
                                                 PHONENET_CHANGE_SIGNAL_STRENGTH,
                                                 this,
                                                 SLOT(startSignalStrengthChannelCb(const QDBusMessage& )));
    }
    else if ( KBluetoothOnInp == aChannel )
    {
        QDBusInterface interface( BLUEZ_SERVICE,
                                  BLUEZ_PATH,
                                  BLUEZ_IF_MANAGER,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , KBluetoothOnInp + KErrDBusIf, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDBusMessage msg = interface.call( BLUEZ_GET_PATH );
        if ( msg.type() == QDBusMessage::ReplyMessage )
        {
            QString path = msg.arguments().at(0).value<QDBusObjectPath>().path();
            QDBusConnection::systemBus().disconnect( BLUEZ_SERVICE ,
                                                     path,
                                                     BLUEZ_IF_ADAPTER,
                                                     BLUEZ_CHANGE_PROPERTY,
                                                     this,
                                                     SLOT(startBluetoothOnChannelCb(const QDBusMessage& )));
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , KBluetoothOnInp + KErrDBusMsg, m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KFlipOpenInp == aChannel )
    {
        QDBusConnection::systemBus().disconnect( HAL_FLIP_SERVICE,
                                                 HAL_FLIP_PATH,
                                                 HAL_FLIP_IF,
                                                 HAL_FLIP_CHANGE_PROPERTY,
                                                 this,
                                                 SLOT(startFlipOpenChannelCb(const QDBusMessage& )));
    }

    m_triggerMap.remove( aChannel );
    m_startMap.remove( aChannel );
    return NULL;
}

//---------------------------------------
// Callback function for resizing of display size.
//---------------------------------------
void SysInfoService::displayResizedCb()
{
    desktop = QApplication::desktop();
    height = desktop->height();
    width = desktop->width();
}

//---------------------------------------
// Callback function for BatteryLevel channel.
//---------------------------------------
void SysInfoService::asyncBatteryLevelChannelCb(const QDBusMessage& message)
{
    if ( message.type() == QDBusMessage::ReplyMessage )
    {
        if ( TRUE == m_asyncMap.contains( KBatteryLevelInp ) )
        {
            qint32 trId = m_asyncMap.value(  KBatteryLevelInp  );
            OutputDataMap m_outputDataMap = getChannelSync( KBatteryLevelInp ).toMap();
            if ( FALSE == m_outputDataMap.contains(KErrCode) &&
                FALSE == m_outputDataMap.contains(KErrMessage) )
            {
                emit asyncCallback( 0, trId, m_outputDataMap );
            }
        }
    }
}

//---------------------------------------
// Callback function for Network channel.
//---------------------------------------
void SysInfoService::asyncNetworkChannelCb(const QDBusMessage& message)
{
    if ( message.type() == QDBusMessage::ReplyMessage )
    {
        if ( TRUE == m_asyncMap.contains( KNetworkInp ) )
        {
            qint32 trId = m_asyncMap.value(  KNetworkInp );
            OutputDataMap m_outputDataMap = getChannelSync( KNetworkInp ).toMap();
            if ( FALSE == m_outputDataMap.contains(KErrCode) &&
                FALSE == m_outputDataMap.contains(KErrMessage) )
            {
                emit asyncCallback( 0, trId, m_outputDataMap );
            }
        }
    }
}

//---------------------------------------
// Callback function for SignalStrength channel.
//---------------------------------------
void SysInfoService::asyncSignalStrengthChannelCb(const QDBusMessage& message)
{
    if ( message.type() == QDBusMessage::ReplyMessage )
    {
        if ( TRUE == m_asyncMap.contains( KSignalStrengthInp ) )
        {
            qint32 trId = m_asyncMap.value(  KSignalStrengthInp );
            OutputDataMap m_outputDataMap = getChannelSync( KSignalStrengthInp ).toMap();
            if ( FALSE == m_outputDataMap.contains(KErrCode) &&
                FALSE == m_outputDataMap.contains(KErrMessage) )
            {
                emit asyncCallback( 0, trId, m_outputDataMap );
            }
        }
    }
}

//---------------------------------------
// Callback function for HomeNetworkChannel channel.
//---------------------------------------
void SysInfoService::asyncHomeNetworkChannelCb(const QDBusMessage& message)
{
    if ( message.type() == QDBusMessage::ReplyMessage )
    {
        if ( TRUE == m_asyncMap.contains( KHomeNetworkInp ) )
        {
            OutputDataMap m_outputDataMap = getChannelSync( KNetworkInp ).toMap();
            m_outputDataMap.insert( KNetworkName, message.arguments().at(0).toString() );

            QDBusInterface interface( HN_SERVICE,
                                      HN_PATH,
                                      HN_IF,
                                      QDBusConnection::systemBus());
            if (FALSE == interface.isValid())
            {
                return;
            }
            QDBusReply<QByteArray> reply = interface.call(HN_GET_SERVICE_INFO);
            // The MNC and MCC are split into Hex numbers in the received byte array.
            // The MNC can be 2 or 3 digits long. If it is 2 digits long, it ends with 0xF.
            // The order of the Hex numbers in the reply is:
            // mcc2 mcc1 mnc3 mcc3 mnc2 mnc1
            QString mnc;
            QString mcc;
            if (TRUE == reply.isValid())
            {
                QString temp = reply.value().toHex();
                if(temp.count() == 6)
                {
                    //MNC
                    mnc.insert(0,temp.at(5));
                    mnc.insert(1,temp.at(4));
                    if(temp.at(2) != 'f')
                    {
                        mnc.insert(2,temp.at(2));
                    }
                    //MCC
                    mcc.insert(0,temp.at(1));
                    mcc.insert(1,temp.at(0));
                    mcc.insert(2,temp.at(3));
                }
            }
            m_outputDataMap.insert( KMobileCountryCode, mcc );
            m_outputDataMap.insert( KNetworkCode, mnc );
            m_outputDataMap.insert( KLocationStatus, (mnc.toInt() == 0 && mcc.toInt() == 0) ? false:true );

            qint32 trId = m_asyncMap.value( KHomeNetworkInp );
            emit asyncCallback( 0, trId, m_outputDataMap );
        }
    }
}

//---------------------------------------
// Callback function for ChargingChannel channel.
//---------------------------------------
void SysInfoService::startChargingChannelCb(const QDBusMessage & message)
{
    if ( TRUE == m_startMap.contains( KChargingInp ) )
    {
        qint32 trId = m_startMap.value( KChargingInp );
        QDBusArgument arg = message.arguments().at(1).value<QDBusArgument>();

        arg.beginArray();
        while ( !arg.atEnd() )
        {
            QStringList element;
            arg >> element;
            if ( HAL_BME_CHARGING == element.at(0) )
            {
                OutputDataMap m_outputDataMap = getChannelSync( KChargingInp ).toMap();
                if ( FALSE == m_outputDataMap.contains(KErrCode) &&
                    FALSE == m_outputDataMap.contains(KErrMessage) )
                {
                    emit asyncCallbackStChannel( 0, trId, m_outputDataMap );
                }
            }
        }
        arg.endArray();
    }
}

//---------------------------------------
// Callback function for BatteryLevel channel.
//---------------------------------------
void SysInfoService::startBatteryLevelChannelCb(const QDBusMessage & message)
{
    if ( TRUE == m_startMap.contains( KBatteryLevelInp ) )
    {
        qint32 trId = m_startMap.value( KBatteryLevelInp );
        QDBusArgument arg = message.arguments().at(1).value<QDBusArgument>();

        arg.beginArray();
        while ( !arg.atEnd() )
        {
            QStringList element;
            arg >> element;
            if ( HAL_BME_CHARGE_LEVEL == element.at(0) )
            {
                OutputDataMap m_outputDataMap = getChannelSync( KBatteryLevelInp ).toMap();
                if ( FALSE == m_outputDataMap.contains(KErrCode) &&
                    FALSE == m_outputDataMap.contains(KErrMessage) )
                {
                    if (FALSE == m_triggerMap.contains(KBatteryLevelInp) ||
                       (TRUE == m_triggerMap.contains(KBatteryLevelInp) &&
                        m_outputDataMap.value(KBatteryLevel).toInt() < m_triggerMap.value(KBatteryLevelInp)))
                    {
                        emit asyncCallbackStChannel( 0, trId, m_outputDataMap );
                    }
                }
            }
        }
        arg.endArray();
    }
}

//---------------------------------------
// Callback function for BluetoothOn channel.
//---------------------------------------
void SysInfoService::startBluetoothOnChannelCb(const QDBusMessage & message)
{
    if ( TRUE == m_startMap.contains( KBluetoothOnInp ) )
    {
        qint32 trId = m_startMap.value( KBluetoothOnInp );
        if ( BLUEZ_POWERED == message.arguments().at(0).toString() )
        {
            OutputDataMap m_outputDataMap = getChannelSync( KBluetoothOnInp ).toMap();
            if ( FALSE == m_outputDataMap.contains(KErrCode) &&
                FALSE == m_outputDataMap.contains(KErrMessage) )
            {
                emit asyncCallbackStChannel( 0, trId, m_outputDataMap );
            }
        }

    }
}

//---------------------------------------
// Callback function for Network channel.
//---------------------------------------
void SysInfoService::startNetworkChannelCb(const QDBusMessage& message)
{
    Q_UNUSED(message);
    if ( TRUE == m_startMap.contains( KNetworkInp ) )
    {
        qint32 trId = m_startMap.value( KNetworkInp );

        OutputDataMap m_outputDataMap = getChannelSync( KNetworkInp ).toMap();
        if ( FALSE == m_outputDataMap.contains(KErrCode) &&
            FALSE == m_outputDataMap.contains(KErrMessage) )
        {
            emit asyncCallbackStChannel( 0, trId, m_outputDataMap );
        }
    }
}

//---------------------------------------
// Callback function for SignalStrength channel.
//---------------------------------------
void SysInfoService::startSignalStrengthChannelCb( const QDBusMessage & message )
{
     Q_UNUSED(message);
     if ( TRUE == m_startMap.contains( KSignalStrengthInp ) )
     {
         qint32 trId = m_startMap.value( KSignalStrengthInp );

         OutputDataMap m_outputDataMap = getChannelSync( KSignalStrengthInp ).toMap();
         if ( FALSE == m_outputDataMap.contains(KErrCode) &&
             FALSE == m_outputDataMap.contains(KErrMessage) )
         {
             if (FALSE == m_triggerMap.contains(KSignalStrengthInp) ||
                (TRUE == m_triggerMap.contains(KSignalStrengthInp) &&
                 m_outputDataMap.value(KSignalStrength).toInt() < m_triggerMap.value(KSignalStrengthInp)))
             {
                emit asyncCallbackStChannel( 0, trId, m_outputDataMap );
             }
         }
     }
}

//---------------------------------------
// Callback function for FlipOpen channel.
//---------------------------------------
void SysInfoService::startFlipOpenChannelCb( const QDBusMessage & message )
{
    if ( TRUE == m_startMap.contains( KFlipOpenInp ) )
    {
        qint32 trId = m_startMap.value( KFlipOpenInp );
        QDBusArgument arg = message.arguments().at(1).value<QDBusArgument>();

        arg.beginArray();
        while ( !arg.atEnd() )
        {
            QStringList element;
            arg >> element;
            if ( HAL_FLIP_STATE  == element.at(0) )
            {
                QDBusInterface interface( HAL_FLIP_SERVICE,
                                          HAL_FLIP_PATH,
                                          HAL_FLIP_IF,
                                          QDBusConnection::systemBus());
                if ( FALSE == interface.isValid() )
                {
                    return;
                }
                QDBusMessage msg = interface.call( HAL_FLIP_GET_PROPERTY ,
                                                   HAL_FLIP_STATE );
                if ( msg.type() == QDBusMessage::ReplyMessage )
                {
                    bool state_flip = !(msg.arguments().at(0).toBool());
                    OutputDataMap m_outputDataMap;
                    m_outputDataMap.insert( KFlipOpen, state_flip );
                    emit asyncCallbackStChannel(0, trId, m_outputDataMap);
                }
            }
        }
        arg.endArray();
    }
}

//---------------------------------------
// Function contains all possible error checking (called at the beginning of each method)
// param aChannel - Name of the channel.
// param Data - contains a date-specific for each mode.
// param Mode - mode (which method calls this function).
// return Map Containing ErrorCode and ErrorMessage.
//---------------------------------------
ErrorMap SysInfoService::CheckingOfErrors( const QString& aChannel , const QVariant& Data , const QString& Mode )
{
    ErrorMap m_errorMap;
    if ( m_getChannelList == Mode ||
         m_setChannelSync == Mode ||
         m_getChannelSync == Mode ||
         m_getChannelASync == Mode ||
         m_cancel == Mode ||
         m_startChannel == Mode ||
         m_stopChannel == Mode )
    {
        if ( aChannel.isEmpty() )
        {
            qDebug()  << "error of arguments";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, MISSING_ARG_ERR );
            m_errorMap.insert( KErrMessage, KStrErrMissingArg );
            return m_errorMap;
        }
        if ( FALSE == m_ChannelList.contains( aChannel) )
        {
            qDebug()  << "channel not supported";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, INVALID_ARG_ERR );
            m_errorMap.insert( KErrMessage, KStrErrInvalidKey );
            return m_errorMap;
        }
    }

    if ( m_setChannelSync == Mode)
    {
        if ( aChannel == KDisplayWallpaperInp  &&
            Data.toMap().value(KWallpaperFileurl).type() != QVariant::String)
        {
            qDebug()  << "incorrect set value";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, INVALID_ARG_ERR );
            m_errorMap.insert( KErrMessage, KStrErrInvalidKey );
            return m_errorMap;
        }
        if ( aChannel == KClipboardInp &&
            Data.toMap().value(KClipboardString).type() != QVariant::String)
        {
            qDebug()  << "incorrect set value";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, INVALID_ARG_ERR );
            m_errorMap.insert( KErrMessage, KStrErrInvalidKey );
            return m_errorMap;
        }
    }

    if ( m_startChannel == Mode )
    {
        qint32 aTrigger = Data.toInt();
        if ( ( aTrigger != -1 ) && ( ( aTrigger < 0 ) || ( aTrigger >= 100 ) ) )
        {
            qDebug()  << "out of range";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, DATA_OUT_OF_RANGE_ERR );
            m_errorMap.insert( KErrMessage, KStrErrTrigger);
            return m_errorMap;
        }
    }

    if ( m_stopChannel == Mode )
    {
        if ( FALSE == m_startMap.contains( aChannel ) )
        {
            qDebug() << "data not found";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, DATA_NOT_FOUND_ERR );
            m_errorMap.insert( KErrMessage, KStrErrStop );
            return m_errorMap;
        }
    }

    if ( m_cancel == Mode )
    {
        if ( FALSE == m_asyncMap.contains( aChannel ) )
        {
            qDebug() << "data not found";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, DATA_NOT_FOUND_ERR );
            m_errorMap.insert( KErrMessage, KStrErrStop );
            return m_errorMap;
        }
    }

    if ( m_dbusErrors  == Mode )
    {
        qDebug() << "dbus errors" << Data.toString();
        m_errorMap.clear();
        m_errorMap.insert( KErrCode, SERVICE_NOT_AVAILABLE  );
        m_errorMap.insert( KErrMessage, "" );
        return m_errorMap;
    }

    if ( m_setWallpaper == Mode)
    {
        m_errorMap.clear();
        m_errorMap.insert( KErrCode, SERVICE_NOT_AVAILABLE  );
        m_errorMap.insert( KErrMessage, "" );
        return m_errorMap;
    }
    return m_errorMap;
}

