/*
* ============================================================================
*  Name        : qsensorservice.cpp
*  Part of     : serviceframework / WRT
*  Description : Qt class for reference service
*  Version     : %version: 5 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

//Project Includes
#include "qsensorservice.h"
#include "qsensordefines.h"

#include <QtDBus/QtDBus>
#include <QDateTime>
#include <qmath.h>

//---------------------------------------
// Default constructor
//---------------------------------------
QSensorService::QSensorService()
{
    //Append channels to the list
    m_channels.clear();
    m_channels.append(KAccelerometerAxis);
    m_channels.append(KOrientation);
    m_channels.append(KRotation);
    m_channels.append(KAccelerometerDoubleTapping);
    m_channels.append(KProximity);
    m_channels.append(KIllumination);

    m_startMap.clear();

    m_Frequencies.clear();
    m_Frequencies.insert(KAccelerometerAxis,MIN_FREQUENCY );                       //continue
    m_Frequencies.insert(KOrientation,MIN_FREQUENCY_NOT_CONTINUE );                //not continue
    m_Frequencies.insert(KRotation,MIN_FREQUENCY );                                //continue
    m_Frequencies.insert(KAccelerometerDoubleTapping,MIN_FREQUENCY_NOT_CONTINUE);  //not continue
    m_Frequencies.insert(KProximity,MIN_FREQUENCY_NOT_CONTINUE );                  //not continue
    m_Frequencies.insert(KIllumination,MIN_FREQUENCY_NOT_CONTINUE );               //not continue

    m_timerMap.clear();
    m_timerMap.insert(KAccelerometerAxis,new QTimer( this ) );
    m_timerMap.insert(KOrientation,new QTimer( this ) );
    m_timerMap.insert(KRotation,new QTimer( this ) );
    m_timerMap.insert(KAccelerometerDoubleTapping,new QTimer( this ) );
    m_timerMap.insert(KProximity,new QTimer( this ) );
    m_timerMap.insert(KIllumination,new QTimer( this ) );

    m_stateChannelsMap.clear();
    m_SingleStart.clear();

    interval_tapping = new QTimer( this );
    history_list.clear();

    scale_factor = SCF_ACCEL_SCALE_FACTOR;
}

//---------------------------------------
// Destructor
//---------------------------------------
QSensorService::~QSensorService()
{
    if (NULL != m_timerMap.value(KAccelerometerAxis) )
    {
        delete m_timerMap.value(KAccelerometerAxis);
    }
    if (NULL != m_timerMap.value(KOrientation))
    {
        delete m_timerMap.value(KOrientation);
    }
    if (NULL != m_timerMap.value(KRotation))
    {
        delete m_timerMap.value(KRotation);
    }
    if (NULL != m_timerMap.value(KAccelerometerDoubleTapping) )
    {
        delete m_timerMap.value(KAccelerometerDoubleTapping);
    }
    if (NULL != m_timerMap.value(KProximity))
    {
        delete m_timerMap.value(KProximity);
    }
    if (NULL != m_timerMap.value(KIllumination))
    {
        delete m_timerMap.value(KIllumination);
    }

    if (NULL != interval_tapping)
    {
        delete interval_tapping;
    }
}

//---------------------------------------
// Implemented from IServiceSecurity.Needed to handle capabilities
//---------------------------------------
void QSensorService::setSecuritySession(WRT::SecSession *aSecSession)
{
    m_secSession = aSecSession;
}

//---------------------------------------
// This method is used to get the list of channels supported by the device.
// This is a Synchronous API
//---------------------------------------
QVariant QSensorService::getChannels()
{
    return m_channels;
}

//---------------------------------------
// Get the properties of the specified channel.
// This is an synchronous method.
//---------------------------------------
QVariant QSensorService::getChannelEvent(const QString& aChannel)
{
    ErrorMap error = CheckingOfErrors( aChannel ,NULL, m_getChannelEvent);
    if ( FALSE == error.empty() )
    {
        return error;
    }

    if ( KAccelerometerAxis == aChannel)
    {
        QDBusInterface interface( MCE_SERVICE,
                                  MCE_REQUEST_PATH,
                                  MCE_REQUEST_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , aChannel + KErrDBusIF , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDateTime datetime = QDateTime::currentDateTime();
        QString time = datetime.time().toString("hh:mm:ss ap");
        QString date = datetime.date().toString("dd/MM/yyyy");

        QDBusMessage msg = interface.call( MCE_GET_DEVICE_ORIENTATION );
        if ( msg.type() == QDBusMessage::ReplyMessage )
        {
            OutputDataMap m_outputDataMap;
            QVector3D axis;
            axis.setX( msg.arguments().at(3).toDouble() );
            axis.setY( msg.arguments().at(4).toDouble() );
            axis.setZ( msg.arguments().at(5).toDouble() );

            m_outputDataMap.insert(KChannelName, KAccelerometerAxis);
            m_outputDataMap.insert(KtimeStamp, date + " " + time);
            m_outputDataMap.insert(KaxisX,scale_factor * axis.x());
            m_outputDataMap.insert(KaxisY,scale_factor * axis.y());
            m_outputDataMap.insert(KaxisZ,scale_factor * axis.z());
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , aChannel + KErrDBusMSG , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KOrientation == aChannel)
    {
        QDBusInterface interface( MCE_SERVICE,
                                  MCE_REQUEST_PATH,
                                  MCE_REQUEST_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , aChannel + KErrDBusIF , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDateTime datetime = QDateTime::currentDateTime();
        QString time = datetime.time().toString("hh:mm:ss ap");
        QString date = datetime.date().toString("dd/MM/yyyy");

        QDBusMessage msg = interface.call( MCE_GET_DEVICE_ORIENTATION );
        if ( msg.type() == QDBusMessage::ReplyMessage )
        {
            OutputDataMap m_outputDataMap;
            QString orientation = getOrientation( msg.arguments().at(0).toString(),
                                                  msg.arguments().at(1).toString(),
                                                  msg.arguments().at(2).toString() );

            m_outputDataMap.insert(KChannelName, KOrientation);
            m_outputDataMap.insert(KtimeStamp, date + " " + time);
            m_outputDataMap.insert(Kdirection,orientation);
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , aChannel + KErrDBusMSG , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KRotation == aChannel)
    {
        QDBusInterface interface( MCE_SERVICE,
                                  MCE_REQUEST_PATH,
                                  MCE_REQUEST_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , aChannel + KErrDBusIF , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDateTime datetime = QDateTime::currentDateTime();
        QString time = datetime.time().toString("hh:mm:ss ap");
        QString date = datetime.date().toString("dd/MM/yyyy");

        QDBusMessage msg = interface.call( MCE_GET_DEVICE_ORIENTATION );
        if ( msg.type() == QDBusMessage::ReplyMessage )
        {
            OutputDataMap m_outputDataMap;
            QVector3D axis;
            axis.setX( msg.arguments().at(3).toDouble() );
            axis.setY( msg.arguments().at(4).toDouble() );
            axis.setZ( msg.arguments().at(5).toDouble() );
            QVector3D rotation = getRotationCoord( axis );

            m_outputDataMap.insert(KChannelName, KRotation);
            m_outputDataMap.insert(KtimeStamp, date + " " + time);
            m_outputDataMap.insert(KrotationAboutXAxis,rotation.x());
            m_outputDataMap.insert(KrotationAboutYAxis,rotation.y());
            m_outputDataMap.insert(KrotationAboutZAxis,rotation.z());
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , aChannel + KErrDBusMSG , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KAccelerometerDoubleTapping == aChannel )
    {
        return NULL; //not supported in this function
    }
    else if ( KProximity == aChannel )
    {
        QDBusInterface interface( HAL_PROXIMITY_SERVICE,
                                  HAL_PROXIMITY_PATH,
                                  HAL_PROXIMITY_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , aChannel + KErrDBusIF , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDateTime datetime = QDateTime::currentDateTime();
        QString time = datetime.time().toString("hh:mm:ss ap");
        QString date = datetime.date().toString("dd/MM/yyyy");

        QDBusMessage msg = interface.call( HAL_PROXIMITY_GET_PROPERTY, HAL_PROXIMITY_STATE );
        if ( msg.type() == QDBusMessage::ReplyMessage )
        {
            OutputDataMap m_outputDataMap;
            qint32 proximity = (int)!(msg.arguments().at(0).toBool());
            m_outputDataMap.insert(KChannelName, KProximity);
            m_outputDataMap.insert(KtimeStamp, date + " " + time);
            m_outputDataMap.insert(KproximityState,proximity);
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel ,aChannel + KErrDBusMSG , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else if ( KIllumination == aChannel )
    {
        QDBusInterface interface( HAL_LIGHTSENSOR_SERVICE,
                                  HAL_LIGHTSENSOR_PATH,
                                  HAL_LIGHTSENSOR_IF,
                                  QDBusConnection::systemBus());
        if ( FALSE == interface.isValid() )
        {
            ErrorMap error = CheckingOfErrors( aChannel , aChannel + KErrDBusIF , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
        QDateTime datetime = QDateTime::currentDateTime();
        QString time = datetime.time().toString("hh:mm:ss ap");
        QString date = datetime.date().toString("dd/MM/yyyy");

        QDBusMessage msg = interface.call( HAL_GET_BRIGHTNESS );
        if ( msg.type() == QDBusMessage::ReplyMessage )
        {

            OutputDataMap m_outputDataMap;
            qint32 illumination = getIllumination( msg.arguments().at(0).toInt() );
            m_outputDataMap.insert(KChannelName, KIllumination);
            m_outputDataMap.insert(KtimeStamp, date + " " + time);
            m_outputDataMap.insert(KBrightnessLevel,illumination);
            return m_outputDataMap;
        }
        else
        {
            ErrorMap error = CheckingOfErrors( aChannel , aChannel + KErrDBusMSG , m_dbusErrors);
            if ( FALSE == error.empty() )
            {
                return error;
            }
        }
    }
    else
    {
        return NULL;
    }
}

//---------------------------------------
// Get the properties of the specified channel.
// This is an synchronous method.
//---------------------------------------
QVariant QSensorService::getChannelData( const QString& aChannel )
{
    OutputDataMap m_outputDataMap;
    OutputDataMap m_dataMap = getChannelEvent(aChannel).toMap();
    if( m_dataMap.contains(KErrCode) )
    {
        m_outputDataMap =  m_dataMap;
    }
    else
    {
        m_outputDataMap.insert( KErrCode, 0 );
        m_outputDataMap.insert( KChannelData, m_dataMap );
    }
    return m_outputDataMap;
}

//---------------------------------------
// Registers for notifications from a particular channel.
// This is an asynchronous method.
// If the notification for a particular channel has already been registered
// api will return an error.
//---------------------------------------
QVariant QSensorService::startChannel( const QString& aChannel,const int transId )
{
    ErrorMap error = CheckingOfErrors( aChannel ,QVariant(transId), m_startChannel);
    if ( FALSE == error.empty() )
    {
        return error;
    }

    m_startMap.insert(aChannel, transId);
    if ( KAccelerometerDoubleTapping != aChannel )
    {
        m_stateChannelsMap.insert(aChannel,getChannelEvent(aChannel));
        m_SingleStart.insert(0,aChannel);
        m_SingleStart.insert(1,transId);
        QTimer::singleShot( SINGLE_START, this, SLOT( singleStartCb() ) );
    }

    if ( KAccelerometerAxis == aChannel)
    {
        //continue
        connect( m_timerMap.value(aChannel), SIGNAL(timeout()), this, SLOT(startAccelerometerChannelCb()) );
        m_timerMap.value(aChannel)->setSingleShot( FALSE );
        m_timerMap.value(aChannel)->start( m_Frequencies.value(aChannel) );
    }
    else if ( KOrientation == aChannel)
    {
        //not continue
        connect( m_timerMap.value(aChannel), SIGNAL(timeout()), this, SLOT(startOrientationChannelCb()) );
        m_timerMap.value(aChannel)->setSingleShot( FALSE );
        m_timerMap.value(aChannel)->start( m_Frequencies.value(aChannel) );
    }
    else if ( KRotation == aChannel)
    {
        //continue
        connect( m_timerMap.value(aChannel), SIGNAL(timeout()), this, SLOT(startRotationChannelCb()) );
        m_timerMap.value(aChannel)->setSingleShot( FALSE );
        m_timerMap.value(aChannel)->start( m_Frequencies.value(aChannel) );
    }
    else if ( KAccelerometerDoubleTapping == aChannel )
    {
        //not continue
        history_list.clear();
        connect( m_timerMap.value(aChannel), SIGNAL(timeout()), this, SLOT(startAccelerometerDoubleTappingChannelCb()) );
        m_timerMap.value(aChannel)->setSingleShot( FALSE );
        m_timerMap.value(aChannel)->start( m_Frequencies.value(aChannel) );
    }
    else if ( KProximity == aChannel )
    {
        //not continue
        connect( m_timerMap.value(aChannel), SIGNAL(timeout()), this, SLOT(startProximityChannelCb()) );
        m_timerMap.value(aChannel)->setSingleShot( FALSE );
        m_timerMap.value(aChannel)->start( m_Frequencies.value(aChannel) );
    }
    else if ( KIllumination == aChannel )
    {
        //not continue
        connect( m_timerMap.value(aChannel), SIGNAL(timeout()), this, SLOT(startIlluminationChannelCb()) );
        m_timerMap.value(aChannel)->setSingleShot( FALSE );
        m_timerMap.value(aChannel)->start( m_Frequencies.value(aChannel) );
    }

    error.insert(KErrCode, SUCCESS);
    return error;
}

//---------------------------------------
// Stops the notifications (started by startChannel) for the channel specified.
// This is a synchronous method.
//---------------------------------------
QVariant QSensorService::stopChannel( const QString& aChannel )
{
    ErrorMap error = CheckingOfErrors( aChannel ,NULL, m_stopChannel);
    if ( FALSE == error.empty() )
    {
        return error;
    }

    m_stateChannelsMap.remove(aChannel);
    m_startMap.remove(aChannel);
    m_timerMap.value(aChannel)->stop();

    error.insert(KErrCode,SUCCESS);
    return error;
}

//---------------------------------------
// Set desired frequency (only for AccelerometerAxis and Rotation channels).
// This is a synchronous method.
//---------------------------------------
QVariant QSensorService::setDesiredUpdateFrequency(const QString& aChannel, const int msec)
{
    ErrorMap error = CheckingOfErrors( aChannel ,NULL, m_setDesiredUpdateFrequency);
    if ( FALSE == error.empty() )
    {
        return error;
    }

    if ( msec > MIN_FREQUENCY )
    {
        m_Frequencies.insert(aChannel, msec);

        m_timerMap.value(aChannel)->stop();
        m_timerMap.value(aChannel)->setSingleShot( FALSE );
        m_timerMap.value(aChannel)->start( m_Frequencies.value(aChannel) );
        return msec;
    }
    else
    {
        return m_Frequencies.value(aChannel);
    }
}

//---------------------------------------
// Get scale factor (only for AccelerometerAxis channel).
// This is a synchronous method.
//---------------------------------------
QVariant QSensorService::getScaleFactor(const QString& aChannel)
{
    ErrorMap error = CheckingOfErrors( aChannel ,NULL, m_getScaleFactor);
    if ( FALSE == error.empty() )
    {
        return error;
    }

    return scale_factor;
}

//---------------------------------------
// Callback function for AccelerometerAxis channel.
//---------------------------------------
void QSensorService::startAccelerometerChannelCb()
{
    if ( TRUE == m_startMap.contains( KAccelerometerAxis ) )
    {
        OutputDataMap m_outputDataMap = getChannelEvent( KAccelerometerAxis ).toMap();
        emit asyncCallback( m_startMap.value(KAccelerometerAxis), 0, m_outputDataMap);
    }
}

//---------------------------------------
// Callback function for Orientation channel.
//---------------------------------------
void QSensorService::startOrientationChannelCb()
{
    if ( TRUE == m_startMap.contains( KOrientation ) )
    {
        OutputDataMap m_outputDataMap = getChannelEvent( KOrientation ).toMap();
        if ( m_stateChannelsMap.value(KOrientation).toMap().value(Kdirection) != m_outputDataMap.value(Kdirection) )
        {
            m_stateChannelsMap.insert(KOrientation,m_outputDataMap);
            emit asyncCallback( m_startMap.value(KOrientation), 0, m_outputDataMap);
        }
    }
}

//---------------------------------------
// Callback function for Rotation channel.
//---------------------------------------
void QSensorService::startRotationChannelCb()
{
    if ( TRUE == m_startMap.contains( KRotation ) )
    {
        OutputDataMap m_outputDataMap = getChannelEvent( KRotation ).toMap();
        emit asyncCallback( m_startMap.value(KRotation), 0, m_outputDataMap );
    }
}

//---------------------------------------
// Callback function for AccelerometerDoubleTapping channel.
//---------------------------------------
void QSensorService::startAccelerometerDoubleTappingChannelCb()
{
    if ( TRUE == m_startMap.contains( KAccelerometerDoubleTapping ) )
    {
        QVector3D axis;
        OutputDataMap axisMap = getChannelEvent( KAccelerometerAxis ).toMap();
        axis.setX( axisMap.value(KaxisX).toInt());
        axis.setY( axisMap.value(KaxisY).toInt());
        axis.setZ( axisMap.value(KaxisZ).toInt());
        qint32 direction = getDbTapping( axis );

        if ( Unknown != direction )
        {
            QDateTime datetime = QDateTime::currentDateTime();
            QString time = datetime.time().toString("hh:mm:ss ap");
            QString date = datetime.date().toString("dd/MM/yyyy");
            OutputDataMap m_outputDataMap;

            m_outputDataMap.insert(KChannelName, KAccelerometerDoubleTapping);
            m_outputDataMap.insert(KtimeStamp, date + " " + time);
            m_outputDataMap.insert(Kdirection, direction);
            emit asyncCallback( m_startMap.value(KAccelerometerDoubleTapping), 0, m_outputDataMap);
        }
    }
}

//---------------------------------------
// Callback function for Proximity channel.
//---------------------------------------
void QSensorService::startProximityChannelCb()
{
    if ( TRUE == m_startMap.contains( KProximity ) )
    {
        OutputDataMap m_outputDataMap = getChannelEvent( KProximity ).toMap();
        if ( m_stateChannelsMap.value(KProximity).toMap().value(KproximityState) != m_outputDataMap.value(KproximityState) )
        {
            m_stateChannelsMap.insert(KProximity,m_outputDataMap);
            emit asyncCallback( m_startMap.value(KOrientation), 0, m_outputDataMap);
        }
    }
}

//---------------------------------------
// Callback function for Illumination channel.
//---------------------------------------
void QSensorService::startIlluminationChannelCb()
{
    if ( TRUE == m_startMap.contains( KIllumination ) )
    {
        OutputDataMap m_outputDataMap = getChannelEvent( KIllumination ).toMap();
        if ( m_stateChannelsMap.value(KIllumination).toMap().value(KBrightnessLevel) != m_outputDataMap.value(KBrightnessLevel) )
        {
            m_stateChannelsMap.insert(KIllumination,m_outputDataMap);
            emit asyncCallback( m_startMap.value(KOrientation), 0, m_outputDataMap);
        }
    }
}

//---------------------------------------
// Callback function for Single Start.
//---------------------------------------
void QSensorService::singleStartCb()
{
    if ( FALSE == m_SingleStart.empty() )
    {
        OutputDataMap m_outputDataMap  = getChannelEvent(m_SingleStart.value(0).toString()).toMap();
        emit asyncCallback( m_SingleStart.value(1).toInt(), 0, m_outputDataMap  );
        m_SingleStart.clear();
    }
}

//---------------------------------------
// Function contains all possible error checking (called at the beginning of each method)
// param aChannel - Name of the channel.
// param Data - contains a date-specific for each mode.
// param Mode - mode (which method calls this function).
// return Map Containing ErrorCode and ErrorMessage.
//---------------------------------------
ErrorMap QSensorService::CheckingOfErrors( const QString& aChannel , const QVariant& Data, const QString& Mode )
{
    ErrorMap m_errorMap;
    if ( m_getChannelEvent           == Mode ||
         m_startChannel              == Mode ||
         m_stopChannel               == Mode ||
         m_setDesiredUpdateFrequency == Mode ||
         m_getScaleFactor            == Mode )
    {
        if ( aChannel.isEmpty() )
        {
            qDebug()  << "error of arguments";
            m_errorMap.insert( KErrCode, MISSING_ARG_ERR );
            m_errorMap.insert( KErrMessage, KStrErrMissingArg );
            return m_errorMap;
        }
        if ( FALSE == m_channels.contains( aChannel) )
        {
            qDebug()  << "channel not supported";
            m_errorMap.insert( KErrCode, NOT_SUPPORTED_ERR );
            m_errorMap.insert( KErrMessage, KStrErrInvalidKey );
            return m_errorMap;
        }
    }

    if ( m_startChannel == Mode )
    {
         if ( TRUE ==  m_startMap.contains(aChannel) )
         {
            qDebug() << "service in use";
            m_errorMap.insert(KErrCode,SERVICE_IN_USE_ERR);
            m_errorMap.insert( KErrMessage, "" );
            return m_errorMap;
         }

         if ( "not_found" != m_startMap.key( Data.toInt() ,"not_found" ) )
         {
            qDebug() << "invalid trans id(already exist)";
            m_errorMap.insert( KErrCode,INVALID_ARG_ERR );
            m_errorMap.insert( KErrMessage, KStrErrInvalidTransID );
            return m_errorMap;
         }
    }

    if ( m_stopChannel == Mode )
    {
         if ( FALSE ==  m_startMap.contains(aChannel) )
         {
            qDebug() << "data not found";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, DATA_NOT_FOUND_ERR );
            m_errorMap.insert( KErrMessage, KStrErrStop );
            return m_errorMap;
         }
    }

    if ( m_setDesiredUpdateFrequency == Mode )
    {
        if ( FALSE == m_startMap.contains(aChannel) )
        {
            qDebug() << "data not found";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, DATA_NOT_FOUND_ERR );
            m_errorMap.insert( KErrMessage, KStrErrStop );
            return m_errorMap;
        }
        if ( KAccelerometerAxis != aChannel &&
            KRotation != aChannel  )
        {
            m_errorMap.insert( KErrCode, NOT_SUPPORTED_ERR );
            m_errorMap.insert( KErrMessage, "" );
            return m_errorMap;
        }
    }

    if ( m_getScaleFactor == Mode )
    {
        if ( FALSE == m_startMap.contains(aChannel) )
        {
            qDebug() << "data not found";
            m_errorMap.clear();
            m_errorMap.insert( KErrCode, DATA_NOT_FOUND_ERR );
            m_errorMap.insert( KErrMessage, KStrErrStop );
            return m_errorMap;
        }

        if ( KAccelerometerAxis != aChannel )
        {
            m_errorMap.insert( KErrCode, NOT_SUPPORTED_ERR );
            m_errorMap.insert( KErrMessage, "" );
            return m_errorMap;
        }
    }

    if ( m_dbusErrors  == Mode )
    {
        qDebug() << "dbus errors" << Data.toString();
        m_errorMap.clear();
        m_errorMap.insert( KErrCode, SERVICE_NOT_AVAILABLE  );
        m_errorMap.insert( KErrMessage, "" );
        return m_errorMap;
    }
    return m_errorMap;
}

 /**
  * returns the correct orientation of analyzing the data from the mce service
  * @since x.y
  *
  * @param orientation - portrait or landscape (including inverted)
  * @param stand - on_stand or off_stand.
  * @param facing - face_up or face_down.
  * @return String correct orientation
  */
QString QSensorService::getOrientation( const QString& orientation, const QString& stand, const QString& facing )
{
    if ( MCE_ORIENTATION_PORTRAIT == orientation )
    {
        return KdisplayUp;
    }
    if ( MCE_ORIENTATION_PORTRAIT_INVERTED == orientation )
    {
        return KdisplayDown;
    }
    if ( MCE_ORIENTATION_LANDSCAPE == orientation )
    {
        return KdisplayLeftUp;
    }
    if ( MCE_ORIENTATION_LANDSCAPE_INVERTED == orientation )
    {
        return KdisplayRightUp;
    }
    if ( MCE_ORIENTATION_UNKNOWN == orientation &&
        MCE_ORIENTATION_OFF_STAND == stand &&
        MCE_ORIENTATION_FACE_UP == facing)
    {
        return KdisplayUpwards;
    }
    if ( MCE_ORIENTATION_UNKNOWN == orientation &&
        MCE_ORIENTATION_OFF_STAND == stand &&
        MCE_ORIENTATION_FACE_DOWN == facing)
    {
        return KdisplayDownwards;
    }

    return KdisplayUnknown;
}

 /**
  * returns the correct rotation of analyzing accelerometer from the mce service
  * @since x.y
  *
  * @param axis - accelerometer data
  * @return vector with rotation coord (in degrees)
  */
QVector3D QSensorService::getRotationCoord( const QVector3D& axis )
{
    QVector3D rotation;
    qreal lenght = axis.length();
    qreal alpha = 180.0 / M_PI;

    rotation.setX( qCeil(alpha*qAcos(axis.x() / lenght)) );
    rotation.setY( qCeil(alpha*qAcos(axis.y() / lenght)) );
    rotation.setZ( qCeil(alpha*qAcos(axis.z() / lenght)) );

    return rotation;
}

/**
  * returns the correct illumination of analyzing light sensor from hal service
  * @since x.y
  *
  * @param brightness - data from hal
  * @return qint32 correct illumination (in percentage)
  */
qint32 QSensorService::getIllumination( qint32 brightness)
{
    if (brightness >= 500)
    {
        return 100;
    }
    else
    {
        return qCeil(qreal(brightness) / 5.0);
    }
}

/**
  * returns the db tapping state of analyzing accelerometer using timer
  * @since x.y
  *
  * @param axis - accelerometer data
  * @return qint32 - state of db tapping
  */
qint32 QSensorService::getDbTapping( const QVector3D& axis )
{
    if ( TRUE == history_list.isEmpty() )
    {
        interval_tapping->stop();
        interval_tapping->setSingleShot( TRUE );
        interval_tapping->start( DB_TAPPING_FREQ );
    }

    if ( TRUE == interval_tapping->isActive() )
    {

        history_list.append( axis );
        return Unknown;
    }
    else
    {
        bool maxX = false;
        bool minX = false;
        bool maxY = false;
        bool minY = false;
        bool maxZ = false;
        bool minZ = false;

        for (int i = 0; i < history_list.size(); ++i)
        {
            if ( history_list.at(i).x() > DB_TAPPING_MAX_COORD )
            {
                maxX = true;
            }
            if ( history_list.at(i).x() < DB_TAPPING_MIN_COORD )
            {
                minX = true;
            }

            if ( history_list.at(i).y() > DB_TAPPING_MAX_COORD )
            {
                maxY = true;
            }
            if ( history_list.at(i).y() < DB_TAPPING_MIN_COORD )
            {
                minY = true;
            }

            if ( history_list.at(i).z() > DB_TAPPING_MAX_COORD )
            {
                maxZ = true;
            }
            if ( history_list.at(i).z() < DB_TAPPING_MIN_COORD )
            {
                minZ = true;
            }
        }

        history_list.clear();
        if (minX && maxX && minZ && maxZ)
        {
           return YPlus|YMinus;
        }

        if (minZ && maxZ && minY && maxY )
        {
           return XPlus|XMinus;
        }

        if (minX && maxX && minY && maxY )
        {
           return ZPlus|ZMinus;
        }
        return Unknown;
    }
}


