/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include "qmessagingconstants.h"
#include "messagefilter.h"


bool MessageFilter::fillFilterData(const QVariantMap& aMatch, const int aSortKey,
            const int aSortOrder, int& aErrorCode, QString& aErrorMessage)
{
    if (aSortKey != SORT_BY_DATE && aSortKey != SORT_BY_SENDER)
    {
        aErrorCode = DATA_OUT_OF_RANGE_ERR;
        aErrorMessage = KCBadArgument;
        return false;
    }
    iSortKey = aSortKey;

    if (aSortOrder != SORT_ASCENDING && aSortOrder != SORT_DESCENDING)
    {
        aErrorCode = DATA_OUT_OF_RANGE_ERR;
        aErrorMessage = KCBadArgument;
        return false;
    }
    iSortMethod = aSortOrder;

    iNeededFilterFlags = EAll;
    iMandatoryFlags = 0;
    iMultiTypes = true;
    return doFillFilterData(aMatch, aErrorCode, aErrorMessage);
}

bool MessageFilter::doFillFilterData(const QVariantMap& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    qDebug() << __FUNCTION__ << aMatch;
    if ((iNeededFilterFlags & EFilterType))
    {
        if (iMultiTypes && !parseFilterTypes(aMatch, aErrorCode, aErrorMessage))
        {
            return false;
        }

        if (!iMultiTypes && !parseFilterType(aMatch, aErrorCode, aErrorMessage))
        {
            return false;
        }
    }

    if ((iNeededFilterFlags & EFilterSenders) && !parseFilterSenders(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterTo) && !parseFilterTo(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterCc) && !parseFilterCc(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterSubject) && !parseFilterSubject(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterBody) && !parseFilterBody(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterUnread) && !parseFilterUnread(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterFolderName) && !parseFilterFolderName(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterPriority) && !parseFilterPriority(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterValidity) && !parseFilterValidity(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterAttachment) && !parseFilterAttachments(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterStartDate) && !parseFilterStartDate(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterEndDate) && !parseFilterEndDate(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if ((iNeededFilterFlags & EFilterId) && !parseFilterId(aMatch, aErrorCode, aErrorMessage))
    {
        return false;
    }

    return true;
}

bool MessageFilter::fillPartialData(const QVariantMap& aMatch, int aNeededFilterFlags, int aMandatoryFlags, int& aErrorCode, QString& aErrorMessage)
{
    iNeededFilterFlags = aNeededFilterFlags;
    iMandatoryFlags = aMandatoryFlags;
    iMultiTypes = false;
    return doFillFilterData(aMatch, aErrorCode, aErrorMessage);
}

bool MessageFilter::checkFilter(const QVariantMap& aEvent) const
{
    bool res = true;
    QVariant value = aEvent.value(KCMessage);
    QVariantMap msgData;

    if (value.type() == QVariant::Map)
    {
        msgData = value.toMap();
    }
    else
    {
        res = false;
    }

    if (res && (iFilterFlags & EFilterId))
    {
        value = aEvent.value(KCID);
        if (value.type() != QVariant::String ||
            value.toString().compare(iFilterId, Qt::CaseInsensitive) != 0)
        {
            qDebug() << "Failed check id" << iFilterId << value;
            res = false;
        }
    }

    if (res && (iFilterFlags & EFilterType))
    {
        value = msgData.value(KCType);
        if (value.type() != QVariant::String ||
            ! (iFilterTypes.contains(value.toString(), Qt::CaseInsensitive) || iFilterTypes.contains("")))
        {
            qDebug() << "Failed check type" << iFilterTypes << value;
            res = false;
        }
    }

    res = res & checkStringList(aEvent, KCSender, EFilterSenders, iFilterSenders);

    res = res & checkStringListMulti(msgData, KCTo, EFilterTo, iFilterTo);
    res = res & checkStringListMulti(msgData, KCCc, EFilterCc, iFilterCc);

    res = res & checkStringWildCard(msgData, KCSubject, EFilterSubject, iFilterSubject);
    res = res & checkStringWildCard(msgData, KCBody, EFilterBody, iFilterBody);

    if (res && (iFilterFlags & EFilterStartDate))
    {
        value = aEvent.value(KCTIME);
        if (value.type() != QVariant::DateTime ||
            value.toDateTime() < iFilterStartTime)
        {
            res = false;
            qDebug() << "Failed check start date" << iFilterStartTime << value;
        }
    }

    if (res && (iFilterFlags & EFilterEndDate))
    {
        value = aEvent.value(KCTIME);
        if (value.type() != QVariant::DateTime ||
            value.toDateTime() > iFilterEndTime)
        {
            res = false;
            qDebug() << "Failed check end date" << iFilterEndTime << value;
        }
    }

    if (res && (iFilterFlags & EFilterFolderName))
    {
        value = aEvent.value(KCFolder);
        if (value.type() != QVariant::String ||
            iFilterFolderName.compare(value.toString(), Qt::CaseInsensitive) != 0)
        {
            res = false;
            qDebug() << "Failed check folder name" << iFilterFolderName << value;
        }
    }

    if (res && (iFilterFlags & EFilterPriority))
    {
        //TODO: check priority here. Currently check only for normal priority for SMS/Email
        if (iFilterPriority != PRIORITY_NORMAL)
        {
            res = false;
            qDebug() << "Failed check priority" << iFilterPriority;
        }
    }

    if (res && (iFilterFlags & EFilterValidity))
    {
        value = msgData.value(KCValidity);
        if (value.type() != QVariant::Int||
            iFilterValidity != value.toInt())
        {
            res = false;
            qDebug() << "Failed check folder name" << iFilterFolderName << value;
        }
    }

    if (res && (iFilterFlags & EFilterUnread))
    {
        value = aEvent.value(KCUNREAD);
        if (value.type() != QVariant::Bool||
            iFilterUnread != value.toBool())
        {
            res = false;
            qDebug() << "Failed check 'unread status'" << iFilterUnread << value;
        }
    }


    if (res && (iFilterFlags & EFilterAttachment))
    {
        value = msgData.value(KCAttachment);
        if (value.type() == QVariant::List)
        {
            QVariantList attList = value.toList();
            foreach (FilterAttachmentData filterData, iFilterAttachments)
            {
                QRegExp regExp (filterData.iFileName, Qt::CaseInsensitive, QRegExp::Wildcard);
                bool found = false;
                foreach (QVariant attach, attList)
                {
                    if (attach.type() == QVariant::Map)
                    {
                        QVariant attachValue = attach.toMap().value(KCUri);
                        if (attachValue.type() == QVariant::String)
                        {
                            QString attachUri = attachValue.toString();
                            if (attachUri.contains(regExp))
                            {
                                found = true;
                                break;
                            }
                        }
                        else
                        {
                            break;
                        }
                    }
                    else
                    {
                        break;
                    }
                }

                if (!found)
                {
                    qDebug() << "Failed check attachments" << filterData.iFileName << attList;
                    res = false;
                    break;
                }
            }
        }
        else
        {
            res = false;
        }
    }
    qDebug() << "Check filter:" << res;
    return res;
}

bool MessageFilter::checkStringList(const QVariantMap& aEvent,
                                    const QString& aParam,
                                    const FilterFlags& aFlag,
                                    const QStringList& aFilter) const
{
    bool res = true;
    QVariant value;
    if (iFilterFlags & aFlag)
    {
        bool found = false;
        value = aEvent.value(aParam);
        if (value.type() == QVariant::String)
        {
            QString attendee = value.toString();
            foreach (const QString filterMember, aFilter)
            {
                //qDebug() << "Checking senders. RegExp=" << filterSender;
                QRegExp regExp (filterMember, Qt::CaseInsensitive, QRegExp::Wildcard);
                if (attendee.contains(regExp))
                {
                    found = true;
                    break;
                }
            }
            if (!found)
            {
                qDebug() << "Failed check attendees: " << aParam << aFilter << attendee;
                res = false;
            }
        }
        else
        {
            res = false;
        }
    }
    return res;
}

bool MessageFilter::checkStringListMulti(const QVariantMap& aEvent, const QString& aParam, const FilterFlags& aFlag, const QStringList& aFilter) const
{
    bool res = true;
    QVariant value;
    if (iFilterFlags & aFlag)
    {
        bool found = true;
        value = aEvent.value(aParam);
        if (value.type() == QVariant::StringList)
        {
            QStringList valueList = value.toStringList();

            foreach (const QString filterMember, aFilter)
            {
                //qDebug() << "Checking senders. RegExp=" << filterSender;
                QRegExp regExp (filterMember, Qt::CaseInsensitive, QRegExp::Wildcard);
                if (valueList.indexOf(regExp) < 0) //no one from recipients matches this wildcard
                {
                    found = false;
                    break;
                }
            }
            if (!found)
            {
                qDebug() << "Failed check attendees: " << aParam << aFilter << valueList;
                res = false;
            }
        }
        else
        {
            res = false;
        }
    }
    return res;
}


bool MessageFilter::checkStringWildCard(const QVariantMap& aEvent,
                                        const QString& aParam,
                                        const FilterFlags& aFlag,
                                        const QString& aFilter) const
{
    bool res = true;
    QVariant value;
    if (res && (iFilterFlags & aFlag))
    {
        value = aEvent.value(aParam);
        QRegExp regExp (aFilter, Qt::CaseInsensitive, QRegExp::Wildcard);
        if (value.type() != QVariant::String ||
            !value.toString().contains(regExp))
        {
            res = false;
            qDebug() << "Failed check" << aParam << aFilter << value;
        }
    }
    return res;
}


bool MessageFilter::allowInbox() const
{
    bool res = true;
    if (iFilterFlags & EFilterFolderName)
    {
        if (iFilterFolderName.compare(KCINBOX, Qt::CaseInsensitive) != 0 &&
            iFilterFolderName.compare(KCALL, Qt::CaseInsensitive) != 0)
        {
            res = false;
        }
    }
    return res;
}

bool MessageFilter::allowOutbox() const
{
    bool res = true;
    if (iFilterFlags & EFilterFolderName)
    {
        if (iFilterFolderName.compare(KCOUTBOX, Qt::CaseInsensitive) != 0 &&
            iFilterFolderName.compare(KCALL, Qt::CaseInsensitive) != 0)
        {
            res = false;
        }
    }
    return res;
}

bool MessageFilter::allowSent() const
{
    bool res = true;
    if (iFilterFlags & EFilterFolderName)
    {
        if (iFilterFolderName.compare(KCSENTBOX, Qt::CaseInsensitive) != 0 &&
            iFilterFolderName.compare(KCALL, Qt::CaseInsensitive) != 0)
        {
            res = false;
        }
    }
    return res;
}

bool MessageFilter::allowDrafts() const
{
    bool res = true;
    if (iFilterFlags & EFilterFolderName)
    {
        if (iFilterFolderName.compare(KCDRAFTS, Qt::CaseInsensitive) != 0 &&
            iFilterFolderName.compare(KCALL, Qt::CaseInsensitive) != 0)
        {
            res = false;
        }
    }
    return res;
}

bool MessageFilter::allowSMS() const
{
    bool res = true;
    if (iFilterFlags & EFilterType)
    {
        if (!iFilterTypes.contains(KCSMS, Qt::CaseInsensitive))
        {
            res = false;
        }
    }
    return res;
}

bool MessageFilter::allowMMS() const
{
    bool res = true;
    if (iFilterFlags & EFilterType)
    {
        if (!iFilterTypes.contains(KCMMS, Qt::CaseInsensitive))
        {
            res = false;
        }
    }
    return res;
}

bool MessageFilter::allowEmail() const
{
    bool res = true;
    if (iFilterFlags & EFilterType)
    {
        if (!iFilterTypes.contains(KCEMAIL, Qt::CaseInsensitive))
        {
            res = false;
        }
    }
    return res;
}


bool MessageFilter::parseFilterTypes(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    QList<QVariant> filterlist;
    if (!checkMandatoryParamExists(aMatch, KCType, EFilterType, aErrorCode, aErrorMessage))
    {
        return false;
    }
    if (aMatch.contains(KCType))
    {
        if (aMatch.value(KCType).type() == QVariant::List )
        {
            filterlist = aMatch.value(KCType).toList();
            for (int i=0;i<filterlist.length();i++)
            {
                if (!addFilterType(filterlist.at(i), aErrorCode, aErrorMessage))
                {
                    break;
                }
            }
        }
        else if (aMatch.value(KCType).isValid())
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = KCBadDataType;
        }
    }

    if ((!(iFilterFlags & EFilterType)) && aErrorCode == SUCCESS)
    {
        iFilterTypes.append(KCSMS);
        iFilterTypes.append(KCMMS);
        iFilterFlags |= EFilterType;
    }
    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage << iFilterTypes;
    return aErrorCode == SUCCESS;
}

bool MessageFilter::parseFilterType(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    QList<QVariant> filterlist;
    if (!checkMandatoryParamExists(aMatch, KCType, EFilterType, aErrorCode, aErrorMessage))
    {
        return false;
    }
    if (aMatch.contains(KCType))
    {
        if (!addFilterType(aMatch.value(KCType), aErrorCode, aErrorMessage))
        {
            return false;
        }
    }

    //if no type is set - add SMS manually
    if ((!(iFilterFlags & EFilterType)) && aErrorCode == SUCCESS)
    {
        iFilterTypes.append(KCSMS);
        iFilterFlags |= EFilterType;
    }

    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage << iFilterTypes;
    return aErrorCode == SUCCESS;
}

bool MessageFilter::parseFilterId(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    if (!checkMandatoryParamExists(aMatch, KCID, EFilterId, aErrorCode, aErrorMessage))
    {
        return false;
    }
    if (aMatch.contains(KCID))
    {
        if (aMatch.value(KCID).type() == QVariant::String)
        {
            QString filter;
            filter = aMatch.value(KCID).toString();
            if ( filter.length() > 0 )
            {
                iFilterId = filter;
                iFilterFlags |= EFilterId;
             }
        }
        else if (aMatch.value(KCID).isValid())
        { //type is not string and value is not null
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = KCBadDataType;
        }
    }
    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage << iFilterId;
    return aErrorCode == SUCCESS;
}


bool MessageFilter::parseFilterSenders(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseStringList(aMatch, KCSenders, EFilterSenders, iFilterSenders, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseFilterTo(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseStringList(aMatch, KCTo, EFilterTo, iFilterTo, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseFilterCc(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseStringList(aMatch, KCCc, EFilterCc, iFilterCc, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseStringList(const QVariantMap& aMatch,
                         const QString& aParam,
                         const FilterFlags& aFlag,
                         QStringList& aResult,
                         int& aErrorCode,
                         QString& aErrorMessage)
{
    QList<QVariant> filterlist;
    if (!checkMandatoryParamExists(aMatch, aParam, aFlag, aErrorCode, aErrorMessage))
    {
        return false;
    }
    if (aMatch.contains(aParam))
    {
        if (aMatch.value(aParam).type() == QVariant::List )
        {
            filterlist = aMatch.value(aParam).toList();
            for (int i=0;i<filterlist.length();i++)
            {
                if (filterlist.at(i).type() == QVariant::String)
                {
                    QString member = filterlist.at(i).toString();
                    if (!member.isEmpty())
                    {
                        aResult.append(member);
                        iFilterFlags |= aFlag;
                    }
                }
                else if (filterlist.at(i).isValid())
                { //Cc is not null and not a string
                    aErrorCode = INVALID_ARG_ERR;
                    aErrorMessage = KCBadDataType;
                    break;
                }
            }
        }
        else if (aMatch.value(aParam).isValid())
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = KCBadDataType;
        }
    }
    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage << aResult;
    return aErrorCode == SUCCESS;
}


bool MessageFilter::parseFilterSubject(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    if (!checkMandatoryParamExists(aMatch, KCSubject, EFilterSubject, aErrorCode, aErrorMessage))
    {
        return false;
    }
    if (aMatch.contains(KCSubject))
    {
        if (aMatch.value(KCSubject).type() == QVariant::String)
        {
            QString filter;
            filter = aMatch.value(KCSubject).toString();
            if ( filter.length() > 0 )
            {
                if (filter.length() > KMaxSubjectLength)
                {
                    aErrorCode = DATA_OUT_OF_RANGE_ERR;
                    aErrorMessage = KCBadArgument;
                }
                iFilterSubject = filter;
                iFilterFlags |= EFilterSubject;
             }
        }
        else if (aMatch.value(KCSubject).isValid())
        { //type is not string and value is not null
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = KCBadDataType;
        }
    }
    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage << iFilterSubject;
    return aErrorCode == SUCCESS;
}

bool MessageFilter::parseFilterFolderName(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseString(aMatch, KCFolder, EFilterFolderName, iFilterFolderName, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseFilterBody(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseString(aMatch, KCBody, EFilterBody, iFilterBody, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseString(const QVariantMap& aMatch,
                     const QString& aParam,
                     const FilterFlags& aFlag,
                     QString& aResult,
                     int& aErrorCode,
                     QString& aErrorMessage)
{
    if (!checkMandatoryParamExists(aMatch, aParam, aFlag, aErrorCode, aErrorMessage))
    {
        return false;
    }
    if (aMatch.contains(aParam))
    {
        if (aMatch.value(aParam).type() == QVariant::String)
        {
            QString member;
            member = aMatch.value(aParam).toString();
            if ( member.length() > 0 )
            {
                aResult = member;
                iFilterFlags |= aFlag;
            }
        }
        else if (aMatch.value(aParam).isValid())
        { //type is not string and value is not null
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = KCBadDataType;
        }
    }
    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage << aParam << aResult;
    return aErrorCode == SUCCESS;
}

bool MessageFilter::parseFilterPriority(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseInt(aMatch, KCMessagePriority, EFilterPriority, iFilterPriority, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseFilterValidity(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseInt(aMatch, KCValidity, EFilterValidity, iFilterValidity, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseInt(const QVariantMap& aMatch,
                 const QString& aParam,
                 const FilterFlags& aFlag,
                 int& aResult,
                 int& aErrorCode, QString& aErrorMessage)
{
    if (!checkMandatoryParamExists(aMatch, aParam, aFlag, aErrorCode, aErrorMessage))
    {
        return false;
    }
    if (aMatch.contains(aParam))
    {
        if (aMatch.value(aParam).type() == QVariant::Double ||
            aMatch.value(aParam).type() == QVariant::Int)
        {
            aResult = aMatch.value(aParam).toInt();
            iFilterFlags |= aFlag;
        }
        else if (aMatch.value(aParam).isValid())
        { //type is not string and value is not null
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = KCBadDataType;
        }
    }
    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage << aParam << aResult ;
    return aErrorCode == SUCCESS;
}

bool MessageFilter::parseFilterUnread(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseBool(aMatch, KCUNREAD, EFilterUnread, iFilterUnread, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseBool(const QVariantMap& aMatch,
                 const QString& aParam,
                 const FilterFlags& aFlag,
                 bool& aResult,
                 int& aErrorCode, QString& aErrorMessage)
{
    if (!checkMandatoryParamExists(aMatch, aParam, aFlag, aErrorCode, aErrorMessage))
    {
        return false;
    }
    if (aMatch.contains(aParam))
    {
        if (aMatch.value(aParam).type() == QVariant::Bool)
        {
            aResult = aMatch.value(aParam).toBool();
            iFilterFlags |= aFlag;
        }
        else if (aMatch.value(aParam).isValid())
        { //type is not string and value is not null
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = KCBadDataType;
        }
    }
    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage << aParam << aResult ;
    return aErrorCode == SUCCESS;
}


bool MessageFilter::parseFilterAttachments(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    if (!checkMandatoryParamExists(aMatch, KCAttachment, EFilterAttachment, aErrorCode, aErrorMessage))
    {
        return false;
    }
    if (aMatch.contains(KCAttachment))
    {
        QVariantList attachmentList;
        QVariantMap attachmentData;
        QString filename;
        if (aMatch.value(KCAttachment).type() == QVariant::List)
        {
            attachmentList = aMatch.value(KCAttachment).toList();
            for (int i = 0; i < attachmentList.length(); i++)
            {
                if (attachmentList.at(i).type() == QVariant::String)
                {
                    filename = attachmentList.at(i).toString();
                    if (!filename.isEmpty())
                    {
                        if (filename.length() <= KMaxSubjectLength)
                        {
                            FilterAttachmentData data;
                            data.iFileName = filename;
                            iFilterAttachments.append(data);
                            iFilterFlags |= EFilterAttachment;
                            qDebug() << __FUNCTION__ << " Add" << filename;
                        }
                        else
                        {
                            aErrorCode = DATA_OUT_OF_RANGE_ERR;
                            aErrorMessage = KCBadArgument;
                            break;
                        }
                    }
                }
                else if (attachmentList.at(i).isValid())
                {
                    aErrorCode = INVALID_ARG_ERR;
                    aErrorMessage = KCBadDataType;
                    break;
                }
            }
        }
        else if (aMatch.value(KCAttachment).isValid())
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = KCBadDataType;
        }
    }
    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage;
    return aErrorCode == SUCCESS;
}

bool MessageFilter::parseFilterStartDate(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseDateTime(aMatch, KCStart, EFilterStartDate, iFilterStartTime, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseFilterEndDate(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage)
{
    return parseDateTime(aMatch, KCEnd, EFilterEndDate, iFilterEndTime, aErrorCode, aErrorMessage);
}

bool MessageFilter::parseDateTime(const QVariantMap& aMatch,
                 const QString& aParam,
                 const FilterFlags& aFlag,
                 QDateTime& aResult,
                 int& aErrorCode,
                 QString& aErrorMessage)
{
    if (!checkMandatoryParamExists(aMatch, aParam, aFlag, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if (aMatch.contains(aParam))
    {
        if (aMatch.value(aParam).type() == QVariant::DateTime)
        {
            aResult = aMatch.value(aParam).toDateTime();
            iFilterFlags |= aFlag;
        }
        else if (aMatch.value(aParam).isValid())
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = KCBadDataType;
        }
    }
    qDebug() << __FUNCTION__ << " End" << aErrorCode << aErrorMessage << aParam << aResult ;
    return aErrorCode == SUCCESS;
}

bool MessageFilter::checkMandatoryParamExists(const QMap<QString, QVariant>& aMatch, const QString& aParam, const FilterFlags& aFlag, int& aErrorCode, QString& aErrorMessage) const
{
    if (iMandatoryFlags & (uint) aFlag)
    {
        if (!aMatch.contains(aParam))
        {
            aErrorCode = MISSING_ARG_ERR;
            aErrorMessage = KCMissingArg;
            return false;
        }
    }
    return true;
}

bool MessageFilter::addFilterType(const QVariant& aData, int& aErrorCode, QString& aErrorMessage)
{
    if (aData.type() == QVariant::String)
    {
        QString type = aData.toString();
        if ((type.compare(KCSMS, Qt::CaseInsensitive) != 0) &&
            (type.compare(KCMMS, Qt::CaseInsensitive) != 0) &&
            (type.compare(KCEMAIL, Qt::CaseInsensitive) != 0))
        {
            if (!type.isEmpty())
            {
                aErrorCode = DATA_OUT_OF_RANGE_ERR;
                aErrorMessage = KCBadArgument;
            }
        }
        else
        {
            if (type.compare(KCEMAIL, Qt::CaseInsensitive) == 0)
            {
                //this is email. We should not have SMS or MMS in the list
                if (iFilterTypes.contains(KCSMS, Qt::CaseInsensitive) ||
                    iFilterTypes.contains(KCMMS, Qt::CaseInsensitive))
                {
                    aErrorCode = DATA_OUT_OF_RANGE_ERR;
                    aErrorMessage = KCBadArgument;
                }
            }
            else if ((type.compare(KCSMS, Qt::CaseInsensitive) != 0) ||
            (type.compare(KCMMS, Qt::CaseInsensitive) != 0))
            {
                //this is SMS or MMS. We should not have Email in the list
                if (iFilterTypes.contains(KCEMAIL, Qt::CaseInsensitive))
                {
                    aErrorCode = DATA_OUT_OF_RANGE_ERR;
                    aErrorMessage = KCBadArgument;
                }
            }

            if (aErrorCode == SUCCESS)
            {
                iFilterTypes.append(type);
                iFilterFlags |= EFilterType;
            }
        }
    }
    else if (aData.isValid())
    {
        aErrorCode = INVALID_ARG_ERR;
        aErrorMessage = KCBadDataType;
    }
    return aErrorCode == SUCCESS;
}

