/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QtCore>
#include "messagedata.h"
#include "qmessagingconstants.h"

//QtMobility includes
#include <QMessageId>
#include <QMessageAddress>
#include <QMessageContentContainer>
#include <QMessageContentContainerId>

MessageData::MessageData():
iType(""),
iBody(""),
iSubject(""),
iValidityMins(0),
iPriority(0),
iStartEditor(false)
{
}


bool MessageData::fillMessageData(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage, bool aStartEditor)
{
    iStartEditor = aStartEditor;

    if (!getMessageType(aMessageData, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if (!getMessageBody(aMessageData, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if (!getMessageToList(aMessageData, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if (!getMessageCcBccList(aMessageData, KCCc, iCcRecipients, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if (!getMessageCcBccList(aMessageData, KCBcc, iBccRecipients, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if (!getMessageValidity(aMessageData, aErrorCode, aErrorMessage))
    {
        return false;
    }

    if (!getMessagePriority(aMessageData, aErrorCode, aErrorMessage))
    {
        return false;
    }


    if (iType == KCMMS || iType == KCEMAIL)
    {

        if (!getMessageSubject(aMessageData, aErrorCode, aErrorMessage))
        {
            return false;
        }

        if (!getMessageAttachments(aMessageData, aErrorCode, aErrorMessage))
        {
            return false;
        }
    }
    return true;
}

bool MessageData::getMessageType(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage)
{
    qDebug("MessageData::getMessageType begin");
    bool isType = aMessageData.contains(KCType);
    aErrorCode = SUCCESS;
    QString type;
    if (!isType)
    {
        iType = KCSMS;
    }
    else
    {
        if (aMessageData.value(KCType).isValid())
        {
            if (aMessageData.value(KCType).type() != QVariant::String)
            {
                aErrorCode = INVALID_ARG_ERR;
                aErrorMessage = KCBadDataType;
            }
            else
            {
                if (!aMessageData.value(KCType).toString().isEmpty())
                {
                    iType = aMessageData.value(KCType).toString();
                    if ((iType.compare(KCSMS, Qt::CaseInsensitive) != 0) &&
                        (iType.compare(KCMMS, Qt::CaseInsensitive) != 0) &&
                        (iType.compare(KCEMAIL, Qt::CaseInsensitive) != 0))
                    {
                        aErrorCode = DATA_OUT_OF_RANGE_ERR;
                        aErrorMessage = KCBadArgument;
                    }
                }
                else
                {
                    iType = KCSMS;
                }
            }
        }
        else
        {
            iType = KCSMS;
        }
    }

    qDebug() << "MessageData::getMessageType end" << iType << aErrorCode << aErrorMessage;
    return aErrorCode == SUCCESS;
}

bool MessageData::getMessageToList(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage)
{
    qDebug() << "MessageData::getMessageToList begin";
    aErrorCode = SUCCESS;
    bool isToList = aMessageData.contains(KCTo);
    bool hasRecepients = false;
    if (isToList)
    {
        QVariantList toList;
        if (aMessageData.value(KCTo).type() == QVariant::List)
        {
            toList = aMessageData.value(KCTo).toList();
            if (toList.length())
            {
                for (int i = 0; i < toList.length(); i++)
                {
                    hasRecepients = true;
                    if (toList.at(i).type() == QVariant::String)
                    {
                        QString number = toList.at(i).toString();
                        if (!number.isEmpty())
                        {
                            bool isNumber = checkValidNumber(number);
                            if (isNumber)
                            {
                                if (!iRecipients.contains(number))
                                {
                                    iRecipients.append(number);
                                    qDebug("MessageData::getMessageToList - recipient[%s]", qPrintable(number));
                                }
                                else
                                {
                                    qDebug("Duplicate recipient[%s] - ignoring", qPrintable(number));
                                }
                            }
                            else
                            {
                                aErrorMessage = KCBadArgument;
                                aErrorCode = DATA_OUT_OF_RANGE_ERR;
                            }
                        }
                        else
                        { //According to specification, 'to' argument is mandatory even for startEditor
                            aErrorMessage = KCMissingArg;
                            aErrorCode = MISSING_ARG_ERR;
                        }
                    }
                    else if (toList.at(i).isValid())
                    {
                        aErrorMessage = KCBadDataType;
                        aErrorCode = INVALID_ARG_ERR;
                    }
                    else
                    {
                        aErrorMessage = KCMissingArg;
                        aErrorCode = MISSING_ARG_ERR;
                    }
                }
            }
            else if (isSend())
            { //According to specification, 'to' argument is mandatory for send
                aErrorMessage = KCMissingArg;
                aErrorCode = MISSING_ARG_ERR;
            }
        }
        else if (aMessageData.value(KCTo).isValid())
        { //'to' type is valid, but it is not list
            aErrorMessage = KCBadDataType;
            aErrorCode = INVALID_ARG_ERR;
        }
        else
        { //'to' list is null
            aErrorMessage = KCMissingArg;
            aErrorCode = MISSING_ARG_ERR;
        }
    }
    else if (isSend())
    { //According to specification, 'to' argument is mandatory for send
        aErrorMessage = KCMissingArg;
        aErrorCode = MISSING_ARG_ERR;
    }

    if (!hasRecepients && aErrorCode == SUCCESS && isSend())
    { //According to specification, 'to' argument is mandatory for send
        aErrorMessage = KCMissingArg;
        aErrorCode = MISSING_ARG_ERR;
    }

    qDebug() << "MessageData::getMessageToList end" << iRecipients << ". error= " << aErrorCode;
    return aErrorCode == SUCCESS;
}

bool MessageData::getMessageCcBccList(const QVariantMap& aMessageData, const QString& aKey, QStringList& aRecipients, int& aErrorCode, QString& aErrorMessage)
{
    qDebug() << "MessageData::getMessageCcBccList begin " << aKey;
    aErrorCode = SUCCESS;
    bool isCcList = aMessageData.contains(aKey);
    if (isCcList)
    {
        QList<QVariant> ccList;
        if (aMessageData.value(aKey).type() == QVariant::List)
        {
            ccList = aMessageData.value(aKey).toList();
            for (int i = 0; i < ccList.length(); i++)
            {
                if (ccList.at(i).type() == QVariant::String)
                {
                    QString number = ccList.at(i).toString();
                    bool isNumber = checkValidNumber(number);
                    if (isNumber)
                    {
                        if (!aRecipients.contains(number))
                        {
                            aRecipients.append(number);
                            qDebug("MessageData::getMessageCcBccList - recipient[%s]", qPrintable(number));
                        }
                        else
                        {
                            qDebug("Duplicate recipient[%s] - ignoring", qPrintable(number));
                        }
                    }
                    else
                    {
                        aErrorMessage = KCBadArgument;
                        aErrorCode = DATA_OUT_OF_RANGE_ERR;
                    }
                }
                else if (ccList.at(i).isValid())
                {
                    aErrorMessage = KCBadDataType;
                    aErrorCode = INVALID_ARG_ERR;
                }
            }
        }
        else if (aMessageData.value(KCCc).isValid())
        {
            aErrorMessage = KCBadDataType;
            aErrorCode = INVALID_ARG_ERR;
        }
    }
    qDebug() << "MessageData::getMessageCcList end" << aRecipients << aErrorCode;
    return aErrorCode == SUCCESS;
}

bool MessageData::getMessageBody(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage)
{
    qDebug() << "MessageData::getMessageBody begin";
    aErrorCode = SUCCESS;
    bool isBodyText = aMessageData.contains(KCBody);
    if (isBodyText)
    {
        if (aMessageData.value(KCBody).type() == QVariant::String)
        {
            iBody = aMessageData.value(KCBody).toString();
        }
        else if (aMessageData.value(KCBody).isValid())
        {
            aErrorMessage = KCBadDataType;
            aErrorCode = INVALID_ARG_ERR;
        }
    }
    qDebug() << "MessageData::getMessageBody end" << iBody <<  aErrorCode;
    return aErrorCode == SUCCESS;
}

bool MessageData::getMessageSubject(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage)
{
    qDebug() << "MessageData::getMessageSubject begin";
    aErrorCode = SUCCESS;
    bool isSubjectText = aMessageData.contains(KCSubject);
    if (isSubjectText)
    {
        if (aMessageData.value(KCSubject).type() == QVariant::String)
        {
            iSubject = aMessageData.value(KCSubject).toString();
            if (iSubject.length() > KMaxSubjectLength)
            {
                aErrorMessage = KCBadArgument;
                aErrorCode = DATA_OUT_OF_RANGE_ERR;
            }
        }
        else if (aMessageData.value(KCSubject).isValid())
        {
            aErrorMessage = KCBadDataType;
            aErrorCode = INVALID_ARG_ERR;
        }
    }
    qDebug() << "MessageData::getMessageSubject end" <<  iSubject << aErrorCode;
    return aErrorCode == SUCCESS;
}

bool MessageData::getMessageAttachments(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage)
{
    qDebug() << "MessageData::getMessageAttachments begin";
    aErrorCode = SUCCESS;
    bool isAttList = aMessageData.contains(KCAttachment);
    if (isAttList)
    {
        QList<QVariant> attList;
        QVariantMap attachmentData;
        if (aMessageData.value(KCAttachment).type() == QVariant::List)
        {
            attList = aMessageData.value(KCAttachment).toList();
            qDebug() << attList;
            for (int i = 0; i < attList.length(); i++)
            {
                if (attList.at(i).type() == QVariant::Map)
                {
                    attachmentData = attList.at(i).toMap();
                    if (attachmentData.value(KCUri).type() == QVariant::String)
                    {
                        QString filename = attachmentData.value(KCUri).toString();
                        if (filename.length() <= KMaxAttachmentUriLength)
                        {
                            if (checkAttachmentValid(filename, aErrorCode, aErrorMessage))
                            {
                                iAttachments.append(filename);
                                qDebug("QMessagingService::getMessageAttachments - attach[%s]", qPrintable(filename));
                            }
                        }
                        else
                        {
                            qDebug() << "Attachment URI is too long";
                            aErrorMessage = KCInvalidUri; //for test case Messaging_Error_208
                            aErrorCode = INVALID_URI_ERR;
//                            aErrorMessage = KCBadArgument; //for test case Messaging_Interface_233
//                            aErrorCode = DATA_OUT_OF_RANGE_ERR;
                        }
                    }
                    else if (attachmentData.value(KCUri).isValid())
                    {
                        qDebug() << "Error: attachment uri is not a string";
                        aErrorMessage = KCBadDataType;
                        aErrorCode = INVALID_ARG_ERR;
                    }
                }
                else if (attList.at(i).isValid())
                {
                    qDebug() << "Error: attachment is not a map";
                    aErrorMessage = KCBadDataType;
                    aErrorCode = INVALID_ARG_ERR;
                }
            }
        }
        else if (aMessageData.value(KCAttachment).isValid())
        {
            qDebug() << "Error: attachment list has incorrect type";
            aErrorMessage = KCBadDataType;
            aErrorCode = INVALID_ARG_ERR;
        }
    }
    qDebug() << "MessageData::getMessageAttachments" << iAttachments << aErrorCode;
    return aErrorCode == SUCCESS;
}

bool MessageData::getMessageValidity(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage)
{
    qDebug() << "MessageData::getMessageValidity begin";
    aErrorCode = SUCCESS;
    bool isValidity = aMessageData.contains(KCValidity);
    if (isValidity)
    {
        if (aMessageData.value(KCValidity).type() == QVariant::Double ||
            aMessageData.value(KCValidity).type() == QVariant::Int)
        {
            iValidityMins = aMessageData.value(KCValidity).toInt();
        }
        else if (aMessageData.value(KCValidity).isValid())
        {
            aErrorMessage = KCBadDataType;
            aErrorCode = INVALID_ARG_ERR;
        }
    }
    qDebug() << "MessageData::getMessageValidity end" << iValidityMins <<  aErrorCode;
    return aErrorCode == SUCCESS;
}

bool MessageData::getMessagePriority(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage)
{
    qDebug() << "MessageData::getMessagePriority begin";
    aErrorCode = SUCCESS;
    bool isPriority = aMessageData.contains(KCMessagePriority);
    if (isPriority)
    {
        if (aMessageData.value(KCMessagePriority).type() == QVariant::Double ||
            aMessageData.value(KCMessagePriority).type() == QVariant::Int)
        {
            iPriority = aMessageData.value(KCMessagePriority).toInt();
        }
        else if (aMessageData.value(KCMessagePriority).isValid())
        {
            aErrorMessage = KCBadDataType;
            aErrorCode = INVALID_ARG_ERR;
        }
    }
    qDebug() << "MessageData::getMessagePriority end" << iPriority <<  aErrorCode;
    return aErrorCode == SUCCESS;
}

bool MessageData::checkValidNumber( const QString& aNumber ) const
{
    if (aNumber.length())
    {
        if (iType.compare(KCSMS, Qt::CaseInsensitive) == 0)
        {
            int pos = 0;
            if (!aNumber.at(pos).isDigit() && !(aNumber.at(pos) == '+'))
            {
                return false;
            }
            for (int i=1 ; i<aNumber.length(); i++)
            {
                if (!aNumber.at(i).isDigit())
                {
                    return false;
                }
            }
        }
        return true;
    }
    return false;

}

bool MessageData::checkAttachmentValid(const QString& aUri, int& aErrorCode, QString& aErrorMessage) const
{
    QUrl uri = aUri;
    if (!uri.isEmpty())
    { //allow empty uri
        if (!uri.isValid())
        {
            aErrorCode = INVALID_URI_ERR;
            aErrorMessage = KCBadDataType;
            return false;
        }

        QString uriFile = uri.toLocalFile();
        if (!QFile::exists(uriFile))
        {
            qDebug() << "Attachment file does not exist " << uriFile;
            aErrorCode = DATA_NOT_FOUND_ERR;
            aErrorMessage = KCDataNotFound;
            return false;
        }

        QDir dir (uriFile);
        if (dir.exists())
        {
            qDebug() << "URL is a directory" << uriFile;
            aErrorCode = DATA_NOT_FOUND_ERR;
            aErrorMessage = KCDataNotFound;
            return false;
        }
    }
    return true;
}

QVariantMap MessageData::fillMessageMap(QMessage& aEvent)
{
    qDebug() << __FUNCTION__ << aEvent.id().toString();
    QVariantMap msg;
    QVariantMap msgData;
    msg.insert(KCID, aEvent.id().toString());
    msg.insert(KCUNREAD, !(aEvent.status() & QMessage::Read));

    int folder = aEvent.standardFolder();
    if (folder == QMessage::InboxFolder)
    {
        msg.insert(KCTIME, aEvent.receivedDate());
        msg.insert(KCSender, aEvent.from().addressee());
        msgData.insert(KCTo, "Your phone");
    }
    else
    {
        msg.insert(KCTIME, aEvent.date());
        msg.insert(KCSender, "Your phone");
        msgData.insert(KCTo, aEvent.from().addressee());        
    }

    QString folderName;
    if (folder == QMessage::DraftsFolder)
    {
        folderName = KCDRAFTS;
    }
    else if (folder == QMessage::InboxFolder)
    {
        folderName = KCINBOX;
    }
    else if (folder == QMessage::SentFolder)
    {
        folderName = KCSENTBOX;
    }
    else if (folder == QMessage::OutboxFolder)
    {
        folderName = KCOUTBOX;
    }
    msg.insert(KCFolder, folderName);

    int type = aEvent.type();
    if (type == QMessage::Sms)
    {
        msgData.insert(KCType, KCSMS);
        //SMS has only normal priority
        msgData.insert(KCMessagePriority, PRIORITY_NORMAL);
    }
    else if (type == QMessage::Mms)
    {
        //Maemo5 doesn't support the MMS messages
        msgData.insert(KCType, KCMMS);
        msgData.insert(KCMessagePriority, PRIORITY_NORMAL);
    }
    else if(type == QMessage::Email)
    {
        msgData.insert(KCType, KCEMAIL);
        int msgPriority = QMessage::NormalPriority;
        if (aEvent.priority() == QMessage::HighPriority)
        {
            msgPriority = PRIORITY_HIGH;
        }
        else if (aEvent.priority() == QMessage::LowPriority)
        {
            msgPriority = PRIORITY_LOW;
        }
        msgData.insert(KCMessagePriority, msgPriority);
    }

    msgData.insert(KCBody, aEvent.find(aEvent.bodyId()).textContent());
    msgData.insert(KCSubject, aEvent.subject());

//    QMessageAddressList addrList = aEvent.to();
//    QStringList recepientsList;
//    foreach (const QMessageAddress& addr, addrList)
//    {
//        recepientsList.append(addr.addressee());
//    }
//    msgData.insert(KCTo, recepientsList);
//    qDebug() << "Message recepients " << recepientsList;

    QList<QVariant> dummylist;
    msgData.insert(KCAttachment, dummylist); //set no attachments for SMS

    msg.insert(KCMessage, msgData);
    return msg;
}
