/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QMapIterator>
#include <QMap>
#include <QVariant>
#include <QFile>
#include <QPixmap>
#include <QDir>
#include <QFileInfo>
#include <QDateTime>


#include "resizeimagetask.h"
#include "qmediaerrorcodes.h"
#include "qmediakeyconstants.h"

ResizeImageTask::ResizeImageTask(int aTransactionId, QUrl aUrl, int aWidth, int aHeight, int aAspectRatioMode, const QString& aDestination)
    : iTransactionId(aTransactionId),
    iSourceUri(aUrl),
    iWidth(aWidth),
    iHeight(aHeight),
    iAspectRatioMode (aAspectRatioMode),
    iDestination(aDestination),
    iOriginal (NULL),
    iResult (NULL)
{
    connect(this, SIGNAL(asyncLoad(bool)), this, SLOT(doLoad(bool)), Qt::QueuedConnection);
    connect(this, SIGNAL(asyncScale(bool)), this, SLOT(doScale(bool)), Qt::QueuedConnection);
    connect(this, SIGNAL(asyncSave(bool)), this, SLOT(doSave(bool)), Qt::QueuedConnection);
}
ResizeImageTask::~ResizeImageTask()
{
    qDebug() << Q_FUNC_INFO;
    delete iOriginal;
    delete iResult;
}

void ResizeImageTask::finishTask()
{
    delete this;
}

void ResizeImageTask::run(bool synchronous)
{
    qDebug() << "ResizeImageTask::run(): " << iSourceUri.toString() << synchronous;
    if (synchronous)
    {
        doLoad(synchronous);
    }
    else
    {
        emit asyncLoad(synchronous);
    }
    qDebug() << "ResizeImageTask::run() end: " << iSourceUri.toString();
}

void ResizeImageTask::runFromScaling(const QPixmap& aOriginalPixmap)
{
    iOriginal = new QPixmap(aOriginalPixmap);

    emit asyncScale(false);

}

void ResizeImageTask::doLoad(bool synchronous)
{
    qDebug() << Q_FUNC_INFO << iTransactionId;
    QString localPath = iSourceUri.toLocalFile();
    iOriginal = new QPixmap(localPath);
    if (synchronous)
    {
        doScale(synchronous);
    }
    else
    {
        emit asyncScale(synchronous);
    }
    qDebug() << Q_FUNC_INFO << "End";
}

void ResizeImageTask::doScale(bool synchronous)
{
    qDebug() << Q_FUNC_INFO << iTransactionId;

    if(iWidth==1 || iHeight==1)
    {
        iAspectRatioMode=Qt::IgnoreAspectRatio;
    }
    iResult = new QPixmap(iOriginal->scaled(iWidth, iHeight, (Qt::AspectRatioMode)iAspectRatioMode, Qt::SmoothTransformation));

    if (synchronous)
    {
        doSave(synchronous);
    }
    else
    {
        emit asyncSave(synchronous);
    }
    // qDebug() << Q_FUNC_INFO << "End";
}

void ResizeImageTask::doSave(bool synchronous)
{
    qDebug() << Q_FUNC_INFO << iTransactionId;
    int err = NO_ERROR;
    QString localPath = iSourceUri.toLocalFile();
    QFileInfo fileInfo(localPath);
    qDebug() << localPath;

    //construct temp file name
    QString tempFilePath = "";
    QString tmpFileName = fileInfo.baseName() + QString::number(
            QDateTime::currentDateTime().toTime_t())+ DOT
                          + fileInfo.completeSuffix();
    if (iDestination.isEmpty())
    {
        tempFilePath = TmpThumbnailPath + FORWARDSLASHSTRING
                       + tmpFileName;
    }
    else
    {
        QUrl destUri = iDestination;
        QFileInfo destFileInfo(destUri.toLocalFile());
        if (destFileInfo.isDir())
        {
            tempFilePath = destFileInfo.absoluteFilePath() + FORWARDSLASHSTRING + tmpFileName;
        }
        else
        {
            tempFilePath = destFileInfo.absoluteFilePath();
        }
    }


    if (iResult->isNull() || !iResult->toImage().convertToFormat(QImage::Format_RGB32).save(tempFilePath))
    {
        if (iResult->isNull())
            qDebug() << "Resized image is null";
        else
            qDebug() << "Failed to save into:" << tempFilePath;
        tempFilePath = EMPTY_STRING;
        err = GENERAL_ERROR;
    }
    else
    {
        tempFilePath = QUrl::fromLocalFile(tempFilePath).toString();
    }

    qDebug() << "Emit complete signal: " << tempFilePath << err;
    if (synchronous)
    {
        iResultError = err;
        iResultUri = tempFilePath;
    }
    else
    {

        emit signalResizeImageCb(this, tempFilePath, err, iTransactionId);
        //WARNING: do not put any code below because signalResizeImageCb handler may kill the task
    }
}
