/*
* ============================================================================
*  Name        : qlocationservice.h
*  Part of     : serviceframework / WRT
*  Description : Qt class for location provider
*  Version     : %version: 16 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef LOCATIONPROVIDER_H_
#define LOCATIONPROVIDER_H_


#include <QObject>
#include <QMap>
#include <QVariant>

#include "servicesecurityinterface.h"
#include "qlocationserviceinterface.h"
#include "timeObserver.h"
#include "emitAsyncTask.h"

// Qt Mobility header
#include <QGeoPositionInfoSource>
QTM_USE_NAMESPACE

//Time conversion factor from milliseconds to Microseconds
const int KTimeConversion = 1000;
//Default value for timeout attribute (from S60 implementation)
const int KTimeInterval = 15000;

//PositionOption object properties
const QString KEnableHighAccOption("enableHighAccuracy");
const QString KTimeOutOption("timeout");
const QString KMaxAgeOption("maximumAge");

const QString KTimeOutOptionUpdated("UpdateTimeOut");
const QString KMaxAgeOptionUpdated("MaxUpdateAge");

//Keys for Position Map
const QString KLongitude = "longitude";
const QString KLatitude = "latitude";
const QString KAltitude = "altitude";
const QString KAccuracy = "accuracy";
const QString KAltitudeAccuracy = "altitudeAccuracy";
const QString KHeading = "heading";
const QString KSpeed = "speed";
const QString KTime = "timestamp";

//Keys for Error Map
const QString KErrCode = "errorCode";
const QString KErrMsg = "errorMessage";
const QString KTransId = "transactionId";
const QString KUnKErr = "Unknown";
const int KInvalidTransId = -1;

//error codes
const int KErrorBase = 100;
const int KMaxLength = 256;
const int KErrNone = 0;
const int KErrTimeOut = -33;
const int KErrArgument = -6;
const int KErrNotSupported = -5;
const int KErrInUse = -14;
const int KErrAccessDenied = -21;
const int KErrNotFound = -1;
const int KErrGeneral = -2;
const int KErrDeviceNotReady = -18;

//operation type
const int KAdd = 0;
const int KUpdate = 1;

const QString KGps = "gps";
const QString KAGps = "agps";
const QString KCellId = "cellid";
const QString KAllMethods = "allMethods";

//General Error Messages
const QString KMissingArgErr("A mandatory argument in a method is missing");
const QString KInvalidArgErr("Argument passed is of different type than what is expected");
const QString KNotSupported("Not supported error");
const QString KDataNotFound("Data  not found");
const QString KNotAllowedErr("user doesnt have permission to access specified content");
const QString KServiceInUse("Service already in use");
const QString KGeneralError("general error");
const QString KMethodName("methodname");
const QString KMethodInvalid("invalid method name argument");
const QString KStrErrInvalidTransID( "Transaction id is invalid" );

// Forward class declaration
class timeObserver;

/**
 * class LocationProvider
 * Contains methods for getting location of the device as well subscribing to get notification
 * whenever device location changes.
 */

class LocationProvider : public QObject,
                         public ILocation,
                         public IServiceSecurity

{
    Q_OBJECT
    Q_INTERFACES(ILocation)

public:
    /**
     * Default constructor
     * @since x.y
     * @return none
     */
    LocationProvider();

    /**
     * Destructor
     * @since x.y
     * @return none
     */
    virtual ~LocationProvider();

    /**
     * setSecuritySession Setting a security session
     * @param secSession handle to the security session
     * @return none
     * @since x.y
     */
    void setSecuritySession(WRT::SecSession *secSession);

    /**
     * Inherited from MLocationCallBack;used to  Notify that position data has
     * been obtained or an error has occured.
     * @param position contains position data in case of no errors
     * @param errorCode contains error code.
     * @parama transactionId
     * @return none
     * @since x.y
     */

signals:
    /**
     * Emitted as signal whenever a new location
     * data is obtained or an error occurs during retrieval.
     * @param errorCode error that has occured
     * @param transactionId uniquely identifies a getCurrentPosition request.
     * @param position position data in case error param is 0.This
     * in turn contains map with position information such longitude,latitude etc.
     * @return none
     * @since x.y
     */
    void AsyncCallback(int errorCode,int transactionId, const QMap<QString,QVariant>& position);

    /**
     * Emitted as signal whenever a new location
     * data is obtained or an error occurs during retrieval.
     * @param errorCode error that has occured
     * @param transactionId uniquely identifies a getCurrentPosition request.
     * @param position position data in case error param is 0.This
     * in turn contains map with position information such longitude,latitude etc.
     * @param methodName method using which position fix was obtained
     * @return none
     * @since x.y
     */
    void MethodAsyncCallback(int errorCode,
                             int transactionId,
                             const QMap<QString,QVariant>& position,
                             QString methodName);

public slots://Methods inherited from ILocation
    /**
     * Get the current position of the device.
     * @param positionOptions position option that is used while
     * retrieving the position information.positionOptions
     * is a  Map described as below:
     * --------------------------------------------------------------------------------------
     *  |Key                | Data type|Description
     *  --------------------------------------------------------------------------------------
     *  |enableHighAccuracy | bool     |Provides a hint that application would like to
     *  |                   |          |receive best possible location estimates.
     *  --------------------------------------------------------------------------------------
     *  |timeout            | double   |Maximum length of time in milliseconds that is allowed
     *  |                   |          |to pass before location estimate is made.
     *  --------------------------------------------------------------------------------------
     *  |maximumAge         | double   |this indicate that application is ready to accept cached
     *  |                   |          |position which is not older than specified maximumAge in
     *  |                   |          |milliseconds.
     *  ----------------------------------------------------------------------------------------
     * @param isLastKnownLocation indicates whether to get the last known location of the device.
     * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
     * below :
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
     * ---------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * --------------------------------------------------------------------------------
     * |transactionId       | int      |Unique Id identifying this request in case of success.
     * ---------------------------------------------------------------------------------
     * @emits newPositionData it emits newPositionData signal to notify the result.
     * @since x.y
     */
    QVariant getCurrentPosition(const QMap<QString,QVariant>& positionOptions,
                                bool isLastKnownLocation);

    /**
     * Subscribe to get notification whenever device position changes.
     * @param positionOptions position option that is used while
     * retrieving the position information.positionOptions is a
     * Map described as below:
     * is a  Map described as below:
     * -------------------------------------------------------------------------------------
     *  |Key                | Data type|Description
     *  --------------------------------------------------------------------------------------
     *  |enableHighAccuracy | bool     |Provides a hint that application would like to
     *  |                   |          |receive best possible location estimates.
     *  --------------------------------------------------------------------------------------
     *  |timeout            | double   |Maximum length of time in milliseconds that is allowed
     *  |                   |          |to pass before location estimate is made.
     *  --------------------------------------------------------------------------------------
     *  |maximumAge         | double   |this indicate that application is ready to accept cached
     *  |                   |          |position which is not older than specified maximumAge in
     *  |                   |          |milliseconds.
     * -------------------------------------------------------------------------------------------
     * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
     * below :
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
     * ---------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * --------------------------------------------------------------------------------
     * |transactionId       | int      |Unique Id identifying this request in case of success.
     * ---------------------------------------------------------------------------------
     * @emits newPositionData it emits newPositionData signal to notify the result.
     * @since x.y
     */
    QVariant watchPosition(const QMap<QString,QVariant>& positionOptions);

    /**
     * Clear the previously subscribed watchPosition.
     * @parama watchId unique ID which corresponds to one active
     * watchPosition.
     * @return  Map containing errorCode,errorMessage.Content of this map is given
     * below :
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
     * ---------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * ---------------------------------------------------------------------------------
     * @since x.y
     */
    QVariant clearWatch(const int watchId);

    /**
     * Get the location data by specifying a particular positioning technology.
     * @param posMethod specifies a particular positioning technology to be used
     * possible values are cellid,gps,agps
     * @param tid transaction ID
     * @return  Map containing errorCode,errorMessage.Content of this map is given
     * below :
     * --------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * --------------------------------------------------------------------------------
     * |errorCode           | int      | error code in case of error, 0 otherwise.
     * ---------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * ---------------------------------------------------------------------------------
     * @since x.y
     */
    QVariant getLocationUsingMethodName(QString methodName,int tid);

//internal utility functions
private:
    /**
     * Set Update options.
     * @param positionOptions Map containing position options to be set
     * @param updateOpts LBS upadate class that gets populated as a result
     * of this method invocation.
     * @return 0 in case of success or non-zero error code.
     */
    static int setUpdateOptions(const QMap<QString,QVariant>& positionOptions, QMap<QString,QVariant>& updateOpts);

    /**
     * hasData : check if supplied value is null or empty string
     * @param datValue : value to be checked
     * @return false if value is emty string or null true otherwise
     */
    static bool hasData(QVariant dataValue);

    void fillData(const QGeoPositionInfo& posInfo, QMap<QString, QVariant>& data);
    void fireSignalAll(int errStatus, QMap<QString, QVariant>& data);
    void fireSignal(int errStatus, QMap<QString, QVariant>& data, qint32 transId);
    bool checkMaxAge(QGeoPositionInfo& posInfo, QVariant maxAge);

private slots:
    void posUpdated(const QGeoPositionInfo& aPosition);
    void timerTimeout(qint32 aId);
    void asyncCB(int errStatus, qint32 transId, QMap<QString, QVariant> retMap);

private:
    //Handle to the security session
    WRT::SecSession *securitySession;

    qint32 m_transIdCount;  // Counter for generating transaction id for async requested
    QMap<qint32, timeObserver*> getPosTimersList;
    QMap<qint32, timeObserver*> watchTimersList;
    QList<qint32> getPosCallsList;
    QList<qint32> watchCallsList;
};
#endif //LOCATIONPROVIDER_H
