/*
* ============================================================================
*  Name        : qlocationservice.cpp
*  Part of     : serviceframework / WRT
*  Description : Qt class for Location service
*  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QMap>
#include <QVariant>
#include <QtCore>
#include <QDateTime>
#include "secsession.h"
#include "qlocationservice.h"
#include "qlocationengine.h"
#include <QtDebug>

#include <QGeoPositionInfo>

LocationProvider::LocationProvider(): m_transIdCount(0)
{
    qDebug() << __FUNCTION__;
}

LocationProvider::~LocationProvider()
{
    //delete all observer objects
    foreach(qint32 id, getPosTimersList.keys())
    {
        if (getPosTimersList.value(id))
        {
            delete getPosTimersList.take(id);
        }
    }
    foreach(qint32 id, watchTimersList.keys())
    {
        if (watchTimersList.value(id))
        {
            delete watchTimersList.take(id);
        }
    }
    LocationEngine::freeResources();
}

void LocationProvider::setSecuritySession(WRT::SecSession *secSession)
{
    securitySession = secSession;
}

QVariant LocationProvider::getCurrentPosition(const QMap<QString,QVariant>& positionOptions,
                                              bool isLastKnownLocation)
{
    int error = KErrNone;
    QMap<QString,QVariant> retMap;
    QMap<QString,QVariant> updateOptions;

    //Gets a new transaction id by incrementing counter
    qint32 transId = ++m_transIdCount;
    LocationEngine* engine = LocationEngine::getInstance(this);

    if (isLastKnownLocation)
    {
        QGeoPositionInfo posInfo;
        if (engine->getPositionInfoSource())
        {
            posInfo = engine->getPositionInfoSource()->lastKnownPosition(false);
            if (posInfo.coordinate().isValid())
            {
                QMap<QString, QVariant> data;
                fillData(posInfo, data);
                asyncTask* task = new asyncTask(KErrNone, transId, data);
                connect(task, SIGNAL(asyncCB(int,qint32,QMap<QString,QVariant>)),
                        this, SLOT(asyncCB(int,qint32,QMap<QString,QVariant>)));
                QThreadPool::globalInstance()->start(task);

                retMap.insert(KErrCode, KErrNone);
                retMap.insert(KTransId, transId);
                return retMap;
            }
        }
        retMap.insert(KErrCode, KErrGeneral);
        retMap.insert(KErrMsg, KUnKErr);
        retMap.insert(KTransId, KInvalidTransId);
        return retMap;
    }

    error = setUpdateOptions(positionOptions, updateOptions);
    if (error != KErrNone)
    {
        //Populate error code Map
        retMap[KErrCode]= error;
        retMap[KErrMsg] = KUnKErr;
        retMap[KTransId] = KInvalidTransId;
        return retMap;
    }

    if (updateOptions.contains(KMaxAgeOptionUpdated))
    {
        if (engine->getPositionInfoSource())
        {
            QGeoPositionInfo posInfo;
            posInfo = engine->getPositionInfoSource()->lastKnownPosition(false);
            if (checkMaxAge(posInfo, updateOptions.value(KMaxAgeOptionUpdated)))
            {
                QMap<QString, QVariant> data;
                fillData(posInfo, data);
                asyncTask* task = new asyncTask(KErrNone, transId, data);
                connect(task, SIGNAL(asyncCB(int,qint32,QMap<QString,QVariant>)),
                        this, SLOT(asyncCB(int,qint32,QMap<QString,QVariant>)));
                QThreadPool::globalInstance()->start(task);

                retMap.insert(KErrCode, KErrNone);
                retMap.insert(KTransId, transId);
                return retMap;
            }
        }
    }

    if (engine->getPositionInfoSource())
    {
        engine->getPositionInfoSource()->requestUpdate();
        getPosCallsList.append(transId);
        timeObserver* timeObs = NULL;
        if (updateOptions.contains(KTimeOutOptionUpdated))
        {
            timeObs = new timeObserver(transId, updateOptions.value(KTimeOutOptionUpdated));
        }
        else
        {
            timeObs = new timeObserver(transId, KTimeInterval);
        }
        if (timeObs)
        {
            getPosTimersList.insert(transId, timeObs);
            connect(timeObs, SIGNAL(timeOutCB(qint32)),
                    this, SLOT(timerTimeout(qint32)));
            timeObs->run();
        }

        retMap.insert(KErrCode, KErrNone);
        retMap.insert(KTransId, transId);
        return retMap;
    }
    else
    {
        retMap.insert(KErrCode, KErrGeneral);
        retMap.insert(KTransId, transId);
        return retMap;
    }
}

QVariant LocationProvider::watchPosition(const QMap<QString,QVariant>& positionOptions)
{
    QMap<QString,QVariant> retMap;
    QMap<QString,QVariant> updateOptions;

    int error = setUpdateOptions(positionOptions, updateOptions);
    if (KErrNone != error)
    {
        //Populate error code Map
        retMap[KErrCode]= error;
        retMap[KErrMsg] = KUnKErr;
        retMap[KTransId] = KInvalidTransId;
        return retMap;
    }

    //Gets a new transaction id by incrementing counter
    qint32 transId = ++m_transIdCount;
    LocationEngine* engine = LocationEngine::getInstance(this);

    if (updateOptions.contains(KMaxAgeOptionUpdated))
    {
        if (engine->getPositionInfoSource())
        {
            QGeoPositionInfo posInfo;
            posInfo = engine->getPositionInfoSource()->lastKnownPosition(false);
            if (checkMaxAge(posInfo, updateOptions.value(KMaxAgeOptionUpdated)))
            {
                QMap<QString, QVariant> data;
                fillData(posInfo, data);
                asyncTask* task = new asyncTask(KErrNone, transId, data);
                connect(task, SIGNAL(asyncCB(int,qint32,QMap<QString,QVariant>)),
                        this, SLOT(asyncCB(int,qint32,QMap<QString,QVariant>)));
                QThreadPool::globalInstance()->start(task);

                retMap.insert(KErrCode, KErrNone);
                retMap.insert(KTransId, transId);
                return retMap;
            }
        }
    }

    if (engine->getPositionInfoSource())
    {
        engine->getPositionInfoSource()->requestUpdate();
        engine->getPositionInfoSource()->startUpdates();
        watchCallsList.append(transId);
        timeObserver* timeObs = NULL;
        if (updateOptions.contains(KTimeOutOptionUpdated))
        {
            timeObs = new timeObserver(transId, updateOptions.value(KTimeOutOptionUpdated));
        }
        else
        {
            timeObs = new timeObserver(transId, KTimeInterval);
        }
        if (timeObs)
        {
            watchTimersList.insert(transId, timeObs);
            connect(timeObs, SIGNAL(timeOutCB(qint32)),
                    this, SLOT(timerTimeout(qint32)));
            timeObs->run();
        }

        retMap.insert(KErrCode, KErrNone);
        retMap.insert(KTransId, transId);
        return retMap;
    }
    else
    {
        retMap.insert(KErrCode, KErrGeneral);
        retMap.insert(KTransId, transId);
        return retMap;
    }
}

QVariant LocationProvider::clearWatch(const int watchId)
{
    QMap<QString,QVariant> errorMap;

    if (watchCallsList.removeOne(watchId))
    {
        if (watchTimersList.contains(watchId))
        {
            if (watchTimersList.value(watchId))
            {
                disconnect(watchTimersList.value(watchId), 0, 0, 0);
                delete watchTimersList.take(watchId);
            }
        }
    }
    else
    {
        errorMap.insert(KErrCode, KErrArgument);
        errorMap.insert(KErrMsg, KStrErrInvalidTransID);
        return errorMap;
    }

    if (watchCallsList.isEmpty())
    {
        LocationEngine* engine = LocationEngine::getInstance(this);
        if (engine->getPositionInfoSource())
        {
            engine->getPositionInfoSource()->stopUpdates();
        }
    }

    return KErrNone;
}

QVariant LocationProvider::getLocationUsingMethodName(QString methodName,int trId)
{
    Q_UNUSED(methodName)
    Q_UNUSED(trId)

    qDebug() << "method getLocationUsingMethodName not supported";

    QMap<QString,QVariant> retMap;

    //Gets a new transaction id by incrementing counter
    qint32 transId = ++m_transIdCount;

    retMap.insert(KErrCode,KErrNone);
    retMap.insert(KTransId,transId);
    return retMap;
}

int LocationProvider::setUpdateOptions(const QMap<QString,QVariant>& positionOptions,
                                       QMap<QString,QVariant>& updateOpts)
{
    bool enableAccuracy = false;

    if (positionOptions.contains(KEnableHighAccOption))
    {
        if (hasData(positionOptions.value(KEnableHighAccOption)))
        {
            if ((positionOptions.value(KEnableHighAccOption)).type() == QVariant::Bool) {
                enableAccuracy = (positionOptions.value(KEnableHighAccOption)).toBool();
            }
            else
            {
                return KErrArgument;
            }
        }
    }

    if (positionOptions.contains(KTimeOutOption))
    {
        if (hasData(positionOptions.value(KTimeOutOption)))
        {
            if ((positionOptions.value(KTimeOutOption)).type() == QVariant::Double)
            {
                if (positionOptions.value(KTimeOutOption).canConvert(QVariant::LongLong))
                {
                    long long timeOut = positionOptions.value(KTimeOutOption).toLongLong();
                    if (timeOut >= 0)
                    {
                        updateOpts.insert(KTimeOutOptionUpdated, timeOut);
                    }
                }
                else
                {
                    return KErrArgument;
                }
            }
            else
            {
                return KErrArgument;
            }
        }
    }

    if (positionOptions.contains(KMaxAgeOption))
    {
        if (hasData(positionOptions.value(KMaxAgeOption)))
        {
            if ((positionOptions.value(KMaxAgeOption)).type() == QVariant::Double)
            {
                if (positionOptions.value(KMaxAgeOption).canConvert(QVariant::LongLong))
                {
                    long long maxAge = positionOptions.value(KMaxAgeOption).toLongLong();
                    if (maxAge >= 0)
                    {
                        updateOpts.insert(KMaxAgeOptionUpdated, maxAge);
                    }
                }
                else
                {
                    return KErrArgument;
                }
            }
            else
            {
                return KErrArgument;
            }
        }
    }
    return KErrNone;
}

bool LocationProvider::hasData(QVariant dataValue)
{
    if (dataValue.isNull())
    {
        return false;
    }
    else if ((dataValue.type() == QVariant::String)
        && (dataValue.toString().isEmpty()))
    {
        return false;
    }
    else
    {
        return true;
    }
}

void LocationProvider::fillData(const QGeoPositionInfo &posInfo, QMap<QString, QVariant> &data)
{
    QGeoCoordinate coordinate = posInfo.coordinate();

    data.insert(KLatitude, coordinate.latitude());
    data.insert(KLongitude, coordinate.longitude());
    data.insert(KAltitude, coordinate.altitude());
    if (posInfo.timestamp().isValid())
    {
        data.insert(KTime, posInfo.timestamp());
    }
    else
    {
        data.insert(KTime, QDateTime::currentDateTime());
    }
    if (posInfo.hasAttribute(QGeoPositionInfo::HorizontalAccuracy))
    {
        data.insert(KAccuracy, posInfo.attribute(QGeoPositionInfo::HorizontalAccuracy));
    }
    else
    {
        data.insert(KAccuracy, NULL);
    }
    if (posInfo.hasAttribute(QGeoPositionInfo::VerticalAccuracy))
    {
        data.insert(KAltitudeAccuracy, posInfo.attribute(QGeoPositionInfo::VerticalAccuracy));
    }
    else
    {
        data.insert(KAltitudeAccuracy, NULL);
    }
    if (posInfo.hasAttribute(QGeoPositionInfo::Direction))
    {
        data.insert(KHeading, posInfo.attribute(QGeoPositionInfo::Direction));
    }
    else
    {
        data.insert(KHeading, NULL);
    }
    if (posInfo.hasAttribute(QGeoPositionInfo::GroundSpeed))
    {
        data.insert(KSpeed, posInfo.attribute(QGeoPositionInfo::GroundSpeed));
    }
    else
    {
        data.insert(KSpeed, NULL);
    }
}

void LocationProvider::fireSignalAll(int errStatus, QMap<QString, QVariant>& data)
{
    qDebug() << __FUNCTION__ << data;

    foreach(qint32 id, getPosCallsList)
    {
        getPosCallsList.removeOne(id);
        if (getPosTimersList.contains(id))
        {
            if (getPosTimersList.value(id))
            {
                disconnect(getPosTimersList.value(id), 0, 0, 0);
                delete getPosTimersList.take(id);
            }
        }
        emit AsyncCallback(errStatus, id, data);
    }
    foreach(qint32 id, watchCallsList)
    {
        if (watchTimersList.contains(id))
        {
            watchTimersList.value(id)->run();
        }
        emit AsyncCallback(errStatus, id, data);
    }
}

void LocationProvider::fireSignal(int errStatus, QMap<QString, QVariant> &data, qint32 transId)
{
    qDebug() << __FUNCTION__ << data;

    if (getPosCallsList.contains(transId))
    {
        getPosCallsList.removeOne(transId);
        if (getPosTimersList.contains(transId))
        {
            if (getPosTimersList.value(transId))
            {
                disconnect(getPosTimersList.value(transId), 0, 0, 0);
                delete getPosTimersList.take(transId);
            }
        }
    }
    if (watchCallsList.contains(transId))
    {
        if (watchTimersList.contains(transId))
        {
            watchTimersList.value(transId)->run();
        }
    }
    emit AsyncCallback(errStatus, transId, data);
}

bool LocationProvider::checkMaxAge(QGeoPositionInfo& posInfo, QVariant maxAge)
{
    if (posInfo.isValid())
    {
        QDateTime timestamp = posInfo.timestamp();
        quint32 diff = timestamp.secsTo(QDateTime::currentDateTime()) * KTimeConversion;
        if (diff < maxAge.toLongLong())
        {
            return true;
        }
        return false;
    }
    return false;
}

void LocationProvider::posUpdated(const QGeoPositionInfo& posInfo)
{
    QMap<QString, QVariant> data;
    int error;

    qDebug() << "Position Information" << posInfo;

    if (posInfo.coordinate().isValid())
    {
        qDebug() << "Position is valid";
        error = KErrNone;
        fillData(posInfo, data);
    }
    else
    {
        qDebug() << "Position is not valid";
        error = KErrGeneral;
        //create errorPosition for ErrorCallback
        data.insert(KErrCode, error);
        data.insert(KErrMsg, "Received position is not valid");
    }
    fireSignalAll(error, data);
}


void LocationProvider::timerTimeout(qint32 aId)
{
    //create errorPosition for ErrorCallback
    QMap<QString, QVariant> data;
    data.insert(KErrCode, KErrTimeOut);
    data.insert(KErrMsg, "Time out");
    fireSignal(KErrTimeOut, data, aId);
}

void LocationProvider::asyncCB(int errStatus, qint32 aId, QMap<QString, QVariant> data)
{
    fireSignal(errStatus, data, aId);
}
