/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef FILESYSTEM_WORKER_HH
#define FILESYSTEM_WORKER_HH

//INCLUDES
#include <QThread>
#include <QMap>
#include <QVariant>
#include "fsconstants.h"

//FORWARD DECLARATION
class IMountPoint;

// CLASS DECLARATION

/**
 *  This class performs the filesystem operations
 *  @lib filesytemserviceplugin.qtplugin
 *  @since x.y
 */
class FileSystemWorker : public QThread {

Q_OBJECT

public:
    //constrcutors
    FileSystemWorker(int taskId, QString srcUri, int callbackId,QHash<QString,QString> mountTable);

    FileSystemWorker(int taskId, QString srcUri,
            QString matchPattern, int callbackId,QHash<QString,QString> mountTable);

    FileSystemWorker(int taskId, QString srcUri,
            bool option, int callbackId,QHash<QString,QString> mountTable );

    FileSystemWorker(int taskId, QString srcUri, QString destUri,
            QString newName, bool overWrite,
            int callbackId,QHash<QString,QString> mountTable);

    FileSystemWorker(int taskId, QString srcUri,
            QString mode, QString encoding, QList<FileInfo>* fileHandleList,
            int callbackId, QHash<QString,QString> mountTable);

    FileSystemWorker(int taskId, const QMap<QString,QVariant>& matchObject,
            int callbackId, QHash<QString,QString> mountTable);

    //Destructor
    virtual ~FileSystemWorker();

    //from base
    virtual void run();

    //public functions used for notification & cancel
    void run_cancelNotify();
    void newMountPointEvent(QMap<QString,QVariant> mountPoint);

    signals:
    //Signal to notify the service class
    void FileSystemWorkerSignal(int task, int tid, QMap<QString,QVariant> retMap);

private:
    //methods the perform the file system operations
    void run_createDir();
    void run_remove();
    void run_rename();
    void run_getDirContents();
    void run_copy(bool removeAfterCopy);
    void run_openFile();
    void run_search();
    void run_getMountPoints();
    void run_getDefaultPath();
    void run_notifyMountEvents();
    void run_getElementInfo();
    void run_searchMatchObject();

    //to extract the path from given uri
    bool ExtractPathFromUri(QString &Uri);

    
    QString chkIfAllowed(QString uri);
    
        
    //Checks if the consumer has permissions to access the uri
    QString validatePath(QString Uri);

    //Called if user hasn't specified paths for search API
    QStringList defaultSearchPaths();

    //removes the given directory recursively
    bool remove_Dir_Recursive(QString path);

    //gets the file open mode
    bool getFileOpenMode(QString mode,bool fileExists, QIODevice::OpenMode &modeVal);

    //copies file based on overwrite parameter
    bool copy_file(QString SrcUri, QString DestUri, bool OverWrite, bool removeAfterCopy,\
            int& errCode, QString& errMessage);

    //copied directory based on overWrite parameter
    bool copy_dir(QString SrcUri, QString DestUri, bool OverWrite, bool removeAfterCopy,\
            QString currDir,\
            int& errCode, QString& errMessage);

    //finds the matching contents in the currDir
    void FindInDir(QString currDir,QStringList matchStr, bool lastDir);

    //current task going on
    int m_task;

    //source uri
    QString m_srcUri;

    //match pattern to be used
    QString m_matchPattern;

    //encoding format that needs to be used
    QString m_encoding;

    //overwrite option
    bool m_option;

    //destination uri
    QString m_destUri;

    //call back id of current async call
    int m_cbId;

    /**
     *  Map to maintain open file handles
     */
    QList<FileInfo>* m_openFileList;

    //notifier of platform sepecific events
    IMountPoint* m_PlatformNotifier;

    //match object for search
    QMap<QString,QVariant> m_MatchObject;

    QHash<QString,QString> m_mountTable;
};

#endif
