/*
 * ============================================================================
 *  Name        : filesysteminterface.h
 *  Part of     : serviceframework / WRT
 *  Description : Interface for file system
 *  Version     : %version: 8 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef FILESYSTEMINTERFACE_H
#define FILESYSTEMINTERFACE_H

//INCLUDES
#include <QObject>
#include "servicebase.h"

static const char* KIFileSystemId = "com.nokia.IFileSystem/1.0";

// ISimpleTypeTest interface id
static const ServiceInterfaceInfo KIFileSystemInfo =
{
        KIFileSystemId, 0
};

// CLASS DECLARATION

/**
 *  The class defines the interface for filesystem methods.
 *  @lib filesystemserviceplugin.qtplugin
 *  @since x.y
 */

class IFileSystem : public IServiceBase
    {
public:
    /**
     * Interface function for createDir.
     * creates the directory with the user specified path
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param srcUri - path to be created.
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant createDir(const QString& srcUri, int callBackId = -1) = 0;

    /**
     * Interface function for remove.
     * removes the user specified path
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param path - path to be removed.
     * @param recursive - remove recursively?.
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant remove(const QString& path, bool recursive = false, int callBackId = -1) = 0;

    /**
     * Interface function for rename.
     * renames the user specified path
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param path - path to be renamed.
     * @param newName - new name proposed.
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant rename(const QString& path, const QString& newName, int callBackId = -1) = 0;
    /**
     * Interface function for getDirContents.
     * Returns the contents of a directory based on match pattern
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param srcUri - source directory path
     * @param matchPattern - pattern to be matched
     * @param callBackId - call back id.
     * @return Returns map with error code & message, data in sync mode
     * or trans id in async mode.
     */
    virtual QVariant getDirContents(const QString& srcUri, const QString& matchPattern = "*", int callBackId = -1) = 0;

    /**
     * Interface function for copy.
     * copies a file or directory from source to destination
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param srcUri - source path
     * @param destUri - desitnation path
     * @param newName - new Name provided
     * @param overWrite - overwrite if destination exists
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant copy(const QString& srcUri, const QString& destUri, QString newName = NULL,
            bool overWrite = false, int callBackId = -1) = 0;

    /**
     * Interface function for move.
     * moves a file or directory from source to destination
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param srcUri - source path
     * @param destUri - desitnation path
     * @param newName - new Name provided
     * @param overWrite - overwrite if destination exists
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant move(const QString& srcUri, const QString& destUri, QString newName = NULL,
            bool overWrite = false, int callBackId = -1) = 0;

    /**
     * Interface function for search.
     * Returns the searched items based on match pattern and input path.
     * This is an asycnhronous method.
     * @since x.y
     * @param callBackId - call back id.
     * @param matchPattern - pattern to be matched
     * @param srcUri - source directory path
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant search(int callBackId,const QString& matchPattern, const QString& srcUri) = 0;

    /**
     * Interface function for getMountPoints.
     * Returns the available Mount points on the device
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant getMountPoints(int callBackId = -1) = 0;

    /**
     * Interface function for getDefaultPath.
     * Returns the available default paths on the device
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param contentType - content type
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant getDefaultPath(const QString& contentType, int callBackId = -1) = 0;

    /**
     * Interface function for notifyMountEvents
     * Notifies about the newly available mount point.
     * This acts as asynchronous notification method.
     * @since x.y
     * @param callBackId - call back id.
     * @return Returns dummy transaction id.
     */
    virtual QVariant notifyMountEvents(int callBackId) = 0;

    /**
     * Interface function for cancelNotify.
     * cancels the notification of mount events.
     * This acts as both synchronous method.
     * @since x.y
     * @return Returns map with error code & message
     */
    virtual QVariant cancelNotify() = 0;

    /**
     * Interface function for openFile.
     * Returns the file handle
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param uri - source file path
     * @param mode - mode in which the file is to be opened
     * @param encoding - encoding format
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant openFile(const QString& uri, const QString& mode, const QString& encoding, int callBackId = -1) = 0;

    /**
     * Interface function for close.
     * closes the file handle
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param handleId - handle id of the file
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant close(int handleId, int callBackId = -1) = 0;

    /**
     * Interface function for read.
     * Reads data of size maxLength from the position mentioned
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param handleId - handle id of the file
     * @param maxLength - maximum number of characters to be read
     * @param pos - position in the file from where to read
     * @param callBackId - call back id.
     * @return Returns map with error code & data in sync mode
     * or trans id in async mode.
     */
    virtual QVariant read(int handleId, int maxLength, int pos = -1, int callBackId = -1) = 0;

    /**
     * Interface function for readLine.
     * Reads data till a new line is encountered
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param handleId - handle id of the file
     * @param maxLength - maximum number of characters to be read
     * @param callBackId - call back id.
     * @return Returns map with error code & data in sync mode
     * or trans id in async mode.
     */
    virtual QVariant readLine(int handleId, int maxLength = 0, int callBackId = -1) = 0;

    /**
     * Interface function for readBase64.
     * Reads data from the file and returns it in Base64 encoded format
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param handleId - handle id of the file
     * @param maxLength - maximum number of characters to be read
     * @param callBackId - call back id.
     * @return Returns map with error code & data in sync mode
     * or trans id in async mode.
     */
    virtual QVariant readBase64(int handleId, int maxLength, int callBackId = -1) = 0;

    /**
     * Interface function for write..
     * writes the given data in the file
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param handleId - handle id of the file
     * @param data - data to be written in the file
     * @param pos - position in the file to write
     * @param callBackId - call back id.
     * @return Returns map with error code & data in sync mode
     * or trans id in async mode.
     */
    virtual QVariant write(int handleId, const QString& data, int pos = -1, int callBackId = -1) = 0;

    /**
     * Interface function for writeLine.
     * writes the given data along with a newline character at the end
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param handleId - handle id of the file
     * @param data - data to be written in the file
     * @param callBackId - call back id.
     * @return Returns map with error code & data in sync mode
     * or trans id in async mode.
     */
    virtual QVariant writeLine(int handleId, const QString& data, int callBackId = -1) = 0;
    /**
     * Interface function for writeBase64.
     * writes the Base64 encoded data in the file
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param handleId - handle id of the file
     * @param data - data to be written in the file
     * @param callBackId - call back id.
     * @return Returns map void in sync mode
     * or trans id in async mode.
     */
    virtual QVariant writeBase64(int handleId, const QString& data, int callBackId = -1) = 0;

    /**
     * Interface function for flush.
     * Performs a flush on a file handle
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param handleId - handle id of the file
     * @param callBackId - call back id.
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant flush(int handleId, int callBackId = -1) = 0;

    /**
     * Interface function for seek.
     * Moves the postion of file handle
     * This is a synchronous method.
     * @since x.y
     * @param handleId - handle id of the file
     * @param seekOption - seekoption shoule be one of SEEKSTART, SEEKCURRENT, SEEKEND
     * @param position - position to be moved.
     * @return Returns map with error code & message along with position
     */
    virtual QVariant seek(int handleId, int seekOption, int position = 0) = 0;

    /* Interface function for cancel.
     * cancels any ongoing async operation based on transaction id
     * This is a synchronous method.
     * @since x.y
     * @param transId - Transaction Id.
     * @return Returns map with error code & message.
     */
    virtual QVariant cancel(int transId) = 0;

    /**
     * Interface function for getElementInfo.
     * Returns the properties of an element
     * This acts as both synchronous & asycnhronous method based on callback id.
     * @since x.y
     * @param srcUri - source element path
     * @param callBackId - call back id.
     * @return Returns map with error code & message, data in sync mode
     * or trans id in async mode.
     */
    virtual QVariant getElementInfo(const QString& srcUri, int callBackId = -1) = 0;

    /**
     * Interface function for search.
     * Returns the searched items based on match object.
     * This is an asycnhronous method.
     * @since x.y
     * @param callBackId - call back id.
     * @param matchObject - may contain <Date> createDate, <String> fileName,
     *                      <String> fileName, <Number> fileSize, <boolean> isDirectory,
     *                      <Date> lastModifyDate
     * @return Returns map with error code & message in sync mode
     * or trans id in async mode.
     */
    virtual QVariant searchMatchObject(int callBackId, const QMap<QString,QVariant>& matchObject) = 0;    
    
    };
Q_DECLARE_INTERFACE(IFileSystem, KIFileSystemId);


#endif //FILESYSTEMINTERFACE_H
