/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QtPlugin>
#include <QTimer>
#include <QString>
#include <QStringList>
#include <QThreadPool>
#include <QFile>
#include <QTime>
#include <QProcess>

#ifdef __MAEMO5__
#include <QDBusInterface>
#include <QDBusConnection>
#endif

#include "qcontactservice.h"
#include "qcontactsiterator.h"
#include "qgroupsiterator.h"

#include "qcontactshelpers.h"
#include "qjsonizer.h"
#include "secsession.h"


QContactService::QContactService()
{
    qRegisterMetaType<QList<QContact> >("QList<QContact>");
    qRegisterMetaType<QList<QContactLocalId> >("QList<QContactLocalId>");
    m_isNotifyInProgress = false;
    m_notifyTrId = -1;
}

QContactService::~QContactService()
{

}

void QContactService::setSecuritySession(WRT::SecSession *aSecSession)
{
    iSecSession = aSecSession;
}

QVariant QContactService::getContactsAsync (const QString& match, const int sortOrder,
                                                                    const int transId)
{
    m_onGoingAsyncTransctions.append(transId);
    GetContactsTask * task = new GetContactsTask(match, sortOrder, transId);

    connect(task, SIGNAL(contactObjects(GetContactsTask*, QList<QContact>, qint32, qint32)), this,
            SLOT(fireJavaScriptCallback(GetContactsTask*, QList<QContact>, qint32, qint32)));
    QThreadPool::globalInstance()->start(task);

    // Error code is present in fireJavaScriptCallback
    QMap<QString,QVariant> data;
    QVariant retData;
    data.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    data.insert("ErrorMessage","");
    retData = data;

    return retData;
}

QVariant QContactService::getContactIdsAsync ( const QString& match , const int sortOrder ,
                                                                        const int transId )
{
    //qDebug() << Q_FUNC_INFO;
    m_onGoingAsyncTransctions.append(transId);

    // Run getContactIds in a thread
    GetContactIdsTask * task = new GetContactIdsTask(match, sortOrder,
            transId);
    connect(task, SIGNAL(contactIds(GetContactIdsTask*, QList<QContactLocalId>, qint32, qint32)), this,
            SLOT(fireJavaScriptStringListCallback(GetContactIdsTask*, QList<QContactLocalId>, qint32, qint32)));
    QThreadPool::globalInstance()->start(task);

    // Error code is present in fireJavaScriptCallback
    QMap<QString,QVariant> data;
    QVariant retData;
    data.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    data.insert("ErrorMessage","");
    retData = data;

    return retData;
}

QVariant QContactService::getContactIds ( const QString& match , const int sortOrder , const int /*transId*/ )
{
    // Utilizing default manager and retrieving contacts synchronously
    QContactManager* cm = new QContactManager();
    QList<QContactLocalId> contactList;
    QList<QContactSortOrder> sortOrders;

    QContactSortOrder order1, order2;
    order1.setDetailDefinitionName(QContactName::DefinitionName, QContactName::FieldLastName);
    order2.setDetailDefinitionName(QContactName::DefinitionName, QContactName::FieldFirstName);

    if (sortOrder == 0) {
        order1.setDirection(Qt::AscendingOrder);
    } else if (sortOrder == 1) {
        order1.setDirection(Qt::DescendingOrder);
    }
    sortOrders << order1 << order2;

    if (!match.isEmpty()) {
        QContactUnionFilter union_filter;
        QContactDetailFilter filter1, filter2;
        filter1.setDetailDefinitionName(QContactName::DefinitionName, QContactName::FieldLastName);
        filter2.setDetailDefinitionName(QContactName::DefinitionName, QContactName::FieldFirstName);
        filter1.setMatchFlags(QContactFilter::MatchContains);
        filter2.setMatchFlags(QContactFilter::MatchContains);
        filter1.setValue(match);
        filter2.setValue(match);

        union_filter << filter1 << filter2;

        contactList = cm->contactIds(union_filter, sortOrders);
    } else {
        contactList = cm->contactIds(sortOrders);
    }

    QMap<QString,QVariant> data;
    QVariant retData;

    QList<QVariant> idList;
    QVariant listVar = NULL;

    for (int i=0 ; i < contactList.count() ; i++) {
        QVariant idVar(contactList.at(i));
        idList.append(idVar);
    }

    if (idList.count() > 0) {
        listVar = idList;
        data.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
        data.insert("ErrorMessage","SUCCESS");
    } else {
        data.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        data.insert("ErrorMessage","DATA_NOT_FOUND_ERR");
    }

    data.insert("ReturnValue", listVar);
    retData = data;

    return retData;
}

QVariant QContactService::getContactInfo ( const QString& contactId )
{

    QVariant retVar = NULL;
    QMap<QString,QVariant> rMap;
    JSonizer conv;
    QVariant var = NULL;
    QContactManager* cm = new QContactManager();

    bool ok;
    QContactLocalId id(contactId.toUInt(&ok));
    QContact contact = cm->contact(id);

    if (!ok) {
        //qDebug() << "Error: No such contact " << contactId;

        rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert("ErrorMessage","Contact Not Found");
        rMap.insert("ReturnValue",var);

        retVar = rMap;
        return retVar;
    }

    var = conv.contact2json(&contact);

    rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    rMap.insert("ErrorMessage","SUCCESS");
    rMap.insert("ReturnValue",var);

    retVar = rMap;

    return retVar;
}

QVariant QContactService::addContact (const QMap<QString, QVariant>& contactData,const int /*transId*/)
{
    //qDebug() << "::addContact :" << contactData;
    QVariant retVar = NULL;
    QMap<QString,QVariant> rMap;
    QContact newContact;
    ContactsHelpers::setContactFields(&newContact, contactData);

    QContactManager* cm = new QContactManager();
    bool saved = cm->saveContact(&newContact);

    if (!saved) {
        //qDebug() << "Saving contact failed!";
        rMap.insert("ErrorCode",CJSEStatusCodes::SERVICE_IN_USE_ERR);
        rMap.insert("ErrorMessage","Service in use");
        rMap.insert("ReturnValue",NULL);
        retVar = rMap;
    } else {
        //QContactName name = newContact.detail(QContactName::DefinitionName);
        //qDebug() << "Saving contact succeeded, id: " << newContact.id().localId() << " name.lastName: " << name.lastName();
        //QContactPhoneNumber tel = newContact.detail(QContactPhoneNumber::DefinitionName);
        //qDebug() << "phonenumber: " << tel.number();
        rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
        rMap.insert("ErrorMessage","SUCCESS");
        rMap.insert("ReturnValue",newContact.id().localId());
        retVar = rMap;
    }

    return retVar;

}

QVariant QContactService::updateContact (const QMap<QString , QVariant>& qcontact,const int /*transId*/)
{
    QVariant retVar = NULL;

    if (qcontact.value("id").toString().length() > 0) {
        //qDebug() << "::updateContact :" << qcontact;
        retVar=this->addContact(qcontact);
    }

    return retVar;
}

QVariant QContactService::deleteContacts (const QVariant& contactIdsvar)
{
    QList<QVariant> contactIds = contactIdsvar.toList();
    QVariant retVar = NULL;
    QVariant retValue = NULL;
    QMap<QString,QVariant> rMap;

    if (contactIds.length() > 0) {
        rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
        rMap.insert("ErrorMessage","SUCCESS");
        rMap.insert("ReturnValue",retValue);
        retVar = rMap;
    } else {
        rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert("ErrorMessage","Contact(s) not found");
        rMap.insert("ReturnValue",retValue);
        retVar = rMap;
    }

    QContactManager* cm = new QContactManager();

    for (int index = 0; index < contactIds.count(); index++) {
        QContactLocalId id = contactIds.at(index).toUInt();

        bool ok = cm->removeContact(id);

        if (!ok) {
            retValue = id;
            rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
            rMap.insert("ErrorMessage","Contact not found");
            rMap.insert("ReturnValue",retValue);
            retVar = rMap;
            break;
        }
    }

    return retVar;
}

QVariant QContactService::addGroup ( const QString& groupName )
{
    QContactManager* cm = new QContactManager();
    QContact group;
    group.setType(QContactType::TypeGroup);
    QVariant retVar = NULL;
    QVariant retValue = NULL;
    QMap<QString,QVariant> rMap;

    // Check if a group with groupName already exist
    QContactDetailFilter groupFilter;
    groupFilter.setDetailDefinitionName(QContactType::DefinitionName, QContactType::FieldType);
    groupFilter.setValue(QString(QLatin1String(QContactType::TypeGroup)));

    QList<QContactLocalId> groupList = cm->contactIds(groupFilter);
    for (int index=0; index < groupList.count(); index++){
        QContact tmp_grp = cm->contact(groupList[index]);
        if (tmp_grp.displayLabel().compare(groupName) == 0) {
            rMap.insert("ErrorCode",CJSEStatusCodes::DATA_ALREADY_EXISTS_ERR);
            rMap.insert("ErrorMessage","A group with this name already exist!");
            rMap.insert("ReturnValue",retValue);
            retVar = rMap;

            return retVar;
        }
    }

    QContactName grpName;
    grpName.setCustomLabel(groupName);
    group.saveDetail(&grpName);

    bool ok = cm->saveContact(&group);
    if (ok) {
        retValue = group.id().localId();
        rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
        rMap.insert("ErrorMessage","Success");
        rMap.insert("ReturnValue",retValue);
        retVar = rMap;

    } else {
        rMap.insert("ErrorCode",CJSEStatusCodes::INVALID_ARG_ERR);
        rMap.insert("ErrorMessage","Could not save the group");
        rMap.insert("ReturnValue",retValue);
        retVar = rMap;
    }

    return retVar;
}

QVariant QContactService::deleteGroups ( const QVariant& groupIds )
{
    QList<QVariant> groupIdList = groupIds.toList();
    QVariant retVar = NULL;
    QVariant retValue = NULL;
    QMap<QString,QVariant> rMap;

    if (groupIdList.length() > 0) {
        QContactManager* cm = new QContactManager();

        for (int index = 0; index < groupIdList.count(); index++) {
            QContactLocalId id = groupIdList.at(index).toUInt();

            bool ok = cm->removeContact(id);

            if (!ok) {
                rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
                rMap.insert("ErrorMessage","Group ID not found");
                rMap.insert("ReturnValue",id);
                retVar = rMap;

                return retVar;
            }
        }
    } else {
        rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert("ErrorMessage","Group IDs not found");
        rMap.insert("ReturnValue",retValue);
        retVar = rMap;

        return retVar;
    }


    rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    rMap.insert("ErrorMessage","Success");
    rMap.insert("ReturnValue",retValue);
    retVar = rMap;

    return retVar;
}

QVariant QContactService::updateGroup ( const QMap<QString, QVariant>& groupData )
{
    QVariant retVar = NULL;
    QVariant retValue = NULL;
    QMap<QString,QVariant> rMap;

    QString groupIdentifier = groupData["groupId"].toString();
    QString groupName = groupData["groupName"].toString();

    if (groupIdentifier.length() <= 0) {
        rMap.insert("ErrorCode",CJSEStatusCodes::SERVICE_IN_USE_ERR);
        rMap.insert("ErrorMessage","Group ID Missing");
        rMap.insert("ReturnValue",retValue);
        retVar = rMap;
    } else if (groupName.length() <=0  ) {
        rMap.insert("ErrorCode",CJSEStatusCodes::SERVICE_IN_USE_ERR);
        rMap.insert("ErrorMessage","Group Name Missing");
        rMap.insert("ReturnValue",retValue);
        retVar = rMap;
    } else {
        QContactManager* cm = new QContactManager();
        QContact group;
        group.setType(QContactType::TypeGroup);
        QContactLocalId groupId = groupData["groupId"].toUInt();

        QContactId grpId;
        grpId.setLocalId(groupId);

        group.setId(grpId);
        QContactDisplayLabel groupLabel;
        groupLabel.setValue(QContactDisplayLabel::FieldLabel, groupName);
        group.saveDetail(&groupLabel);
        //group.setDisplayLabel(groupName);

        bool ok = cm->saveContact(&group);

        if (ok) {
            rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
            rMap.insert("ErrorMessage","Success");
            rMap.insert("ReturnValue",retValue);
            retVar = rMap;

        } else {
            rMap.insert("ErrorCode",CJSEStatusCodes::SERVICE_IN_USE_ERR);
            rMap.insert("ErrorMessage","Could not update the group");
            rMap.insert("ReturnValue",retValue);
            retVar = rMap;
        }
    }

    return retVar;
}

QVariant QContactService::addContactsToGroup ( const QString& groupId, const QVariant& contactIDs )
{
    QList<QVariant> contactIds = contactIDs.toList();
    QVariant retVar = NULL;
    QVariant retValue = NULL;
    QMap<QString,QVariant> rMap;

    if (!contactIds.isEmpty() && (groupId.length() > 0) ) {
        QContactManager* cm = new QContactManager();

        bool ok;
        QContactLocalId id(groupId.toUInt(&ok));

        if (ok) {
            QContact group = cm->contact(id);

            for (int index = 0; index < contactIds.count(); index++) {
                QContactLocalId id = contactIds.at(index).toUInt();

                QContactRelationship relShip;
                relShip.setRelationshipType(QContactRelationship::HasMember);

                relShip.setFirst(group.id());
                QContactId cId;
                cId.setLocalId(id);
                relShip.setSecond(cId);

                // Save relationship
                bool added = cm->saveRelationship(&relShip);
                if (!added) {
                    rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
                    rMap.insert("ErrorMessage","Could not add a member");
                    retValue = id;
                    rMap.insert("ReturnValue",retValue);
                    retVar = rMap;

                    return retVar;
                }
            }
        } else {
            rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
            rMap.insert("ErrorMessage","Group id not found");
            rMap.insert("ReturnValue",retValue);
            retVar = rMap;

            return retVar;
        }
    } else {
        rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert("ErrorMessage","Group id not found or no contacts to add");
        rMap.insert("ReturnValue",retValue);
        retVar = rMap;

        return retVar;
    }


    rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    rMap.insert("ErrorMessage","Success");
    rMap.insert("ReturnValue",retValue);
    retVar = rMap;

    return retVar;
}

QVariant QContactService::removeContactsFromGroup ( const QString& groupId, const QVariant& contactIDs )
{
    QList<QVariant> contactIds = contactIDs.toList();
    QVariant retVar = NULL;
    QVariant retValue = NULL;
    QMap<QString,QVariant> rMap;

    if (!contactIds.isEmpty() && (groupId.length() > 0) ) {
        QContactManager* cm = new QContactManager();

        bool ok;
        QContactLocalId groupLocalId(groupId.toUInt(&ok));

        if (ok) {
            QContact group = cm->contact(groupLocalId);
            QList<QVariant> errContacts;

            // Retrieve relationships for this group
            QList<QContactRelationship> groupRels =
                cm->relationships(QContactRelationship::HasMember, group.id());

            for (int index = 0; index < contactIds.count(); index++) {
                bool removed = false;
                QContactLocalId id = contactIds.at(index).toUInt();

                foreach (const QContactRelationship& cRel, groupRels) {
                    QContactId cId;
                    cId.setLocalId(id);
                    if (cRel.second() == cId) {
                        // Remove relationship
                        removed = cm->removeRelationship(cRel);

                        if (!removed) {
                            errContacts.append(id);
                        }
                        break;
                    }
                } // foreach
            } // for

            if (errContacts.count() > 0) {
                rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
                rMap.insert("ErrorMessage","Could not remove member(s)");
                // List of contacts not found
                retValue = errContacts;
                rMap.insert("ReturnValue",retValue);
                retVar = rMap;

                return retVar;
            }

        } else {
            rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
            rMap.insert("ErrorMessage","Group id not found");
            rMap.insert("ReturnValue",retValue);
            retVar = rMap;

            return retVar;
        }
    } else {
        rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert("ErrorMessage","Group id not found or no contacts to remove");
        rMap.insert("ReturnValue",retValue);
        retVar = rMap;

        return retVar;
    }


    rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    rMap.insert("ErrorMessage","Success");
    rMap.insert("ReturnValue",retValue);
    retVar = rMap;

    return retVar;
}

QVariant QContactService::getGroupsAsync (const int transId)
{
    m_onGoingAsyncTransctions.append(transId);

    // Run getContacts in a thread
    GetGroupsTask * task = new GetGroupsTask(transId);
    connect(task, SIGNAL(groupInfos(GetGroupsTask*, QList<QContact>, qint32, qint32)),
            this, SLOT(fireGroupsCallback(GetGroupsTask*, QList<QContact>, qint32, qint32)));

    QThreadPool::globalInstance()->start(task);

    // Error code is present in fireJavaScriptCallback
    QMap<QString,QVariant> data;
    QVariant retData;
    data.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    data.insert("ErrorMessage","");
    retData = data;

    return retData;
}

QVariant QContactService::getGroupIdsAsync (const int transId)
{
    //qDebug() << Q_FUNC_INFO;
    m_onGoingAsyncTransctions.append(transId);

    // Run getGroupIds in a thread
    GetGroupIdsTask * task = new GetGroupIdsTask(transId);
    connect(task, SIGNAL(groupIds(GetGroupIdsTask*, QList<QContactLocalId>, qint32, qint32)), this,
            SLOT(fireJavaScriptStringListCallback(GetGroupIdsTask*, QList<QContactLocalId>, qint32, qint32)));
    QThreadPool::globalInstance()->start(task);

    // Error code is present in fireJavaScriptCallback
    QMap<QString,QVariant> data;
    QVariant retData;
    data.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    data.insert("ErrorMessage","");
    retData = data;

    return retData;
}

QVariant QContactService::getGroupIds()
{
    // Utilizing default manager and retrieving contacts synchronously
    QContactManager* cm = new QContactManager();

    // Set filter to group type
    QContactDetailFilter groupFilter;
    groupFilter.setDetailDefinitionName(QContactType::DefinitionName, QContactType::FieldType);
    groupFilter.setValue(QString(QLatin1String(QContactType::TypeGroup)));

    QList<QContactLocalId> contactList = cm->contactIds(groupFilter);

    QMap<QString,QVariant> data;
    QVariant retData;

    QList<QVariant> idList;
    QVariant listVar = NULL;

    for (int i=0 ; i < contactList.count() ; i++) {
        QVariant idVar(contactList.at(i));
        idList.append(idVar);
    }

    if (idList.count() > 0) {
        listVar = idList;
        data.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
        data.insert("ErrorMessage","SUCCESS");
    } else {
        data.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        data.insert("ErrorMessage","DATA_NOT_FOUND_ERR");
    }

    data.insert("ReturnValue", listVar);
    retData = data;

    return retData;
}

QVariant QContactService::getGroupInfo ( const QString& groupId )
{
    QVariant retVar = NULL;
    QMap<QString,QVariant> rMap;
    QVariant var = NULL;
    QContactManager* cm = new QContactManager();

    bool ok;
    QContactLocalId id(groupId.toUInt(&ok));
    QContact group = cm->contact(id);

    if (!ok) {
        //qDebug() << "Error: No such group " << groupId;

        rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert("ErrorMessage","Group Not Found");
        rMap.insert("ReturnValue",var);

        retVar = rMap;
        return retVar;
    }

    var = ContactsHelpers::createGroupInfoJSON(&group);

    rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    rMap.insert("ErrorMessage","SUCCESS");
    rMap.insert("ReturnValue",var);

    retVar = rMap;

    return retVar;
}

QVariant QContactService::cancel( const int transactionID )
{

    QVariant retVar    = NULL;
    QMap<QString,QVariant> rMap;

    if (!m_onGoingAsyncTransctions.contains(transactionID)) {
        rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert("ErrorMessage","DATA_NOT_FOUND_ERR");
        rMap.insert("ReturnValue",1);

        retVar = rMap;

        return retVar;
    }

    rMap.insert("ErrorCode", CJSEStatusCodes::SUCCESS);
    rMap.insert("ErrorMessage","SUCCESS");
    rMap.insert("ReturnValue",1);

    m_onGoingAsyncTransctions.removeAll(transactionID);

    retVar = rMap;
    return retVar;
}

QVariant QContactService::startNotification(const int transId)
{
    QVariant retVar    = NULL;
    QMap<QString,QVariant> rMap;

    if (m_isNotifyInProgress) {
        rMap.insert("ErrorCode",CJSEStatusCodes::SERVICE_BUSY_ERR);
        rMap.insert("ErrorMessage","Notification already started!");
        rMap.insert("ReturnValue",NULL);

        retVar = rMap;
        return retVar;
    }

    QContactManager* cm = new QContactManager();

    connect(cm, SIGNAL(contactsAdded(QList<QContactLocalId>)), this,
        SLOT(handleContactsAdded(QList<QContactLocalId>)));
    connect(cm, SIGNAL(contactsChanged(QList<QContactLocalId>)), this,
        SLOT(handleContactsChanged(QList<QContactLocalId>)));
    connect(cm, SIGNAL(contactsRemoved(QList<QContactLocalId>)), this,
        SLOT(handleContactsRemoved(QList<QContactLocalId>)));

    m_notifyTrId = transId;
    m_isNotifyInProgress = true;
    rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
    rMap.insert("ErrorMessage","Success!");
    rMap.insert("ReturnValue",NULL);

    retVar = rMap;
    return retVar;
}

QVariant QContactService::stopNotification()
{
    QVariant retVar    = NULL;
    QMap<QString,QVariant> rMap;

    if (m_isNotifyInProgress) {
        m_notifyTrId = -1;
        m_isNotifyInProgress = false;
        // Disconnect signals to own slots
        QContactManager* cm = new QContactManager();

        disconnect(cm, SIGNAL(contactsAdded(QList<QContactLocalId>)), this,
            SLOT(handleContactsAdded(QList<QContactLocalId>)));
        disconnect(cm, SIGNAL(contactsChanged(QList<QContactLocalId>)), this,
            SLOT(handleContactsChanged(QList<QContactLocalId>)));
        disconnect(cm, SIGNAL(contactsRemoved(QList<QContactLocalId>)), this,
            SLOT(handleContactsRemoved(QList<QContactLocalId>)));

        rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
        rMap.insert("ErrorMessage","Notification already started!");
        rMap.insert("ReturnValue",NULL);

        retVar = rMap;
    } else {
        rMap.insert("ErrorCode",CJSEStatusCodes::DATA_NOT_FOUND_ERR);
        rMap.insert("ErrorMessage","No notification requests running!");
        rMap.insert("ReturnValue",NULL);
    }

    return retVar;
}

QVariant QContactService::startEditor(const QMap<QString, QVariant>& contactData)
{
    qint32 startEdId = -1;
    QVariant retVar = NULL;
    QMap<QString,QVariant> rMap;

    if (!contactData.isEmpty()) {
        if (contactData.value(QString("id")).toString().length() <= 0) {
            // Add new contact
            QVariant mapVariant = addContact(contactData);
            QMap<QString, QVariant> mapData = mapVariant.toMap();
            startEdId = mapData.value(QString("ReturnValue")).toInt();
        } else {
            startEdId = contactData.value(QString("id")).toInt();
        }

        if (startEdId > 0) {
            rMap.insert("ErrorCode",CJSEStatusCodes::SUCCESS);
            rMap.insert("ErrorMessage","Success!");
            rMap.insert("ReturnValue",NULL);
            retVar = rMap;
            // Open native contact card editor
            #ifndef __MAEMO5__
            ContactsInterface contactsIf;
            contactsIf.openContactCard(startEdId);
            #else
            QDBusConnection dbus = QDBusConnection::sessionBus();
            QDBusInterface *addressbook = new QDBusInterface("com.nokia.osso_addressbook", 
                        "/com/nokia/osso_addressbook", "com.nokia.osso_addressbook", dbus);
            addressbook->call("top_application", startEdId);
            #endif
        } else {
            rMap.insert("ErrorCode",CJSEStatusCodes::INVALID_ARG_ERR);
            rMap.insert("ErrorMessage","Could not resolve contact!");
            rMap.insert("ReturnValue",NULL);
            retVar = rMap;
        }
    } else {
        rMap.insert("ErrorCode",CJSEStatusCodes::MISSING_ARG_ERR);
        rMap.insert("ErrorMessage","Missing editor arguments!");
        rMap.insert("ReturnValue",NULL);
        retVar = rMap;
    }

    return retVar;
}

void QContactService::fireJavaScriptCallback(qint32 /*trId*/, qint32 /*status*/)
{

}

void QContactService::fireJavaScriptCallback(QStringList* json, qint32 trId,
                                                                qint32 status)
{

    if (m_onGoingAsyncTransctions.contains(trId)) {
        QList<QString> list(json->mid(0, -1));
        // Make iterator of returned contacts list
        QListIterator<QString> aObj( list );

        //emit signal
        emit asyncCallback( status, trId, static_cast<QVariant>(list));
    }
    delete json;

}

void QContactService::fireJavaScriptCallback(GetContactsTask* aTask,
                QList<QContact> json, qint32 trId, qint32 status)
{
    if (m_onGoingAsyncTransctions.contains(trId)) {
        if (status != CJSEStatusCodes::DATA_NOT_FOUND_ERR) {
            QList<QContact> list(json.mid(0, -1));
            // Make iterator of returned contacts list
            QListIterator<QContact> aIter(list);
            QContactsIterator *aQIter=new  QContactsIterator(list);
            if (aQIter) {
                aQIter->setParent(this);
            }
            //emit signal
            emit asyncCallbackC( status, trId, static_cast<QObject*>(aQIter));
        } else {
            emit asyncCallbackC( status, trId, NULL);
        }

    }
    //delete json;
    aTask->finishTask();
}

void QContactService::fireGroupsCallback(GetGroupsTask* aTask,
                QList<QContact> json, qint32 trId, qint32 status)
{
    if (m_onGoingAsyncTransctions.contains(trId)) {
        if (status != CJSEStatusCodes::DATA_NOT_FOUND_ERR) {
            QList<QContact> list(json.mid(0, -1));
            // Make iterator of returned contacts list
            QListIterator<QContact> aIter(list);
            QGroupsIterator *aQIter=new  QGroupsIterator(list);
            if (aQIter) {
                aQIter->setParent(this);
            }
            //emit signal
            emit asyncCallbackG( status, trId, static_cast<QObject*>(aQIter));
        } else {
            emit asyncCallbackG( status, trId, NULL);
        }
    }
    //delete json;
    aTask->finishTask();
}

void QContactService::fireJavaScriptStringListCallback(GetContactIdsTask* aTask,
                QList<QContactLocalId> json, qint32 trId, qint32 status)
{
    QList<QVariant> idList;
    QVariant listVar = NULL;

    for (int i=0 ; i<json.count() ;i++ ) {
        QVariant idVar(json.at(i));
        idList.append(idVar);
    }

    listVar=idList;

    emit asyncCallback( status, trId, listVar);

    //delete json;
    aTask->finishTask();
}

void QContactService::fireJavaScriptStringListCallback(GetGroupIdsTask* aTask,
                QList<QContactLocalId> json, qint32 trId, qint32 status)
{
    QList<QVariant> idList;
    QVariant listVar = NULL;

    for (int i=0 ; i<json.count() ;i++ ) {
        QVariant idVar(json.at(i));
        idList.append(idVar);
    }

    listVar=idList;

    emit asyncCallback( status, trId, listVar);

    //delete json;
    aTask->finishTask();
}

void QContactService::handleContactsAdded(const QList<QContactLocalId>& contactIds)
{
    QMap<QString,QVariant> notifyData;
    qint32 status = CJSEStatusCodes::SUCCESS;

    if (!contactIds.isEmpty()) {
        QContactManager* cm = new QContactManager();
        // Check contact type
        QContact addedCnt = cm->contact(contactIds.first());
        QContactName name = addedCnt.detail(QContactName::DefinitionName);

        if (addedCnt.type().compare(QString(QLatin1String(QContactType::TypeGroup))) == 0) {
            notifyData.insert("eventType", CJSENotifications::EVENT_GROUP_ADDED);
        } else {
            notifyData.insert("eventType", CJSENotifications::EVENT_CONTACT_ADDED);
        }
        notifyData.insert("id", QString(contactIds.first()));
    } else {
        status = CJSEStatusCodes::INVALID_ARG_ERR;
        notifyData.insert("eventType", -1);
        notifyData.insert("id", QString(""));
    }

    emit asyncCallbackN(status, m_notifyTrId, notifyData);
}

void QContactService::handleContactsChanged(const QList<QContactLocalId>& contactIds)
{
    QMap<QString,QVariant> notifyData;
    qint32 status = CJSEStatusCodes::SUCCESS;

    if (!contactIds.isEmpty()) {
        QContactManager* cm = new QContactManager();
        // Check contact type
        QContact addedCnt = cm->contact(contactIds.first());
        QContactName name = addedCnt.detail(QContactName::DefinitionName);

        if (addedCnt.type().compare(QString(QLatin1String(QContactType::TypeGroup))) == 0) {
            notifyData.insert("eventType", CJSENotifications::EVENT_GROUP_UPDATED);
        } else {
            notifyData.insert("eventType", CJSENotifications::EVENT_CONTACT_UPDATED);
        }
        notifyData.insert("id", QString(contactIds.first()));
    } else {
        status = CJSEStatusCodes::INVALID_ARG_ERR;
        notifyData.insert("eventType", -1);
        notifyData.insert("id", QString(""));
    }

    emit asyncCallbackN(status, m_notifyTrId, notifyData);
}

void QContactService::handleContactsRemoved(const QList<QContactLocalId>& contactIds)
{
    QMap<QString,QVariant> notifyData;
    qint32 status = CJSEStatusCodes::SUCCESS;

    if (!contactIds.isEmpty()) {
        QContactManager* cm = new QContactManager();
        // Check contact type
        QContact addedCnt = cm->contact(contactIds.first());
        QContactName name = addedCnt.detail(QContactName::DefinitionName);

        if (addedCnt.type().compare(QString(QLatin1String(QContactType::TypeGroup))) == 0) {
            notifyData.insert("eventType", CJSENotifications::EVENT_GROUP_DELETED);
        } else {
            notifyData.insert("eventType", CJSENotifications::EVENT_CONTACT_DELETED);
        }
        notifyData.insert("id", QString(contactIds.first()));
    } else {
        status = CJSEStatusCodes::INVALID_ARG_ERR;
        notifyData.insert("eventType", -1);
        notifyData.insert("id", QString(""));
    }

    emit asyncCallbackN(status, m_notifyTrId, notifyData);
}

