/*
 * ============================================================================
 *  Name        : qcommlogservice.h
 *  Part of     : serviceframework / WRT
 *  Description : Qt class for CommLog Service Provider
 *  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef QCOMMLOGSERVICE_H
#define QCOMMLOGSERVICE_H

#include <QObject>
#include <QVariant>
#include <QMap>
#include <QString>
#include <QList>

#include "servicesecurityinterface.h"
#include "qcommlogserviceinterface.h"
#include "qcommlogcallandsmsmodel.h"
#include "qcommloggetlisttask.h"


class QCommLogService : public QObject,
public ICommLog,
public IServiceSecurity
{
    Q_OBJECT
    Q_INTERFACES(ICommLog)

public:
    /**
    * constructor
    * @since x.y
    * @param
    * @return
    */
    QCommLogService();

    /**
     * destructor
     *     1 ) Loops thru Transaction Map and deletes all iterator Objects
     *     2 ) Delete core class instance
     * @since x.y
     * @param
     * @return
     */
    virtual ~QCommLogService();

    /**
     * Implemented from IServiceSecurity.Needed to handle capabilities.
     * @since x.y
     * @param WRT::SecSession
     * @return
     */
    void setSecuritySession(WRT::SecSession *aSecSession);

public slots:
    /**
     * This method is used to get the list of call and messaging events from default database.
     * This is an Asynchronous API
     * @since  x.y
     * @param  match         Specifies a matching filter for the event to return.
     *                       If the match string is NULL or Empty then all the events will be returned
     *@param   transId       An ID to uniquely identify this transaction
     *
     *@return 0 or Error Code
     *@emits  void asyncCallback(QString status,int transactionId,QObject iterator )
     *
     *@algo
     * -> Parse Input params
     * -> CallGetList
     */
    QVariant getList (const QMap<QString, QVariant>& match,  const int& transId);

    /**
     * This method is used to get the list of call and messaging events from default database.
     * This is an Synchronous API
     * @since  x.y
     * @param  match         Specifies a matching filter for the event to return.
     *                       If the match string is NULL or Empty then all the events will be returned
     *@return 0 or Error Code
     *@algo
     * -> Parse Input params
     * -> CallGetList
     */
    QObject* getList (const QMap<QString, QVariant>& match);

    /**
     * Cancels any asynchronous request identified uniquely by trasactionid.
     * This is a Synchronous API
     * @since x.y
     *
     * @param transactionID    Identifier of the ongoing asynchronous request to be canceled.
     */
    QVariant cancel(const int& transactionID);

    /**
     * Method takes one or more argments. This method registers the callback for listening to SMS and CALL events.
     * Invoking this method again would replace the previously registered callback function.
     * @since x.y
     * @param   transId       An ID to uniquely identify this transaction
     * @return 0 or ErrorCode
     */
    QVariant setNotification (const int& transId=-1);

    /**
     * Method unregisters the callback registered using setNotification() method.
     * @since x.y
     * @param   transId       An ID to uniquely identify this transaction
     * @return 0 or ErrorCode
     */
    QVariant cancelNotification(const int& transId=-1);

    /**
     * Method removes LogEntry by specified logId
     * @since x.y
     * @param   logId    Specified logId which niquely identifies the log.
     * @return 0 or ErrorCode
     */
    QVariant deleteLogEntry(const int& logId=-1);



signals:
    /**
    * The signal emitted for the all the Async functions.
    * @since x.y
    * @param status   - 0 or Error code
    * @param aTransId - TransactionID of the Async operation
    * @param iterator - The actual response data
    * @return
    */
    void asyncCallback(int status, int transactionId, QObject *eventIter);//getList callback

    /**
     * The signal emitted for the notification about new event
     * @since x.y
     * @param status - 0 or Error code
     * @param aTransId - TransactionID of the Async operation
     * @param logEntry - logData structure
     */
    void notificationCallback(int status,int transactionId,const QMap<QString,QVariant>&);//Notification about new event

    //internal signal about any errors
    void asyncGetListComplete(GetCommLogListTask* aTask, qint32 aTransactionId, int aErrorCode, QVariantList aMessages);

protected:

    WRT::SecSession *iSecSession;

private:
    //List async transactions
    QList<qint32> iGoingAsyncTransactions;
    //EventModel object
    CommCallAndSMSModel iEventModel;
    qint32 iNotifierTransactionId;
    bool iNotifierSet;

private slots:
    //internal slots for communicating with task
    void getListCallback(GetCommLogListTask *task, qint32, int, QVariantList);

    void newEventCallback(int aEventId);
};

#endif // QCOMMLOGSERVICE_H
