/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef CALENDARINTERFACE_H
#define CALENDARINTERFACE_H
//QT Include
#include <QVariant>
#include <QString>
#include <QMap>

//Service Framework Include
#include "servicebase.h"

static const char* KICalendarId = "com.nokia.ICalendar/1.0";

// IDataSource interface id
static const ServiceInterfaceInfo KICalendar =
{
        KICalendarId, 0
};

/**
 * Base class for calendar service.
 * @lib calendarplugin.lib
 * @since x.y
 */
class ICalendar : public IServiceBase
    {
public:
    /**
     * Get a list of calender entries.This is an Asynchronous call.
     * @param matchCriteria specifies criteria to select and return
     * calendar entries, is a  Map described as below:
     * --------------------------------------------------------------------------------------
     *  |Key             | Data type|Description
     *  --------------------------------------------------------------------------------------
     *  |id              | string   |Unique identifier of the entry to be retireved. If this
     *  |                |          |param is present all other params are ignored and the
     *  |                |          |entry matching this param is returned.
     *  --------------------------------------------------------------------------------------
     *  |range           | Map|begin|All calendar entries after this Time  should be obtained.
     *  |                |    |end  |All calendar entries before this Time should be obtained.
     *  --------------------------------------------------------------------------------------
     *  |type            | string   |Includes only entries of the type specified in the output.
     *  |                |          |Meeting,Reminder,ToDo,Anniversary and DayEvent
     *  ----------------------------------------------------------------------------------------
     *  | text           | string   |Matches the text string with the summary field of the entry.
     *  |                |          |It is not case sensitive
     *  ----------------------------------------------------------------------------------------
     * @param transId a unique id identifying one async request
     * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
     * below :
     * -------------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * -------------------------------------------------------------------------------------
     * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
     * -------------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * --------------------------------------------------------------------------------------
     * |transactionId       | int      |Unique Id identifying this request in case of success.
     * ---------------------------------------------------------------------------------------
     * @emits asyncCallback it emits asyncCallback signal to notify the result.
     * @since x.y
     */
    virtual QVariant getList(const QMap<QString,QVariant>& matchCriteria,int transId) = 0;

    /**
     * Get a list of calendar entries.This is a synchronous call.
     * @param matchCriteria specifies criteria to select and return
     * calendar entries, is a  Map described as below:
     * --------------------------------------------------------------------------------------
     *  |Key             | Data type|Description
     *  --------------------------------------------------------------------------------------
     *  |id              | string   |Unique identifier of the entry to be retireved. If this
     *  |                |          |param is present all other params are ignored and the
     *  |                |          |entry matching this param is returned.
     *  --------------------------------------------------------------------------------------
     *  |range           | Map|begin|All calendar entries after this Time  should be obtained.
     *  |                |    |end  |All calendar entries before this Time should be obtained.
     *  --------------------------------------------------------------------------------------
     *  |type            | string   |Includes only entries of the type specified in the output.
     *  |                |          |Meeting,Reminder,ToDo,Anniversary and DayEvent
     *  ----------------------------------------------------------------------------------------
     *  | text           | string   |Matches the text string with the summary field of the entry.
     *  |                |          |It is not case sensitive
     *  ----------------------------------------------------------------------------------------
     * @return  Returns the iterator as a QObject*
     * @since x.y
     */
    virtual QObject* getList(const QMap<QString,QVariant>& matchCriteria) = 0;


    /**
     * Deletes a list of calender entries.This is a synchronous call.
     * @param data the match criteria that is used when selecting the
     * entries to be deleted,is a  Map described as below:
     * --------------------------------------------------------------------------------------
     *  |Key             | Data type|Description
     *  --------------------------------------------------------------------------------------
     *  |id              | string   |Unique identifier of the entry to be deleted. If this
     *  |                |          |param is present all other params are ignored and the
     *  |                |          |entry matching this param is returned.
     *  --------------------------------------------------------------------------------------
     *  |range           | Map|begin|All calendar entries after this Time  should be deleted.
     *  |                |    |end  |All calendar entries before this Time should be deleted.
     *  --------------------------------------------------------------------------------------
     * @return  Map containing errorCode & errorMessage.Content of this map is given
     * below :
     * -----------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * -----------------------------------------------------------------------------------
     * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
     * -----------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english. //TODO
     * -----------------------------------------------------------------------------------
     * @since x.y
     */
    virtual QVariant deleteEntry(const QMap<QString,QVariant>& data) = 0;
    /**
     * Adds a list of entries to the calendar.This is a synchronous call.
     * @param calendarEntry specifies the values for params for the entry to be added,
     * is a  Map described as below:
     * --------------------------------------------------------------------------------------------
     *  |Key             | Data type|Description
     *  -------------------------------------------------------------------------------------------
     *  |type            | string   |Includes only entries of the type specified in the output.
     *  |                |          |Meeting,Reminder,ToDo,Anniversary and DayEvent
     *  -------------------------------------------------------------------------------------------
     *  |time            | Map|start|This represents the start time for the entry, applicable
     *  |                |    |     |for Meeting,  Anniversary, DayEvent and Reminder .
     *  |                |    |end  |The end time for the entry, applicable for Meeting,ToDo.
     *  |                |    |alarm|This field is applicable for all entry types.
     *  --------------------------------------------------------------------------------------------
     *  |summary         | string   |Defines a short summary or subject of the entry.
     *  --------------------------------------------------------------------------------------------
     *  |description     | string   |Provides a more complete description of the entry.
     *  --------------------------------------------------------------------------------------------
     *  |location        | string   |Specifies the location of a Meeting.
     *  --------------------------------------------------------------------------------------------
     *  |priority        | int   |The property defines the relative priority for the entry.
     *  |                |          |Range is 0(default) to 255.
     *  ---------------------------------------------------------------------------------------------
     *  |status          | string   |This property defines the overall status or confirmation .
     *  ----------------------------------------------------------------------------------------------
     *  |exceptionDates  | Array:Dates |Contains a list of dates in the original repeat sequence that
     *  |                |          |have been removed.
     *  ----------------------------------------------------------------------------------------------
     *  |repeatRule      | Map|frequency  |Specifies the repeat type. The possible values are daily,
     *  |                |    |           |weekly, monthly, yearly.
     *  |                |    |startDate  |Date on which the repeat starts. The startdate if not
     *  |                |    |           |specified will be taken from the Entry startdate.
     *  |                |    |untilDate  |Date on which this repeat rule finishes. If not specified
     *  |                |    |           |it will be taken as the maximum date by the platform.
     *  |                |    |interval   |Positive integer representing how often the rule repeats.
     *  |                |    |weekDays   |On what days of the week the rule should repeat.
     *  |                |    |month      |Specifies the month for a yearly repeat rule.[0(January) to 11(December)].
     *  |                |    |           |If not specified will be taken from the Entry startdate.
     *  |                |    |daysOfMonth|List of weekdays within a month; applicable
     *  |                |    |           |for monthly & yearly repeats.
     *  |                |    |monthDates |Specifies on what dates of the month the rule should repeat.
     *  -----------------------------------------------------------------------------------------------
     * @return  Map containing errorCode,errorMessage,uid.Content of this map is given
     * below :-
     * -------------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * -------------------------------------------------------------------------------------
     * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
     * -------------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * -------------------------------------------------------------------------------------
     * |uid                 | int      |Unique Id identifying the entry that has been added.
     * --------------------------------------------------------------------------------------
     * @since x.y
     */
    virtual QVariant addEntry(const QMap<QString,QVariant>& calendarEntry) = 0;

    /**
     * Subscribe for calendar entry change notification.This is an Asynchronous call.
     * @param transactionId a unique id identifying one async request
     * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
     * below :
     * -------------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * -------------------------------------------------------------------------------------
     * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
     * -------------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * --------------------------------------------------------------------------------------
     * |transactionId       | int      |Unique Id identifying this request in case of success.
     * ---------------------------------------------------------------------------------------
     * @emits asyncNotifyCallback it emits asyncNotifyCallback signal to notify the result.
     * @since x.y
     */
    virtual QVariant subscribeNotification(int transId) = 0;

    /**
     * Cancel notification on calendar entries.
     * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
     * below :
     * -------------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * -------------------------------------------------------------------------------------
     * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
     * -------------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * --------------------------------------------------------------------------------------
     * |transactionId       | int      |Unique Id identifying this request in case of success.
     * ---------------------------------------------------------------------------------------
     * @since x.y
     */
    virtual QVariant cancelNotification() = 0;

    /*
     * Cancels an outstanding asynchrnous request. This is a synchronous call.
     * @param transactionId a unique id identifying one async request
     * @return  Map containing errorCode and errorMessage.Content of this map is given
     * below :
     * -------------------------------------------------------------------------------------
     * |Key                 |DataType  |String
     * -------------------------------------------------------------------------------------
     * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
     * -------------------------------------------------------------------------------------
     * |errorMessage        | string   |error description in engineering english.
     * --------------------------------------------------------------------------------------
     */
    virtual QVariant cancel(const int transactionId) = 0;

        /**
         * Start the device native calendar editor UI with pre-populated calendar entry data.
         * This is an asynchronous method. This api can be used to add a new entry or update an existing entry.
         * * @param calendarEntry specifies the values for params for the entry to be added,
         * is a  Map described as below:
         * --------------------------------------------------------------------------------------------
         *  |Key             | Data type|Description
         *  -------------------------------------------------------------------------------------------
         *  |type            | string   |Includes only entries of the type specified in the output.
         *  |                |          |Meeting,Reminder,ToDo,Anniversary and DayEvent
         *  -------------------------------------------------------------------------------------------
         *  |time            | Map|start|This represents the start time for the entry, applicable
         *  |                |    |     |for Meeting,  Anniversary, DayEvent and Reminder .
         *  |                |    |end  |The end time for the entry, applicable for Meeting,ToDo.
         *  |                |    |alarm|This field is applicable for all entry types.
         *  --------------------------------------------------------------------------------------------
         *  |summary         | string   |Defines a short summary or subject of the entry.
         *  --------------------------------------------------------------------------------------------
         *  |description     | string   |Provides a more complete description of the entry.
         *  --------------------------------------------------------------------------------------------
         *  |location        | string   |Specifies the location of a Meeting.
         *  --------------------------------------------------------------------------------------------
         *  |priority        | int   |The property defines the relative priority for the entry.
         *  |                |          |Range is 0(default) to 255.
         *  ---------------------------------------------------------------------------------------------
         *  |status          | string   |This property defines the overall status or confirmation .
         *  ----------------------------------------------------------------------------------------------
         *  |exceptionDates  | Array:Dates |Contains a list of dates in the original repeat sequence that
         *  |                |          |have been removed.
         *  ----------------------------------------------------------------------------------------------
         *  |repeatRule      | Map|frequency  |Specifies the repeat type. The possible values are daily,
         *  |                |    |           |weekly, monthly, yearly.
         *  |                |    |startDate  |Date on which the repeat starts. The startdate if not
         *  |                |    |           |specified will be taken from the Entry startdate.
         *  |                |    |untilDate  |Date on which this repeat rule finishes. If not specified
         *  |                |    |           |it will be taken as the maximum date by the platform.
         *  |                |    |interval   |Positive integer representing how often the rule repeats.
         *  |                |    |weekDays   |On what days of the week the rule should repeat.
         *  |                |    |month      |Specifies the month for a yearly repeat rule.[0(January) to 11(December)].
         *  |                |    |           |If not specified will be taken from the Entry startdate.
         *  |                |    |daysOfMonth|List of weekdays within a month; applicable
         *  |                |    |           |for monthly & yearly repeats.
         *  |                |    |monthDates |Specifies on what dates of the month the rule should repeat.
         *  -----------------------------------------------------------------------------------------------
         * @param transactionId a unique id identifying one async request
         * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
         * below :
         * -------------------------------------------------------------------------------------
         * |Key                 |DataType  |String
         * -------------------------------------------------------------------------------------
         * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
         * -------------------------------------------------------------------------------------
         * |errorMessage        | string   |error description in engineering english.
         * --------------------------------------------------------------------------------------
         * |transactionId       | int      |Unique Id identifying this request in case of success.
         * ---------------------------------------------------------------------------------------
         * @emits asyncCallback it emits asyncCallback signal to notify the result.
         * @since x.y
         *
         */
         virtual QVariant startEditor(const QMap<QString,QVariant>& calendarEntry, const int transactionId) = 0;

    };

Q_DECLARE_INTERFACE(ICalendar, KICalendarId);

#endif //CALENDARINTERFACE_H
