/*
 * ============================================================================
 *  Name        : qcalendarservice.h
 *  Part of     : serviceframework / WRT
 *  Description : Qt class for calendar provider
 *  Version     : %version: 16 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef QCALENDARSERVICE_H_
#define QCALENDARSERVICE_H_

#include <QObject>
#include <QMap>
#include <QVariant>
#include <QList>
#include <QString>
#include <QtDBus/QtDBus>
#include <calendar-backend/CCalendar.h>
#include <calendar-backend/CMulticalendar.h>

#include "servicesecurityinterface.h"
#include "qcalendarserviceinterface.h"
#include "qcalendariterator.h"

//---------------

namespace JSEStatusCodes {

    const qint32 SUCCESS = 0;

    // NON-RECOVERABLE Exception codes
    const qint32 MISSING_ARG_ERR = 1;
        //const qint32 INVALID_ARG_ERR = 2;
        const qint32 INVALID_ARG_ERR = -6;
        //const qint32 NOT_SUPPORTED_ERR = 3;
        const qint32 NOT_SUPPORTED_ERR = -5;

    // RECOVERABLE Exception codes
    const qint32 TIMEOUT_ERR = 100;
        //const qint32 DATA_NOT_FOUND_ERR = 101;
        const qint32 DATA_NOT_FOUND_ERR = -1;
        //const qint32 DATA_ALREADY_EXISTS_ERR = 102;
        const qint32 DATA_ALREADY_EXISTS_ERR = 11;
        //const qint32 SERVICE_BUSY_ERR = 103;
        const qint32 SERVICE_BUSY_ERR = -16;
        //const qint32 SERVICE_IN_USE_ERR = 104;
        const qint32 SERVICE_IN_USE_ERR = -14;
        //const qint32 DATA_OUT_OF_RANGE_ERR = 105;
        const qint32 DATA_OUT_OF_RANGE_ERR = -15;
        //const qint32 NOT_ALLOWED_ERR = 106;
        const qint32 NOT_ALLOWED_ERR = -21;
        //const qint32 SIZE_EXCEEDED_ERR = 107;
        const qint32 SIZE_EXCEEDED_ERR = -4;
    const qint32 INVALID_URI_ERR = 108;
    const qint32 URI_NOT_FOUND_ERR = 109;
    const qint32 URI_ALREADY_EXISTS_ERR = 110;

    // DOMAIN SPECIFIC Status codes
    const qint32 DIR_NOT_EMPTY_ERR = 500;
    const qint32 FILE_ALREADY_OPEN_ERR = 501;
    const qint32 URI_NOT_SUPPORTED = 502;
}


/**
 * Calendar provider class.Provides method to add a new calendar entry...
 * changes.
 * @lib libcalendarserviceplugin.so
 * @since x.y
 */
class QCalendarService : public QObject, 
public ICalendar,
public IServiceSecurity
    {
    Q_OBJECT
    Q_INTERFACES(ICalendar)

public:
    /**
     * Default constructor
     * @since x.y
     * @return none
     */
    QCalendarService();

    /**
     * Destructor
     * @since x.y
     * @return none
     */
    virtual ~QCalendarService();

    /**
     * setSecuritySession Setting a security session
     * @param secSession handle to the security session
     * @return none
     * @since x.y
     */
    void setSecuritySession(WRT::SecSession *secSession);


    signals:
    /**
         * Emitted as signal whenever a calendar requests are complete
         * with data obtained or error if it occurs during retrieval.
         * @param errorCode error that has occured
         * @param transactionId uniquely identifies a getCurrentPosition request.
         * @param data calendar request data in case error param is 0.This
         * in turn contains map with calendar information such as uid,type,etc
         * @return none
         * @since x.y
         */

    void getListCallback(int errorCode, int transactionId, QObject* data);

    /**
         * Emitted as signal whenever a calendar requests are complete
         * with data obtained or error if it occurs during retrieval.
         * @param errorCode error that has occured
         * @param transactionId uniquely identifies a getCurrentPosition request.
         * @param data calendar request data in case error param is 0.This
         * in turn contains map with calendar information such as uid,type,etc
         * @return none
         * @since x.y
         */

    void notifyCallback(int errorCode, int transactionId, QObject* data);

        void asyncCallback();
        //void asyncCallback(int errorCode,int transactionId, QVariant* data);

 public slots:
        //slot for listOfCalendarEntries, mapList contains founded entries in Calendar DB
        void fireGetListCallback(const QList<QVariantMap>* mapList, qint32 trId, qint32 status);
        void fireSubscribeCallback(const QDBusMessage& messages);
 public slots://Methods inherited from ICalendar

/**
 * Get a list of calender entries.This is an Asynchronous call.
 * @param matchCriteria specifies criteria to select and return
 * calendar entries, is a  Map described as below:
 * --------------------------------------------------------------------------------------
 *  |Key             | Data type|Description
 *  --------------------------------------------------------------------------------------
 *  |id              | string   |Unique identifier of the entry to be retireved. If this
 *  |                |          |param is present all other params are ignored and the
 *  |                |          |entry matching this param is returned.
 *  --------------------------------------------------------------------------------------
 *  |range           | Map|begin|All calendar entries after this Time  should be obtained.
 *  |                |    |end  |All calendar entries before this Time should be obtained.
 *  --------------------------------------------------------------------------------------
 *  |type            | string   |Includes only entries of the �type� specified in the output.
 *  |                |          |Meeting,Reminder,ToDo,Anniversary and DayEvent
 *  ----------------------------------------------------------------------------------------
 *  | text           | string   |Matches the text string with the �summary� field of the entry.
 *  |                |          |It is not case sensitive
 *  ----------------------------------------------------------------------------------------
 * @param transId a unique id identifying one async request
 * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
 * below :
 * -------------------------------------------------------------------------------------
 * |Key                 |DataType  |String
 * -------------------------------------------------------------------------------------
 * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
 * -------------------------------------------------------------------------------------
 * |errorMessage        | string   |error description in engineering english.
 * --------------------------------------------------------------------------------------
 * |transactionId       | int      |Unique Id identifying this request in case of success.
 * ---------------------------------------------------------------------------------------
 * @emits asyncCallback it emits asyncCallback signal to notify the result.
 * @since x.y
 */
QVariant getList(const QMap<QString,QVariant>& matchCriteria,int transId);

/**
 * Get a list of calendar entries.This is a synchronous call.
 * @param matchCriteria specifies criteria to select and return
 * calendar entries, is a  Map described as below:
 * --------------------------------------------------------------------------------------
 *  |Key             | Data type|Description
 *  --------------------------------------------------------------------------------------
 *  |id              | string   |Unique identifier of the entry to be retireved. If this
 *  |                |          |param is present all other params are ignored and the
 *  |                |          |entry matching this param is returned.
 *  --------------------------------------------------------------------------------------
 *  |range           | Map|begin|All calendar entries after this Time  should be obtained.
 *  |                |    |end  |All calendar entries before this Time should be obtained.
 *  --------------------------------------------------------------------------------------
 *  |type            | string   |Includes only entries of the type specified in the output.
 *  |                |          |Meeting,Reminder,ToDo,Anniversary and DayEvent
 *  ----------------------------------------------------------------------------------------
 *  | text           | string   |Matches the text string with the summary field of the entry.
 *  |                |          |It is not case sensitive
 *  ----------------------------------------------------------------------------------------
 * @return  Returns the iterator as a QObject*
 * @since x.y
 */
QObject* getList(const QMap<QString,QVariant>& matchCriteria);

/**
 * Deletes a list of calender entries.This is a synchronous call.
 * @param data the match criteria that is used when selecting the
 * entries to be deleted,is a  Map described as below:
 * --------------------------------------------------------------------------------------
 *  |Key             | Data type|Description
 *  --------------------------------------------------------------------------------------
 *  |id              | string   |Unique identifier of the entry to be deleted. If this
 *  |                |          |param is present all other params are ignored and the
 *  |                |          |entry matching this param is returned.
 *  --------------------------------------------------------------------------------------
 *  |range           | Map|begin|All calendar entries after this Time  should be deleted.
 *  |                |    |end  |All calendar entries before this Time should be deleted.
 *  --------------------------------------------------------------------------------------
 * @return  Map containing errorCode & errorMessage.Content of this map is given
 * below :
 * -----------------------------------------------------------------------------------
 * |Key                 |DataType  |String
 * -----------------------------------------------------------------------------------
 * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
 * -----------------------------------------------------------------------------------
 * |errorMessage        | string   |error description in engineering english.
 * -----------------------------------------------------------------------------------
 * @since x.y
 */
QVariant deleteEntry(const QMap<QString,QVariant>& data);

/**
 * Adds a list of entries to the calendar.Also updates an existing entry if id is specified.
 * This is a synchronous call.
 * @param calendarEntry specifies the values for params for the entry to be added,
 * is a  Map described as below:
 * --------------------------------------------------------------------------------------------
 *  |Key             | Data type      |Description
 *  -------------------------------------------------------------------------------------------
 *  |id              | string         |Id of the entry. If specified, that entry in the calendar is
 *  |                |                |updated with other fields.
 *  -------------------------------------------------------------------------------------------
 *  |type            | string         |Includes only entries of the �type� specified in the output.
 *  |                |                |Meeting,Reminder,ToDo,Anniversary and DayEvent
 *  -------------------------------------------------------------------------------------------
 *  |time            | Map|start      |This represents the start time for the entry, applicable
 *  |                |    |           |for �Meeting�,  �Anniversary�, �DayEvent� and �Reminder� .
 *  |                |    |end        |The end time for the entry, applicable for �Meeting�,�ToDo�.
 *  |                |    |alarm      |This field is applicable for all entry types.
 *  --------------------------------------------------------------------------------------------
 *  |summary         | string         |Defines a short summary or subject of the entry.
 *  --------------------------------------------------------------------------------------------
 *  |description     | string         |Provides a more complete description of the entry.
 *  --------------------------------------------------------------------------------------------
 *  |location        | string         |Specifies the location of a �Meeting�.
 *  --------------------------------------------------------------------------------------------
 *  |priority        | int            |The property defines the relative priority for the entry.
 *  |                |                |Range is 0(default) to 255.
 *  ---------------------------------------------------------------------------------------------
 *  |status          | string         |This property defines the overall status or confirmation .
 *  ----------------------------------------------------------------------------------------------
 *  |exceptionDates  | Array:Dates    |Contains a list of dates in the original repeat sequence that
 *  |                |                |have been removed.
 *  ----------------------------------------------------------------------------------------------
 *  |repeatRule      | Map|frequency  |Specifies the repeat type. The possible values are daily,
 *  |                |    |           |weekly, monthly, yearly.
 *  |                |    |startDate  |Date on which the repeat starts. The startdate if not
 *  |                |    |           |specified will be taken from the Entry startdate.
 *  |                |    |untilDate  |Date on which this repeat rule finishes. If not specified
 *  |                |    |           |it will be taken as the maximum date by the platform.
 *  |                |    |interval   |Positive integer representing how often the rule repeats.
 *  |                |    |weekDays   |On what days of the week the rule should repeat.
 *  |                |    |month      |Specifies the month for a yearly repeat rule.[0(January) to 11(December)].
 *  |                |    |           |If not specified will be taken from the Entry startdate.
 *  |                |    |daysOfMonth|List of weekdays within a month; applicable
 *  |                |    |           |for monthly & yearly repeats.
 *  |                |    |monthDates |Specifies on what dates of the month the rule should repeat.
 *  -----------------------------------------------------------------------------------------------
 * @return  Map containing errorCode,errorMessage,uid.Content of this map is given
 * below :-
 * -------------------------------------------------------------------------------------
 * |Key                 |DataType  |Description
 * -------------------------------------------------------------------------------------
 * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
 * -------------------------------------------------------------------------------------
 * |errorMessage        | string   |error description in engineering english.
 * -------------------------------------------------------------------------------------
 * |uid                 | int      |Unique Id identifying the entry that has been added.
 * --------------------------------------------------------------------------------------
 * @emits asyncCallback it emits asyncCallback signal to notify the result.
 * @since x.y
 */
QVariant addEntry(const QMap<QString,QVariant>& calendarEntry);

/*
 * Cancels an outstanding asynchrnous request. This is a synchronous call.
 * @param transactionId a unique id identifying one async request
 * @return  Map containing errorCode and errorMessage.Content of this map is given
 * below :
 * -------------------------------------------------------------------------------------
 * |Key                 |DataType  |String
 * -------------------------------------------------------------------------------------
 * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
 * -------------------------------------------------------------------------------------
 * |errorMessage        | string   |error description in engineering english.
 * --------------------------------------------------------------------------------------
 */
QVariant cancel(const int transactionId);

/**
 * Subscribe for calendar entry change notification.This is an Asynchronous call.
 * @param transactionId a unique id identifying one async request
 * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
 * below :
 * -------------------------------------------------------------------------------------
 * |Key                 |DataType  |String
 * -------------------------------------------------------------------------------------
 * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
 * -------------------------------------------------------------------------------------
 * |errorMessage        | string   |error description in engineering english.
 * --------------------------------------------------------------------------------------
 * |transactionId       | int      |Unique Id identifying this request in case of success.
 * ---------------------------------------------------------------------------------------
 * @emits asyncNotifyCallback it emits asyncNotifyCallback signal to notify the result.
 * @since x.y
 */
QVariant subscribeNotification(int transactionId);

/**
 * Cancel notification on calendar entries.
 * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
 * below :
 * -------------------------------------------------------------------------------------
 * |Key                 |DataType  |String
 * -------------------------------------------------------------------------------------
 * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
 * -------------------------------------------------------------------------------------
 * |errorMessage        | string   |error description in engineering english.
 * --------------------------------------------------------------------------------------
 * |transactionId       | int      |Unique Id identifying this request in case of success.
 * ---------------------------------------------------------------------------------------
 * @since x.y
 */
QVariant cancelNotification();

/**
 * Start the device native calendar editor UI with pre-populated calendar entry data.
 * This is an asynchronous method. This api can be used to add a new entry or update an existing entry.
 * * @param calendarEntry specifies the values for params for the entry to be added,
 * is a  Map described as below:
 * --------------------------------------------------------------------------------------------
 *  |Key             | Data type|Description
 *  -------------------------------------------------------------------------------------------
 *  |type            | string   |Includes only entries of the type specified in the output.
 *  |                |          |Meeting,Reminder,ToDo,Anniversary and DayEvent
 *  -------------------------------------------------------------------------------------------
 *  |time            | Map|start|This represents the start time for the entry, applicable
 *  |                |    |     |for Meeting,  Anniversary, DayEvent and Reminder .
 *  |                |    |end  |The end time for the entry, applicable for Meeting,ToDo.
 *  |                |    |alarm|This field is applicable for all entry types.
 *  --------------------------------------------------------------------------------------------
 *  |summary         | string   |Defines a short summary or subject of the entry.
 *  --------------------------------------------------------------------------------------------
 *  |description     | string   |Provides a more complete description of the entry.
 *  --------------------------------------------------------------------------------------------
 *  |location        | string   |Specifies the location of a Meeting.
 *  --------------------------------------------------------------------------------------------
 *  |priority        | int   |The property defines the relative priority for the entry.
 *  |                |          |Range is 0(default) to 255.
 *  ---------------------------------------------------------------------------------------------
 *  |status          | string   |This property defines the overall status or confirmation .
 *  ----------------------------------------------------------------------------------------------
 *  |exceptionDates  | Array:Dates |Contains a list of dates in the original repeat sequence that
 *  |                |          |have been removed.
 *  ----------------------------------------------------------------------------------------------
 *  |repeatRule      | Map|frequency  |Specifies the repeat type. The possible values are daily,
 *  |                |    |           |weekly, monthly, yearly.
 *  |                |    |startDate  |Date on which the repeat starts. The startdate if not
 *  |                |    |           |specified will be taken from the Entry startdate.
 *  |                |    |untilDate  |Date on which this repeat rule finishes. If not specified
 *  |                |    |           |it will be taken as the maximum date by the platform.
 *  |                |    |interval   |Positive integer representing how often the rule repeats.
 *  |                |    |weekDays   |On what days of the week the rule should repeat.
 *  |                |    |month      |Specifies the month for a yearly repeat rule.[0(January) to 11(December)].
 *  |                |    |           |If not specified will be taken from the Entry startdate.
 *  |                |    |daysOfMonth|List of weekdays within a month; applicable
 *  |                |    |           |for monthly & yearly repeats.
 *  |                |    |monthDates |Specifies on what dates of the month the rule should repeat.
 *  -----------------------------------------------------------------------------------------------
 * @param transactionId a unique id identifying one async request
 * @return  Map containing errorCode,errorMessage,transactionId.Content of this map is given
 * below :
 * -------------------------------------------------------------------------------------
 * |Key                 |DataType  |String
 * -------------------------------------------------------------------------------------
 * |errorCode           | int      |symbian error code in case of error, 0 otherwise.
 * -------------------------------------------------------------------------------------
 * |errorMessage        | string   |error description in engineering english.
 * --------------------------------------------------------------------------------------
 * |transactionId       | int      |Unique Id identifying this request in case of success.
 * ---------------------------------------------------------------------------------------
 * @emits asyncCallback it emits asyncCallback signal to notify the result.
 * @since x.y
 *
 */
QVariant startEditor(const QMap<QString,QVariant>& calendarEntry, const int transactionId);

private:
    //creates native entry based on passed entry (as QMap)
    QVariant _createEntry();
    //updates native entry based on passed entry (as QMap)
    QVariant _updateEntry(QString uid);
    //validates passed entry (as QMap), performs conversion from js entry data to native data
    int _parseEntry(const QMap<QString,QVariant>& calendarEntry, bool isUpdate);
    //merges passed updated entry data with existing entry data
    QMap<QString,QVariant> _mergeQmap(const QMap<QString,QVariant>&  qTarget, const QMap<QString,QVariant>&  qSource);
    //creates native recurrence rule based on passed repeat rule
    vector <string> _createRecurrenceRuleFromRepeatRule();
    //creates native exception rule based on passed exception dates
    vector <string> _createRecurrenceRuleFromExceptionDates();
    //creates recurrence from recurrence rules
    bool _createRecurrence(CRecurrence *pRecurrence);
    //parses received d-bus message
    QList<QString> _parseDbusMessage(QString message);
    //validates matchpattern data
    QMap<QString,QVariant> _validateMatchPattern(QMap<QString,QVariant>& matchCriteria);

    //Handle to the security session
    WRT::SecSession *securitySession;
    //CMulticalendar instance
    CMulticalendar* multiCalendar;
    //CCalendar instance
    CCalendar* calendar;
    //calendar id
    int calendarId;
    //transaction ids list
    QList<qint32> m_onGoingAsyncTransactions;
    //entry with data converted to native data
    QMap<QString,QVariant> parsedEntry;
    //notification flag
    bool m_isNotifyInProgress;
    //transaction id
    qint32 m_notifyTrId;
};
#endif //CALENDARSERVICE_H_
