/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QDir>
#include <serviceframeworkdefs.h>
#include <serviceipcserver.h>
#include <serviceipcrequest.h>
#include <servicedatabase.h>
#include <serviceplugininterface.h>
#include <servicedebug.h>
#include <serviceresolverdefines.h>
#include <sfwlog.h>
#include "serviceresolver.h"
#include "clientinfo.h"

#ifdef __SYMBIAN32__
#include "e32std.h"
#include "sysutil.h"
#endif

namespace WRT {

    static const char VERSION_SEPARATOR = '.';
    static const char WIN_LIB_EXT[] = ".dll";
#if defined(Q_OS_LINUX)
    static const char LINUX_LIB_EXT[] = ".so";
#endif
#if defined(Q_OS_MAC)
    static const char MAC_LIB_EXT[] = ".dylib";
#endif
    static const char XML_EXT[] = ".xml";
    static const char QTPLUGIN_EXT[] = ".qtplugin";
    static const char QT_PATH_SEPARATOR = '/';
    static const char DRIVE_SEPARATOR[] = ":\\";
    static const char SERVICE_NOT_SUPPORT[] = "ServiceNotSupport";
#ifdef __linux__
    static const char IMPORT_FOLDER[] = "import/";
    static const char REGISTERED_FOLDER[] = "registered/";
    static const char PATH_SEPARATOR = '/';
    static const char LIB_PREFIX[] = "lib";
#else
    static const char IMPORT_FOLDER[] = "import\\";
    static const char REGISTERED_FOLDER[] = "registered\\";
    static const char PATH_SEPARATOR = '\\';
#endif
    const int MAX_RETRY = 10;

#ifdef __SYMBIAN32__
    static const char  RESOLVER_PRIVATE_PATH[] ="C:\\private\\102829B8\\";
    static const char  C_DRIVE[]="C:";
    static const char  Z_DRIVE[]="Z:";
    static const char  SYMBIAN_BIN_FOLDER[] =":\\sys\\bin\\";
    static const char  QT_PLUGIN_FOLDER[] =":\\resource\\qt\\sfwplugin\\";
#endif

/*!
    \class ServiceResolver
    Public class that implements the service registry server. \n
    The server handles requests received from the service handler and uses IPC server \n
    for communicating with it. \n
    The server offers support for service registration and service metadata retrieval. \n
    The server uses a Sqlite database to store metadata for each registered service. \n
*/

/*!
    Constructor
    @param aParent Parent to this QObject
*/
ServiceResolver::ServiceResolver( QObject *aParent )
    : QObject( aParent ),
    m_parser( NULL ),
    m_fsWatcher( NULL )
{
    //Start the ipc server and listen for incoming requests
    m_server = new ServiceFwIPCServer( this, this );
    if (!m_server->listen( SERVICE_RESOLVER_NAME )) {
        //Close server if it is already started
        QTimer::singleShot(1, this, SLOT( closeServer()));
        return;
    }
    connect( m_server,
           SIGNAL( handleExit()),
           this,
           SLOT( closeServer()));

#ifdef __SYMBIAN32__
    TFileName path;
    RFs fileServer;
    bool bPrivatePath(false);
    if (fileServer.Connect() == KErrNone)
        {
        if (fileServer.PrivatePath(path) == KErrNone)
            {
            TFindFile findFile(fileServer);
            if (findFile.FindByDir(path, KNullDesC) == KErrNone)
                {
                path = findFile.File();
                m_privatePath = QString::fromUtf16(path.Ptr(),path.Length());
                if (m_privatePath.indexOf(PATH_SEPARATOR) == 2 || m_privatePath.indexOf(QT_PATH_SEPARATOR) == 2)
                    {
                    m_privatePath.replace(0, 2, C_DRIVE);
                    }
                else
                   {
                    m_privatePath.insert(0, C_DRIVE);
                    }
                if (m_privatePath.lastIndexOf(QT_PATH_SEPARATOR) != m_privatePath.length() - 1 &&
                    m_privatePath.lastIndexOf(PATH_SEPARATOR) != m_privatePath.length() - 1)
                    {
                    m_privatePath.append(PATH_SEPARATOR);
                    }
                m_importFolder = m_privatePath + IMPORT_FOLDER;
                m_registeredFolder = m_privatePath + REGISTERED_FOLDER;
                bPrivatePath = true;
                }
            }
        fileServer.Close();
        }
    //If the private path was not retrieved
    if (!bPrivatePath)
        {
        m_privatePath = RESOLVER_PRIVATE_PATH;
        m_importFolder = m_privatePath + IMPORT_FOLDER;
        m_registeredFolder = m_privatePath + REGISTERED_FOLDER;
        }
#else
    m_importFolder = QString("/home/user/.cache/wrt/api/import/");
    m_registeredFolder = QString("/home/user/.cache/wrt/api/registered/");
#endif

#if defined(Q_OS_WIN32) || defined(Q_OS_SYMBIAN)
    m_libraryExt = QString(WIN_LIB_EXT);
#elif defined(Q_OS_LINUX)
    m_libraryExt = QString(LINUX_LIB_EXT);
#elif defined(Q_OS_MAC)
    m_libraryExt = QString(MAC_LIB_EXT);
#endif

    //Create IMPORT folder in case it does not exist
    QDir dir;
    if (!dir.exists(m_importFolder)) {
        dir.mkpath(m_importFolder);;
    }

    //Register files from import folder (first time service discovery)
    //first services installed by user will be registered then services in rom image (for s60)
    //will be copied to import folder to be registered
    onDirectoryChanged(m_importFolder);

#if (!defined(__WINSCW__) && defined(__SYMBIAN32__))
    TBuf<KSysUtilVersionTextLength> vBuf;
    if ( SysUtil::GetSWVersion( vBuf ) == KErrNone ) {
        QString romImageVersion((QChar*)vBuf.Ptr(),vBuf.Length());
        ServiceDatabase database;
        database.setDatabasePath(m_privatePath);
        int dbResult = database.open();
        if (dbResult ==  ServiceDatabase::SFW_ERROR_DB_RECREATED) {
            database.close();
            recoverDB();
            dbResult = database.open();
        }
        QString oldRomImage;
        if (dbResult == 0) {
            database.getRomImageVersion(oldRomImage);
        }
        if (oldRomImage.compare(romImageVersion) != 0) {
            //first time registering rom image version
            if (oldRomImage.isEmpty()) {
                database.insertRomImageVersion(romImageVersion);
            } else {
                //update rom image version
                database.updateRomImageVersion(romImageVersion);
            }
            QString romImportPath = m_importFolder;
            romImportPath.replace(0, 2, Z_DRIVE);
            QDir romImportDir(romImportPath, '*' + QString(XML_EXT) );
            QStringList list = romImportDir.entryList();
            //copy xml files from resolver import folder on z: drive to c: drive
            if (list.count() > 0) {
                foreach (const QString &fileName, list ) {
                    if (!QFile::copy(romImportPath + fileName, m_importFolder + fileName)) {
                        SFW_DEBUG2(ERROR_COPY_ROM_XMLFILE_MESSAGE, fileName.toLatin1());
                    }
                }
                onDirectoryChanged(m_importFolder);
            }
        }
        database.close();
    }
#endif

    // Watch the import folder
    m_fsWatcher = new QFileSystemWatcher( QStringList(m_importFolder) );
    connect( m_fsWatcher,
        SIGNAL( directoryChanged(const QString &)),
        this,
        SLOT( onDirectoryChanged( const QString &)));
}

/*!
    Destructor
*/
ServiceResolver::~ServiceResolver()
{
    m_server->disconnect();
    delete m_parser;
    delete m_fsWatcher;
}


/*!
    Recovers DB from xml files in registered folder
*/
void ServiceResolver::recoverDB()
{
    QDir registeredDir(m_registeredFolder, '*' + QString(XML_EXT) );
    QStringList list = registeredDir.entryList();
    if (list.count() >0) {
#ifdef __SYMBIAN32__
        TInt fsResult(KErrNone);
        fsResult = m_fsSession.Connect();
        if (fsResult == KErrNone) {
#endif
            foreach (const QString& file, list ) {
                // Register the service
                QString filename = m_registeredFolder + file;
                int result = registerService( filename );
                if ( result ==  ServiceDatabase::SFW_ERROR_SERVICE_ALREADY_REGISTERED ||
                    (result >= ServiceMetaData::SFW_ERROR_NO_SERVICE && result <= ServiceMetaData::SFW_ERROR_PARSE_INTERFACE)) {
                    registeredDir.remove(file);
                    SFW_DEBUG1(ERROR_CAN_NOT_RECOVER_DB_ERROR_PARSING_MESSAGE);
                //parsing was OK, error in registration
                } else if ( result ==  ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN ||
                            result == ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION) {
                            SFW_DEBUG1(ERROR_CAN_NOT_RECOVER_DB_ERROR_DB_CONNECTION_MESSAGE);
                }
            }
#ifdef __SYMBIAN32__
            m_fsSession.Close();
        }
#endif
    }
}

/*!
    File system watcher notification as result of directory change
    @param aPath path of the directory that was modified
*/
void ServiceResolver::onDirectoryChanged( const QString &aPath )
{

    QString path;
    if (aPath.isEmpty()) {
        m_count++;
        path = m_importFolder;
    }
    else{
        m_count = 0;
        path = aPath;
    }
    QDir currentDir;
    if (!aPath.isEmpty() && !currentDir.exists(aPath)) {
        //The registration does not work if the IMPORT folder is deleted
        //Recreate and reset watch on the IMPORT folder in case it was deleted
        m_fsWatcher->removePath(aPath);
        currentDir.mkpath( aPath );

        // If the directory being monitored by m_fsWatcher is deleted and recreated
        // multiple times, m_fsWatcher seems to behave incorrectly. To be more specific,
        // when "m_fsWatcher->removePath(m_importFolder);" above gets called, m_importFolder
        // might NOT be removed from the monitor list successfully. When this happens,
        // the newly creately m_importFolder folder can't be added to the monitor list because
        // the old one with the same name still exists. As a result, any changes made to the new
        // m_importFolder won't be detected by m_fsWatcher. As a workaround, whenever we can't remove
        // old m_importFolder successfully, we destroy the file system watcher, and create a new one.
        QStringList directoryList = m_fsWatcher->directories();
        //remove workaround, it does not work on Tube.
        /*if (directoryList.count()>0) {
            qDebug()<<"##### Destroy file system watcher and create new one";
            disconnect( m_fsWatcher,
                SIGNAL( directoryChanged(const QString &)),
                this,
                SLOT( onDirectoryChanged( const QString &)));
            delete m_fsWatcher;
            m_fsWatcher = new QFileSystemWatcher( QStringList(aPath) );
            connect( m_fsWatcher,
                SIGNAL( directoryChanged(const QString &)),
                this,
                SLOT( onDirectoryChanged( const QString &)));
        } else {
            m_fsWatcher->addPath(aPath);
        }*/
        m_fsWatcher->addPath(aPath);
    } else {
        //Get all xml files from import folder and register services
        QDir importDir(path, '*' + QString(XML_EXT) );
        QStringList list = importDir.entryList();
        //Create REGISTERED folder
        if (list.count() >0) {
#ifdef __SYMBIAN32__
            TInt result(KErrNone);
            result = m_fsSession.Connect();
            if (result == KErrNone) {
#endif
                if (!currentDir.exists(m_registeredFolder)) {
                    currentDir.mkpath(m_registeredFolder);
                }
                bool UNABLE_TO_OPEN_FILE = false;
                foreach (const QString &file, list ) {
                    QFile registeredXmlFile(m_registeredFolder + file);
                    QFile importXmlFile(path + file);

                    int result = registerService(path + file);
                    //if no error in parsing and registration
                    if (result == 0) {
                        // check rights and make it writeable
                        SFW_DEBUG1(NEW_SERVICE_REGISTERED_MESSAGE);
                        QFile::Permissions perm = importXmlFile.permissions();
                        if (!perm.testFlag( QFile::WriteOwner )) {
                            perm |= QFile::WriteOwner;
                            if (!importXmlFile.setPermissions( perm )) {
                                SFW_DEBUG1(ERROR_SET_FILE_PERMISSION_MESSAGE);
                            }
                        }
                        // Move xml file to registered folder, if a file with the same name already exists it will be overwritten
                        if (!QFile::copy(path + file, m_registeredFolder + file)) {
                            SFW_DEBUG1(ERROR_COPY_FILE_TO_REGISTERED_FOLDER_MESSAGE);
                        }
                        //Remove file from import folder
                        if (!importDir.remove(file)) {
                            SFW_DEBUG1(ERROR_DELETE_FILE_FROM_IMPORT_FOLDER_MESSAGE);
                        }
                    //Error opening xml file, keep the file in import folder to try again next time
                    //if already tried MAX_RETRIED times, delete the file from import folder
                    } else if (result == ServiceMetaData::SFW_ERROR_UNABLE_TO_OPEN_FILE) {
                        UNABLE_TO_OPEN_FILE = true;
                        if ( m_count == MAX_RETRY ) {
                            importDir.remove(file);
                        }
                    //error parsing xml file or service already registered,
                    //no new registration, remove the xml file from import folder
                    } else if ( result ==  ServiceDatabase::SFW_ERROR_SERVICE_ALREADY_REGISTERED ||
                                result == ServiceDatabase::SFW_ERROR_FILE_PATH_UPDATED ||
                                result == ServiceResolver::SFW_ERROR_DLL_VERSION_NOT_MATCH_SERVICE_VERSION_IN_XML ||
                               (result >= ServiceMetaData::SFW_ERROR_NO_SERVICE && result <= ServiceMetaData::SFW_ERROR_PARSE_INTERFACE)) {
                        importDir.remove(file);
                    //parsing was OK, error in registration
                    //keep the file in import folder to try again
                    } else if ( result ==  ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN ||
                                result == ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION) {
                        UNABLE_TO_OPEN_FILE = true;
                    }
                }
                // If any xml files can't be opened, try opening them again. The maximum number of
                // tries is set to 10. This is needed because sometimes even after a notification
                // of directory change has been received from the file system watcher, the newly
                // added xml files might not be available yet(i.e. copying hasn't finished yet, etc...)
                if (UNABLE_TO_OPEN_FILE) {
                    if ( m_count < MAX_RETRY ) {
                        QTimer::singleShot(100, this, SLOT(onDirectoryChanged()));
                    }
                }

#ifdef __SYMBIAN32__
                m_fsSession.Close();
            }
#endif
            //TODO :return error if can not connect to file session
        }
    }
}

/*!
    Gets the metadata for the specified service
    @param aSearchCriteria service search criteria
    @return service metadata or empty object in case there is an error \a
            or there in no such service
*/
ServiceInfo ServiceResolver::getService(ServiceResolutionCriteria &aSearchCriteria)
{
    ServiceInfo service;
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(m_privatePath);
#endif
    int dbResult = database.open();
    if (dbResult ==  ServiceDatabase::SFW_ERROR_DB_RECREATED) {
        database.close();
        recoverDB();
        dbResult = database.open();
    }
    if (dbResult == 0) {
        //Search for the service using specified criteria
        if (database.getService(aSearchCriteria,service) == 0) {
            if (!pluginExists(service.serviceDllFileName())) {
                //dll does not exist, service might have been uninstalled, remove service record from DB
#ifdef __SYMBIAN32__
                TInt fsResult(KErrNone);
                fsResult = m_fsSession.Connect();
                if (fsResult == KErrNone) {
#endif
                    if (completeUnistallation(database, service.serviceName(), service.serviceDllFileName(), service.xmlFileName()) == SFW_ERROR_NoError) {
                        //clear service object
                        service.setServiceDllFileName("");
                        service.setServiceName("");
                        service.setInterfaceVersion("");
                        service.setXmlFileName("");
                        service.setInterfaceCapabilities("");
                        service.setServiceVersion("");
                    } else {
                        //TODO : send error to handler
                    }
#ifdef __SYMBIAN32__
                    m_fsSession.Close();
                }
#endif
            }
        }
        database.close();
    }
    return service;
}



/*!
    Remove DB records and XML file (from registered folder) for the unistalled service
    @param database database object opened once for all service providers
    @param serviceName service name
    @param dllPath service .dll file path
    @param xmlFileName registry xml filename
    @return error code or SFW_ERROR_NoError if successfully completed cleaning DB and registered folder
*/
int ServiceResolver::completeUnistallation(ServiceDatabase &database, const QString &serviceName,
                                           const QString &dllPath, const QString &xmlFileName)
{
    int result(SFW_ERROR_NoError);
    QString dllDrive = dllPath.section(DRIVE_SEPARATOR, 0, 0);
#ifdef __SYMBIAN32__
    //check if service drive exists (for cases such as when a memory card has been removed temporarily)
    RFile file;
    TDriveList drivelist;
    TChar driveLetter;
    if ((result = m_fsSession.DriveList(drivelist)) == KErrNone) {
        result = ServiceResolver::SFW_ERROR_SERVICE_INSTALLED_DRIVED_HAS_BEEN_REMOVED;
        for (TInt driveNumber=EDriveA;driveNumber<=EDriveZ;++driveNumber)  {
            if (drivelist[driveNumber]) {
                if (m_fsSession.DriveToChar(driveNumber,driveLetter) == KErrNone) {
                    QString driveStringValue(QChar((uint)driveLetter.GetUpperCase()));
                    if (driveStringValue == dllDrive){
                        result = SFW_ERROR_NoError;
                        break;
                    }
                }
            }
        }
    }
#endif
    if (result == SFW_ERROR_NoError) {
        if (database.deleteService(serviceName) != 0) {
            SFW_DEBUG1(ERROR_CAN_NOT_REMOVE_SERVICE_ENTRY_FROM_DB_MESSAGE);
            result = ServiceResolver::SFW_ERROR_CAN_NOT_REMOVE_SERVICE_ENTRY_FROM_DB;
        } else {
            SFW_DEBUG1(REMOVED_SERVICE_ENTRY_FROM_DB_MESSAGE);
            QFile registeredXmlFile(m_registeredFolder + xmlFileName);
            if (registeredXmlFile.exists()) {
                if (!registeredXmlFile.remove()) {
                    SFW_DEBUG1(ERROR_CAN_NOT_DELETE_REGISTERED_XML_FILE_MESSAGE);
                    result = ServiceResolver::SFW_ERROR_CAN_NOT_DELETE_REGISTERED_XML_FILE;
                }
            } else {
                SFW_DEBUG1(ERROR_CAN_NOT_FIND_REGISTERED_XML_FILE_MESSAGE);
            }
        }
    }
    return result;
}


int ServiceResolver::completeBinaryFilePath(QString& dllName)
{
    int result(0);
    if (dllName.indexOf(".") == -1) {
        dllName += m_libraryExt;
    }
    // In case the dllName is not a path, construct full path to the
    // service dll file
    if ((dllName.indexOf(PATH_SEPARATOR) == -1) &&
        (dllName.indexOf(QT_PATH_SEPARATOR) == -1)) {
        //prepend service name with 'lib' on linux
#ifdef __linux__
        QString libPrefix = KCwrtSvcProvInstallFolder;
        libPrefix += LIB_PREFIX;
        dllName.prepend(libPrefix);
#endif
#ifdef __SYMBIAN32__
        //find the installation drive
        RFile file;
        TDriveList drivelist;
        TChar driveLetter;
        if ((result = m_fsSession.DriveList(drivelist)) == KErrNone) {
            result = ServiceResolver::SFW_ERROR_DLL_FILE_NOT_FOUND;
            for (TInt driveNumber=EDriveA;driveNumber<=EDriveZ;++driveNumber) {
                if (drivelist[driveNumber] && m_fsSession.DriveToChar(driveNumber,driveLetter) == KErrNone) {
                    QString driveStringValue(QChar((uint)driveLetter.GetUpperCase()));
                    QString stubPath;
                    stubPath.append(driveStringValue);
                    stubPath.append(QT_PLUGIN_FOLDER);
                    stubPath.append(dllName.section(WIN_LIB_EXT, 0, 0));
                    stubPath.append(QTPLUGIN_EXT);
                    QFile stub(stubPath);
                    if (stub.exists()) {
#ifndef __WINSCW__
                        QString dllPath;
                        dllPath.append(driveStringValue);
                        dllPath.append(SYMBIAN_BIN_FOLDER);
                        dllPath.append(dllName);
                        QFile dllFile(dllPath);
                        if (dllFile.exists()) {
#endif
                            dllName = stubPath;
                            result = 0;
                            break;
#ifndef __WINSCW__
                        }
#endif
                    }
                }
            }
        }
#else
        dllName = QDir::current().absoluteFilePath(dllName);
#endif
    }
    return result;
}

int ServiceResolver::checkBinaryVersion(const QString& dllPath, const QString& xmlDllVersion) {
    int error(0);

#if defined(__SYMBIAN32__) && !defined(__WINSCW__)
    //dllPath is the path to stub, tmpPath will keep the real dll file path
    QString tmpPath(dllPath);
    //replace :\\resource\\qt\\sfwplugin\\ with :\\sys\\bin\\ in tmpPath
    tmpPath.replace(QString(QT_PLUGIN_FOLDER), QString(SYMBIAN_BIN_FOLDER));
    //replace .qtPlugin with .dll in tmpPath
    tmpPath = tmpPath.left(tmpPath.lastIndexOf(QTPLUGIN_EXT, -1)).append(WIN_LIB_EXT);
    RLibrary::TInfo info;
    TPckg<RLibrary::TInfo> infoBuf(info);
    TPtrC ptr = reinterpret_cast<const TText*> (tmpPath.constData());
    error = RLibrary::GetInfo(ptr, infoBuf); //get filename with full path
    if (error == KErrNone) {
        TUint major = info.iModuleVersion >> 16;
        TUint minor = info.iModuleVersion & 0x0000ffffu;
        QStringList xmlVersionList = xmlDllVersion.split(VERSION_SEPARATOR);
        // Symbian binary version only has major and minor components, so compress
        // Qt's minor and patch values into the minor component. Since Symbian's minor
        // component is a 16 bit value, only allow 8 bits for each to avoid overflow.
        if ( xmlVersionList.size() > 2 && xmlVersionList[0].toInt() <= 0xFFFF &&
                xmlVersionList[1].toInt() <= 0xFF && xmlVersionList[2].toInt() <= 0xFF ) {
            if (major != xmlVersionList[0].toInt() ||
                minor != ((xmlVersionList[1].toInt() << 8 )+ xmlVersionList[2].toInt()))
                error = ServiceResolver::SFW_ERROR_DLL_VERSION_NOT_MATCH_SERVICE_VERSION_IN_XML;
            else
                error = 0;
        } else {
            error = ServiceResolver::SFW_ERROR_SERVICE_VERSION_NOT_CORRECT_IN_XML;
        }
    } else {
        error = ServiceResolver::SFW_ERROR_READING_LIBRARY_INFO;
    }
    if (error == ServiceResolver::SFW_ERROR_DLL_VERSION_NOT_MATCH_SERVICE_VERSION_IN_XML ||
        error == ServiceResolver::SFW_ERROR_SERVICE_VERSION_NOT_CORRECT_IN_XML){
        //remove plugin .dll and stub files
        QFile::remove(dllPath);
        QFile::remove(tmpPath);
    }
#else
    //for windows platform we do not check version matches for now.
    Q_UNUSED(dllPath);
    Q_UNUSED(xmlDllVersion);
#endif
    return error;
}

/*!
    Registers a new service \n
    At each time at most one version of a service is registered in resolver db. \n
    Reinstalling the same version of a service will be ignored. \n
    New versions of a service can be registered. Basically the old version will be uninstalled and removed from DB before
    new version being installed. New XML file name should be the same as old version. \n
    @param aXmlFilePath full path to the service description XML file
    @return true if operation was successful, false otherwise
*/
int  ServiceResolver::registerService( const QString &aXmlFilePath)
{
    int result(0);
    // Parse xml file
    if ( !m_parser ) {
        m_parser = new ServiceMetaData(aXmlFilePath);
    } else {
        m_parser->setXMLFilePath(aXmlFilePath);
    }

    if ( m_parser->extractMetadata()) {
        QString dllPath(m_parser->getServiceMetadata()->filePath());
        if ((result = completeBinaryFilePath(dllPath)) == 0) {

#ifdef __SYMBIAN32__
            //check dll version matches the service version specified in xml file
            result = checkBinaryVersion(dllPath, m_parser->getServiceMetadata()->version());
            if (result == 0 || result == ServiceResolver::SFW_ERROR_DLL_VERSION_NOT_MATCH_SERVICE_VERSION_IN_XML ||
                               result == SFW_ERROR_DLL_VERSION_NOT_MATCH_SERVICE_VERSION_IN_XML) {
#endif
                ServiceDatabase database;
#ifdef __SYMBIAN32__
                database.setDatabasePath(m_privatePath);
#endif
                int dbResult = database.open();
                if (dbResult ==  ServiceDatabase::SFW_ERROR_DB_RECREATED) {
                    database.close();
                    recoverDB();
                    dbResult = database.open();
                }
#ifdef __SYMBIAN32__
                if (result == 0) {
#endif
                    m_parser->getServiceMetadata()->setFilePath(dllPath);
                    if (dbResult == 0) {
                        result = database.registerService(*(m_parser->getServiceMetadata()));
                        database.close();
                    }
#ifdef __SYMBIAN32__
                } else {
                    //when a new version of service is installed but registration fails due to xml version and binary version mismatch,
                    //entries for old registration have to be deleted from DB and also old service binary file (if applicable)
                    if (dbResult == 0) {
                        QString serviceName = m_parser->getServiceMetadata()->name();
                        database.deleteService(serviceName);
                        database.close();
                    }
                }
            }
#endif
        }
    } else {
        result = m_parser->getLatestError();
    }
    return result;
}

/*!
    Gets the list of registered services
    @return list of service metadata or empty object in case there is an error or
                there is no registered service
*/
ServiceInfoList ServiceResolver::listServices()
{
    ServiceInfoList serviceList;
    ServiceDatabase database;
#ifdef __SYMBIAN32__
    database.setDatabasePath(m_privatePath);
#endif
    int dbResult = database.open();
    if (dbResult ==  ServiceDatabase::SFW_ERROR_DB_RECREATED) {
        database.close();
        recoverDB();
        dbResult = database.open();
    }
    if (dbResult == 0) {
        database.listServices(serviceList);
        int listCount = serviceList.count();
        int i = 0;
#ifdef __SYMBIAN32__
        TInt fsResult(KErrNone);
        fsResult = m_fsSession.Connect();
        if (fsResult == KErrNone) {
#endif
            while (i < listCount) {
                QString dllFilePath = serviceList.getService(i).serviceDllFileName();
                QString serviceName = serviceList.getService(i).serviceName();
                if (!pluginExists(dllFilePath)){
                    if (completeUnistallation(database, serviceName, dllFilePath, serviceList.getService(i).xmlFileName()) != SFW_ERROR_NoError) {
                        //TODO : send error to handler
                    }
                    serviceList.deleteService(i);
                    if (i == --listCount) {
                        break;
                    }
                } else {
                    if (++i == listCount) {
                        break;
                    }
                }
            }
#ifdef __SYMBIAN32__
            m_fsSession.Close();
        }
#endif
        database.close();
    }
    return serviceList;
}


/*!
    Gets the service version for the specified service
    @param aServiceName service name
    @return service version
*/
QString ServiceResolver::getServiceVersion(const QString &aServiceName)
{
    ServiceDatabase database;
    QString serviceVersion;
    QString filePath;

#ifdef __SYMBIAN32__
    database.setDatabasePath(m_privatePath);
#endif
    int dbResult = database.open();
    if (dbResult ==  ServiceDatabase::SFW_ERROR_DB_RECREATED) {
        database.close();
        recoverDB();
        dbResult = database.open();
    }
    if (dbResult == 0) {
        if ((database.getServiceVersion(aServiceName, serviceVersion, filePath) != 0)
           || (serviceVersion.isEmpty())) {
            serviceVersion = SERVICE_NOT_SUPPORT;
        } else { // check if file exist or not
            if (!pluginExists(filePath)) {
                serviceVersion = SERVICE_NOT_SUPPORT;
            }
        }
        database.close();
    }
    return serviceVersion;
}


/*!
    Verifies if service provider's binary file exists
    @param filePath service provider's binary file path
    @return true if file exists
*/
bool ServiceResolver::pluginExists(QString filePath)
{
    bool retVal = false;
    if (QFileInfo(filePath).exists())
        retVal = true;
    return retVal;
}

/*!
    Handles request from service handler
    The currently supported ooperations are: \n
    GETSERVICEREQ  for retirieving service information \n
    LISTSERVICEREQ for listing all available services \n
    @param aRequest IPC request
    @return true if request was handled successfuly, false otherwise
*/
bool ServiceResolver::handleRequest( ServiceIPCRequest *aRequest )
{
    bool bReturn(false);
    bool vidValid(true);
#ifdef __SYMBIAN32__
    ClientInfo* client = const_cast<ClientInfo*>(aRequest->clientInfo());
    qint64 vendorId = client->vendorId();
    // check if the calling process's vendor ID is Nokia VID
    if (vendorId != 0x101FB657) {
        vidValid = false;
    }
#endif
    if (vidValid) {
        if (aRequest->getOperation() == GETSERVICEREQ ) {
            ServiceResolutionCriteria criteria;
            if (criteria.deserialize(aRequest->getData())) {
                //get metadata info of requested interface service provider
                ServiceInfo service = getService(criteria);
                aRequest->write(service.serialize());
                bReturn = true;
            }
        } else if (aRequest->getOperation() == LISTSERVICEREQ) {
            ServiceInfoList list = listServices();
            aRequest->write( list.serialize() );
            bReturn = true;
        } else if (aRequest->getOperation() == GETSERVICEVERSIONREQ) {
            QString serviceName(aRequest->getData());
            QString serviceVersion = getServiceVersion(serviceName);
            aRequest->write( serviceVersion.toAscii() );
            bReturn = true;
        } else {
            qDebug("Unknown Request");
        }
    }
    aRequest->completeRequest();
    return bReturn;
}

/*!
    Handles request cancelation
    @param aRequest IPC request
    @return void
*/
void ServiceResolver::handleCancelRequest( ServiceIPCRequest* /*aRequest*/ )
{
    // No outstanding requests
}

/*!
 * From MServiceIPCObserver
 * \see MServiceIPCObserver::handleClientConnect( ClientInfo *aNewClient)
 */
void ServiceResolver::handleClientConnect( ClientInfo* /*aNewClient*/ )
{
}

/*!
 * From MServiceIPCObserver
 * \see MServiceIPCObserver::handleClientDisconnect(const ClientInfo& aClient)
 */
void ServiceResolver::handleClientDisconnect( ClientInfo* /*aClient*/ )
{
}
/*!
 * Close server
 */
void ServiceResolver::closeServer()
{
    QCoreApplication::quit();
}

}
// END OF FILE
