/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include "serviceinfolist.h"

#define SERVICE_INFO_SEP "<?>"

namespace WRT {

/*!
   \class ServiceInfoList
    Utility class that encapsulates a list of service metadata \n
    Class is used by service resolver and service handler
*/

/*!
    Constructor
*/
ServiceInfoList::ServiceInfoList()
{

}

/*!
    Destructor
*/
ServiceInfoList::~ServiceInfoList(void)
{
     m_services.clear();
}

/*!
    Gets the number of m_services in the list
    @return number of m_services
*/
int ServiceInfoList::count()
{
    return m_services.count();
}

/*!
    Gets the metadata for the service with the specified index in the list
    Method asserts in case of index out of bounds
    @param aServiceIndex service index in the list
    @return service metadata
*/
ServiceInfo ServiceInfoList::getService(int aServiceIndex)
{
    Q_ASSERT((aServiceIndex >= 0) && (aServiceIndex < count()));
    return m_services[aServiceIndex];
}


/*!
    Remove service metadata info for a service with the specified index in the list
    Method asserts in case of index out of bounds
    @param aServiceIndex service index in the list
    @return void
*/
void ServiceInfoList::deleteService(int aServiceIndex)
{
    m_services.removeAt(aServiceIndex);
}

/*!
    Adds service to the list
    @param aServiceInfo aServiceInfo
    @return void
*/
void ServiceInfoList::addServiceInfo(const ServiceInfo& aServiceInfo)
{
    m_services.append(aServiceInfo);
}

/*!
    Creates data stream from the list of service metadata
    @return serialized data stream
*/
QByteArray ServiceInfoList::serialize()
{
    QByteArray stream;
    int count = m_services.count();
    if (count > 0) {
        QString strCount;
        strCount.setNum(count);
        stream.append(strCount.toAscii());
        stream.append(SERVICE_INFO_SEP);
        for (int index = 0;index < count;index++) {
            stream.append(m_services[index].serialize());
            stream.append(SERVICE_INFO_SEP);
        }
    }
    return stream;
}

/*!
    Extracts list of service metadata from input stream
    @return true if operation successful, false otherwise
*/
bool ServiceInfoList::deserialize(const QByteArray& aDataStream)
{
    bool bValid(false);
    QString strm_services(aDataStream);
    QStringList serviceData = strm_services.split(SERVICE_INFO_SEP);
    if (serviceData.count() > 1) {
        int m_servicesCount = serviceData[0].toInt();
        bValid = true;
        for (int index = 0;index < m_servicesCount;index++) {
            ServiceInfo serviceInfo;
            if (serviceInfo.deserialize(serviceData[index+1].toAscii())) {
                m_services.append(serviceInfo);
            } else {
                m_services.clear();
                bValid = false;
                break;
            }
        }
    }
    return bValid;
}

}
