/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include <QtTest>
#include <QtCore>
#include "servicemetadata.h"
#include "testhelpers.h"

using namespace WRT;

#ifdef __SYMBIAN32__
    static const char KTESTDATAFOLDER[]         = "c:/testdata/";
#else
    static const char KTESTDATAFOLDER[]         = "testdata/";
#endif

class ServiceMetadataTest: public QObject
{
    Q_OBJECT

private slots:
    void initTestCase();
    void parseServiceXMLTest();
    void InvalidFileTest0();
    void InvalidFileTest1();
    void InvalidFileTest2();
    void InvalidFileTest3();
    void InvalidFileTest4();
    void InvalidFileTest5();
    void InvalidFileTest6();
    void InvalidFileTest7();
    void InvalidFileTest8();
    void InvalidFileTest9();
    void InterfaceNoCapabilityTest();
    void cleanupTestCase();

private:
    ServiceMetaData* parser;
    QDir dir;

};

void ServiceMetadataTest::initTestCase()
{
    __countTestsHelper__(this);
    dir = QDir(KTESTDATAFOLDER);
    parser = new ServiceMetaData(dir.absoluteFilePath("ServiceTest4.xml"));
}


void ServiceMetadataTest::parseServiceXMLTest()
{
    QCOMPARE(parser->extractMetadata(),true);
    SFWService* service = parser->getServiceMetadata();
    QCOMPARE(service->name(), QString("TestService"));
    QCOMPARE(service->version(), QString("1.4.56"));
    QCOMPARE(service->description(), QString("Test service description"));
#ifdef __SYMBIAN32__
    QCOMPARE(service->filePath(), QString("testservice"));
#else
    QCOMPARE(service->filePath(), QString(KTESTDATAFOLDER) + "testservice.dll");
#endif
    QCOMPARE(service->interfaceCount(), 5);
    SFWInterface* aInterface = service->getInterface(0);
    QCOMPARE(aInterface->name(), QString("com.nokia.IDownloader"));
    QCOMPARE(aInterface->version(), QString("1.0"));
    QStringList capabilitiesList = aInterface->capabalitiesList();
    QCOMPARE(capabilitiesList.count(),2);
    QCOMPARE(capabilitiesList[0],QString("ReadUserData"));
    QCOMPARE(capabilitiesList[1],QString("WriteUserData"));
    QCOMPARE(aInterface->description(), QString("Interface that provides download support"));

    aInterface = service->getInterface(1);
    QCOMPARE(aInterface->name(),QString("com.nokia.ILocation"));
    QCOMPARE(aInterface->version(), QString("1.0"));
    capabilitiesList = aInterface->capabalitiesList();
    QCOMPARE(capabilitiesList.count(),0);
    QCOMPARE(aInterface->description(), QString("Interface that provides location support"));

    aInterface = service->getInterface(2);
    QCOMPARE(aInterface->name(),QString("com.nokia.ISysInfo"));
    capabilitiesList = aInterface->capabalitiesList();
    QCOMPARE(capabilitiesList.count(),1);
    QCOMPARE(capabilitiesList[0],QString("ReadUserData"));
    QCOMPARE(aInterface->description(), QString("Interface that provides system information support"));


    aInterface = service->getInterface(3);
    QCOMPARE(aInterface->name(),QString("com.nokia.ISendMessage"));
    capabilitiesList = aInterface->capabalitiesList();
    QCOMPARE(capabilitiesList.count(),1);
    QCOMPARE(capabilitiesList[0],QString("ReadUserData"));
    QCOMPARE(aInterface->description(), QString("Interface that provides message sending support"));

    aInterface = service->getInterface(4);
    QCOMPARE(aInterface->name(), QString("com.nokia.IReceiveMessage"));
    capabilitiesList = aInterface->capabalitiesList();
    QCOMPARE(capabilitiesList.count(),1);
    QCOMPARE(capabilitiesList[0],QString("WriteUserData"));
    QCOMPARE(aInterface->description(), QString("Interface that provides message receiving support"));

    aInterface = service->getInterface(5);
    if (aInterface != NULL) {
        QFAIL("Incorrect interface return");
    }
}

void ServiceMetadataTest::InvalidFileTest0()
{
    parser->setXMLFilePath("!#@!CSC");
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_UNABLE_TO_OPEN_FILE);
}


void ServiceMetadataTest::InvalidFileTest1()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test11.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_INVALID_XML_FILE);
}

void ServiceMetadataTest::InvalidFileTest2()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test12.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_NO_SERVICE);
}

void ServiceMetadataTest::InvalidFileTest3()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test3.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_NO_SERVICE_NAME);
}

void ServiceMetadataTest::InvalidFileTest4()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test5.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_NO_SERVICE_FILEPATH);
}

void ServiceMetadataTest::InvalidFileTest5()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test7.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_NO_SERVICE_INTERFACE);
}

void ServiceMetadataTest::InvalidFileTest6()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test8.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_NO_INTERFACE_NAME);
}

void ServiceMetadataTest::InvalidFileTest7()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test4.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_NO_INTERFACE_VERSION);
}

void ServiceMetadataTest::InvalidFileTest8()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test9.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_PARSE_SERVICE);
}

void ServiceMetadataTest::InvalidFileTest9()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test10.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),(int)ServiceMetaData::SFW_ERROR_NO_SERVICE_INTERFACE);
}

void ServiceMetadataTest::InterfaceNoCapabilityTest()
{
    parser->setXMLFilePath(dir.absoluteFilePath("Test6.xml"));
    QCOMPARE(parser->extractMetadata(),true);
}

/*void ServiceMetadataTest::InvalidFileTest13()
{
    parser->setXMLFile(dir.absoluteFilePath("Test13.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_NO_METHOD_DESCRIPTION);
}

void ServiceMetadataTest::InvalidFileTest14()
{
    parser->setXMLFile(dir.absoluteFilePath("Test14.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_NO_METHOD_RESPONSE);
}

void ServiceMetadataTest::InvalidFileTest15()
{
    parser->setXMLFile(dir.absoluteFilePath("Test15.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_INVALID_METHOD_RESPONSE_VALUE);
}

void ServiceMetadataTest::InvalidFileTest16()
{
    parser->setXMLFile(dir.absoluteFilePath("Test16.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_NO_METHOD_DETAIL_DESCRIPTION);
}

void ServiceMetadataTest::InvalidFileTest17()
{
    parser->setXMLFile(dir.absoluteFilePath("Test17.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_NO_CALLBACK_NAME);
}
void ServiceMetadataTest::InvalidFileTest18()
{
    parser->setXMLFile(dir.absoluteFilePath("Test18.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_NO_CALLBACK_DESCRIPTION);
}

void ServiceMetadataTest::InvalidFileTest19()
{
    parser->setXMLFile(dir.absoluteFilePath("Test19.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_NO_CALLBACK_DETAIL_DESCRIPTION);
}

void ServiceMetadataTest::InvalidFileTest20()
{
    parser->setXMLFile(dir.absoluteFilePath("Test20.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_NO_PARAMETER_NAME);
}

void ServiceMetadataTest::InvalidFileTest21()
{
    parser->setXMLFile(dir.absoluteFilePath("Test21.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_NO_PARAMETER_TYPE);
}

void ServiceMetadataTest::InvalidFileTest22()
{
    parser->setXMLFile(dir.absoluteFilePath("Test22.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_NO_PARAMETER_DESCRIPTION);
}

void ServiceMetadataTest::InvalidFileTest23()
{
    parser->setXMLFile(dir.absoluteFilePath("Test23.xml"));
    QCOMPARE(parser->extractMetadata(),false);
    QCOMPARE(parser->getLatestError(),ERROR_INVALID_MANDATORY_VALUE);
}



void ServiceMetadataTest::parseDetailedServiceXMLTest()
{
    parser->setXMLFile(dir.absoluteFilePath("Test.xml"));
    QCOMPARE(parser->extractMetadata(),true);
    QCOMPARE(parser->name(), QString("com.nokia.TestServices"));
    QCOMPARE(parser->version(), QString("1.0.456"));
    QCOMPARE(parser->capabilities(), QString("ReadUserData,WriteUserData"));
    QCOMPARE(parser->description(), QString("Test service description"));
    QCOMPARE(parser->interfaceCount(), 2);

    CSFWInterface* interface = parser->getInterface(0);
    QCOMPARE(interface->name(), QString("Test Interface 1"));
    QCOMPARE(interface->description(), QString("First test interface description"));
    QCOMPARE(interface->methodCount(), 2);
    QCOMPARE(interface->callbackCount(), 1);

    CSFWMethod* method = interface->getMethod(0);
    QCOMPARE(method->name(), QString("Method 1"));
    QCOMPARE(method->description(), QString("Sample test method 1"));
    QCOMPARE(method->response(), QString("synchronous"));
    QCOMPARE(method->detailDescription(), QString("Test method 1 of the interface"));

    QCOMPARE(method->inputParametersCount(),3);
    QCOMPARE(method->outputParametersCount(),1);

    CSFWParameter* param = method->getInputParameter(0);
    QCOMPARE(param->name(), QString("inValue1"));
    QCOMPARE(param->type(), QString("String"));
    QCOMPARE(param->description(), QString("First in parameter of method 1"));
    QCOMPARE(param->mandatory(), true);

    param = method->getInputParameter(1);
    QCOMPARE(param->name(), QString("inValue2"));
    QCOMPARE(param->type(), QString("Int"));
    QCOMPARE(param->description(), QString("Second in parameter of method 1"));
    QCOMPARE(param->mandatory(), false);

    param = method->getInputParameter(2);
    QCOMPARE(param->name(), QString("inValue3"));
    QCOMPARE(param->type(), QString("Bool"));
    QCOMPARE(param->description(), QString("Third in parameter of method 1"));
    param = method->getInputParameter(2);

    param = method->getInputParameter(3);
    if (param != NULL) {
        QCOMPARE(0,1);
    }

    param = method->getOutputParameter(0);
    QCOMPARE(param->name(), QString("returnValue1"));
    QCOMPARE(param->type(), QString("Object"));
    QCOMPARE(param->description(), QString("return value parameter of Method1"));
    QCOMPARE(param->mandatory(), true);

    param = method->getOutputParameter(2);
    if (param != NULL) {
        QCOMPARE(0,1);
    }

    method = interface->getMethod(1);
    QCOMPARE(method->name(), QString("Method 2"));
    QCOMPARE(method->description(), QString("Sample test method 2"));
    QCOMPARE(method->response(), QString("asynchronous"));
    QCOMPARE(method->detailDescription(), QString("Test method 2 of the interface"));
    QCOMPARE(method->inputParametersCount(),2);
    QCOMPARE(method->outputParametersCount(),1);

    param = method->getInputParameter(0);
    QCOMPARE(param->name(), QString("inValue21"));
    QCOMPARE(param->type(), QString("String"));
    QCOMPARE(param->description(), QString("First in parameter of method 2"));
    QCOMPARE(param->mandatory(), true);

    param = method->getInputParameter(1);
    QCOMPARE(param->name(), QString("inValue22"));
    QCOMPARE(param->type(), QString("Int"));
    QCOMPARE(param->description(), QString("Second in parameter of method 2"));
    QCOMPARE(param->mandatory(), false);

    param = method->getOutputParameter(0);
    QCOMPARE(param->name(), QString("returnValue2"));
    QCOMPARE(param->type(), QString("Object"));
    QCOMPARE(param->description(), QString("return value parameter of Method2"));
    QCOMPARE(param->mandatory(), true);

    method = interface->getMethod(2);
    if (method != NULL) {
        QCOMPARE(0,1);
    }

    method = interface->getCallback(0);
    QCOMPARE(method->name(), QString("Test CallBack 1"));
    QCOMPARE(method->description(), QString("Notification callback 1."));
    QCOMPARE(method->inputParametersCount(),2);
    QCOMPARE(method->outputParametersCount(),0);

    param = method->getInputParameter(0);
    QCOMPARE(param->name(), QString("Error"));
    QCOMPARE(param->type(), QString("Int32"));
    QCOMPARE(param->description(), QString("Error code if any."));
    QCOMPARE(param->mandatory(), true);

    param = method->getInputParameter(1);
    QCOMPARE(param->name(), QString("Error Value"));
    QCOMPARE(param->type(), QString("Variant"));
    QCOMPARE(param->description(), QString("Error description"));
    QCOMPARE(param->mandatory(), true);

    method = interface->getCallback(1);
    if (method != NULL) {
        QCOMPARE(0,1);
    }

    interface = parser->getInterface(1);
    QCOMPARE(interface->name(),QString("Test Interface 2"));
    QCOMPARE(interface->description(), QString("Test interface description 2"));
    QCOMPARE(interface->methodCount(), 1);
    QCOMPARE(interface->callbackCount(), 0);

    method = interface->getMethod(0);
    QCOMPARE(method->name(), QString("Method"));
    QCOMPARE(method->description(), QString("Sample test method of  interface 2"));
    QCOMPARE(method->response(), QString("asynchronous"));
    QCOMPARE(method->detailDescription(), QString("Test method of the interface 2"));
    QCOMPARE(method->inputParametersCount(),1);
    QCOMPARE(method->outputParametersCount(),1);

    param = method->getInputParameter(0);
    QCOMPARE(param->name(), QString("inValue"));
    QCOMPARE(param->type(), QString("String"));
    QCOMPARE(param->description(), QString("First in parameter of method"));
    QCOMPARE(param->mandatory(), true);

    param = method->getOutputParameter(0);
    QCOMPARE(param->name(), QString("returnValue"));
    QCOMPARE(param->type(), QString("Object"));
    QCOMPARE(param->description(), QString("return value parameter of Method"));
    QCOMPARE(param->mandatory(), true);

    interface = parser->getInterface(2);
    if (interface != NULL) {
        QCOMPARE(0,1);
    }
}*/

void  ServiceMetadataTest::cleanupTestCase()
{
    delete parser;
    parser = 0;
}

QTEST_MAIN(ServiceMetadataTest)
#include "servicemetadatatest.moc"
