/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QFile>
#include <sfwlog.h>
#include "servicemetadatadefines.h"
#include "servicemetadata.h"

namespace WRT {

#ifdef __linux__
static const char  PATH_SEPARATOR[] = "/";
#else
static const char  PATH_SEPARATOR[] = "\\";
#endif

/*!
    \class ServiceMetaData

    Utility class (used by service database) that offers support for
    parsing metadata service xml registry file during service registration. \n

    It uses QXMLStreamReader class for parsing. Supproted Operations are:
        - Parse the service and interfaces defined in XML file
*/

/*!
 *  Class constructor
 *
 * @param aXmlFilePath path to the xml file that describes the service.
 */
ServiceMetaData::ServiceMetaData(const QString &aXmlFilePath)
{
    m_serviceInfo = new SFWService();
    m_xmlFilePath = aXmlFilePath;
    m_serviceInfo->setXMLFileName(m_xmlFilePath.section(PATH_SEPARATOR, -1));
    m_latestError = 0;
}

/*!
 *  Class destructor
 *
 */
ServiceMetaData::~ServiceMetaData(void)
{
    if (m_serviceInfo) {
        m_serviceInfo->clear();
        delete(m_serviceInfo);
    }
}

/*!
 *  Sets the path to the xml file
 *
 * @param aXmlFilePath path to the xml file that describes the service.
 */
void ServiceMetaData::setXMLFilePath(const QString &aXmlFilePath)
{
    m_xmlFilePath = aXmlFilePath;
    m_serviceInfo->clear();
    m_serviceInfo->setXMLFileName(m_xmlFilePath.section(PATH_SEPARATOR, -1));
    m_latestError =0;
}

/*!
    Gets the latest parsing error \n
    @return parsing error(negative value) or 0 in case there is none
 */
int ServiceMetaData::getLatestError()
{
    return m_latestError;
}


/*!
    Parses the file and extracts the service metadata \n
    Custom error codes: \n
    SFW_ERROR_UNABLE_TO_OPEN_FILE in case can not open the XML file \n
    SFW_ERROR_INVALID_XML_FILE in case service registry is not a valid XML file \n
    SFW_ERROR_NO_SERVICE in case XML file has no service tag\n
    @return true if the metadata was read properly, false if there is an error
 */
bool ServiceMetaData::extractMetadata()
{
    m_latestError = 0;
    m_serviceInfo->clear();
    QXmlStreamReader xmlReader;
    QFile file(m_xmlFilePath);
    bool parseError = false;

    //Open xml file
    if (!file.open(QIODevice::ReadOnly)) {
        SFW_DEBUG1(ERROR_UNABLE_TO_OPEN_FILE_MESSAGE);
        m_latestError = ServiceMetaData::SFW_ERROR_UNABLE_TO_OPEN_FILE;
        parseError = true;
    } else {
        //Load xml content
        xmlReader.setDevice(&file);
        // Read XML doc
        while (!xmlReader.atEnd() && !parseError) {
            xmlReader.readNext();
            //Found a <service> node, read service related metadata
            if (xmlReader.isStartElement() && xmlReader.name() == METADATA_SERVICE_TAG) {
                if (!processServiceElement(xmlReader)) {
                    parseError = true;
                }
                break;
            }
            else if (xmlReader.isStartElement() && xmlReader.name() != METADATA_SERVICE_TAG) {
                m_latestError = ServiceMetaData::SFW_ERROR_NO_SERVICE;
                parseError = true;
            }
            else if (xmlReader.tokenType() == QXmlStreamReader::Invalid) {
                SFW_DEBUG1(ERROR_INVALID_XML_FILE_MESSAGE);
                m_latestError = ServiceMetaData::SFW_ERROR_INVALID_XML_FILE;
                parseError = true;
            }
        }
        file.close();
    }
    if (parseError) {
        m_serviceInfo->clear();
    }
    return !parseError;
}

/*!
    Parses and extracts the service metadata from the current xml <service> node \n
    Custom error codes: \n
    SFW_ERROR_PARSE_SERVICE in case can not parse service section in XML file \n
    SFW_ERROR_INVALID_XML_FILE in case XML file is not valid \n
    SFW_ERROR_NO_SERVICE_NAME in case no service name in XML file \n
    SFW_ERROR_NO_SERVICE_VERSION in case no service version in XML file \n
    SFW_ERROR_NO_SERVICE_FILEPATH in case no service file path in XML file \n
    SFW_ERROR_NO_SERVICE_INTERFACE in case no interface defined for service in XML file \n
    @param aXMLReader xml stream reader
    @return true if the metadata was read properly, false if there is an error
 */
bool ServiceMetaData::processServiceElement(QXmlStreamReader &aXMLReader)
{
    Q_ASSERT(aXMLReader.isStartElement() && aXMLReader.name() == METADATA_SERVICE_TAG);
    bool parseError = false;

    while (!parseError && !aXMLReader.atEnd()) {
        aXMLReader.readNext();
        if (aXMLReader.name() == METADATA_NAME) {
            m_serviceInfo->setName(aXMLReader.readElementText());
        } else if (aXMLReader.name() == METADATA_VERSION) {
            m_serviceInfo->setVersion(aXMLReader.readElementText());
        } else if (aXMLReader.name() == METADATA_SERVICE_FILEPATH) {
            m_serviceInfo->setFilePath(aXMLReader.readElementText());
        } else if (aXMLReader.name() == METADATA_DESCRIPTION) {
            m_serviceInfo->setDescription(aXMLReader.readElementText());
        //Found a <interface> node, read module related metadata
        } else if (aXMLReader.isStartElement() && aXMLReader.name() == METADATA_INTERFACE_TAG) {
            if (!processInterfaceElement(aXMLReader))
                parseError = true;
        //Found </service>, leave the loop
        } else if (aXMLReader.isEndElement() && aXMLReader.name() == METADATA_SERVICE_TAG) {
            break;
        } else if (aXMLReader.isEndElement() || aXMLReader.isStartElement()) {
            SFW_DEBUG1(ERROR_PARSE_SERVICE_MESSAGE);
            m_latestError = ServiceMetaData::SFW_ERROR_PARSE_SERVICE;
            parseError = true;
        } else if (aXMLReader.tokenType() == QXmlStreamReader::Invalid) {
            SFW_DEBUG1(ERROR_INVALID_XML_FILE_MESSAGE);
            m_latestError = ServiceMetaData::SFW_ERROR_INVALID_XML_FILE;
            parseError = true;
        }
    }

    if ((m_serviceInfo->name()).isEmpty()) {
        SFW_DEBUG1(ERROR_NO_SERVICE_NAME_MESSAGE);
        m_latestError = ServiceMetaData::SFW_ERROR_NO_SERVICE_NAME;
        parseError = true;
    } else if ((m_serviceInfo->version()).isEmpty()) {
        SFW_DEBUG1(ERROR_NO_SERVICE_VERSION_MESSAGE);
        m_latestError = ServiceMetaData::SFW_ERROR_NO_SERVICE_VERSION;
        parseError = true;
    } else if ((m_serviceInfo->filePath()).isEmpty()) {
        SFW_DEBUG1(ERROR_NO_SERVICE_FILEPATH_MESSAGE);
        m_latestError = ServiceMetaData::SFW_ERROR_NO_SERVICE_FILEPATH;
        parseError = true;
    } else if (m_serviceInfo->interfaceCount() == 0 && m_latestError == 0) {
        SFW_DEBUG1(ERROR_NO_SERVICE_INTERFACE_MESSAGE);
        m_latestError = ServiceMetaData::SFW_ERROR_NO_SERVICE_INTERFACE;
        parseError = true;
    }

    if (parseError) {
        m_serviceInfo->clear();
    }
    return !parseError;
}

/*!
    Parses and extracts the interface metadata from the current xml <interface> node \n
    Custome error codes: \n
    SFW_ERROR_NO_INTERFACE_NAME in case no interface name in XML file \n
    SFW_ERROR_NO_INTERFACE_VERSION in case no interface version in XML file \n
    SFW_ERROR_PARSE_INTERFACE in case error parsing interface section \n
    SFW_ERROR_INVALID_XML_FILE in case XML file is not valid \n
    @param aXMLReader xml stream reader
    @return true if the metadata was read properly, false if there is an error
 */
bool ServiceMetaData::processInterfaceElement(QXmlStreamReader &aXMLReader)
{
    Q_ASSERT(aXMLReader.isStartElement() && aXMLReader.name() == METADATA_INTERFACE_TAG);
    bool parseError = false;
    SFWInterface* aInterface = new SFWInterface();

    while (!parseError && !aXMLReader.atEnd()) {
        aXMLReader.readNext();
        //Read interface description
        if (aXMLReader.isStartElement() && aXMLReader.name() == METADATA_NAME) {
            aInterface->setName(aXMLReader.readElementText());
        } else if (aXMLReader.isStartElement() && aXMLReader.name() == METADATA_VERSION) {
            aInterface->setVersion(aXMLReader.readElementText());
        } else if (aXMLReader.isStartElement() && aXMLReader.name() == METADATA_DESCRIPTION) {
            aInterface->setDescription(aXMLReader.readElementText());
        } else if (aXMLReader.isStartElement() && aXMLReader.name() == METADATA_INTERFACE_CAPABILITIES) {
            aInterface->setCapabilities(aXMLReader.readElementText());
        //Found </interface>, leave the loop
        } else if (aXMLReader.isEndElement() && aXMLReader.name() == METADATA_INTERFACE_TAG) {
            break;
        } else if (aXMLReader.isStartElement() || aXMLReader.isEndElement()) {
            SFW_DEBUG1(ERROR_PARSE_INTERFACE_MESSAGE);
            m_latestError = ServiceMetaData::SFW_ERROR_PARSE_INTERFACE;
            parseError = true;
        } else if (aXMLReader.tokenType() == QXmlStreamReader::Invalid) {
            SFW_DEBUG1(ERROR_INVALID_XML_FILE_MESSAGE);
            m_latestError = ServiceMetaData::SFW_ERROR_INVALID_XML_FILE;
            parseError = true;
        }
    }

    if ((aInterface->name()).isEmpty()) {
        SFW_DEBUG1(ERROR_NO_INTERFACE_NAME_MESSAGE);
        m_latestError = ServiceMetaData::SFW_ERROR_NO_INTERFACE_NAME;
        parseError = true;
        delete(aInterface);
    } else if ((aInterface->version()).isEmpty()) {
        SFW_DEBUG1(ERROR_NO_INTERFACE_VERSION_MESSAGE);
        m_latestError = ServiceMetaData::SFW_ERROR_NO_INTERFACE_VERSION;
        parseError = true;
        delete(aInterface);
    } else if (!parseError) {
        m_serviceInfo->addInterface(aInterface);
    }

    return !parseError;
}


/*!
 *  Gets service metadata
 *
 * @return a SFWService object contating service and interface metadata
 */
SFWService* ServiceMetaData::getServiceMetadata()
{
    return m_serviceInfo;
}

}
