/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
/* resolver test module requires serviceresolver be running (otherwise 2 instances of resolver are launched and that fails some test cases),
   and a copy of resources/testserviceprovider.xml and access_policy and trust_policy files be in the bin dir
*/
#include <QtCore>
#include <QtTest/QtTest>
#include "servicehandler.h"
#include "testhelpers.h"

using namespace WRT;
const int KWAITINTERVAL                   = 2000;     // 1 seconds
#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
    static const char KIMPORTFOLDER[]         = "import/";
    static const char KREGISTEREDFOLDER[]     = "registered/";
#else
    static const char KIMPORTFOLDER[]         = "/usr/lib/cwrt/serviceproviders/import/";
    static const char KREGISTEREDFOLDER[]     = "/usr/lib/cwrt/serviceproviders/registered/";
#endif
static const char KDATABASEFILE[]         = "services.db";
static const char KTESTXMLFILE[]          = "ServiceTest.xml";
static const char KTESTXMLFILE2[]         = "ServiceTest1.xml";
static const char KTESTXMLFILE3[]         = "ServiceTest2.xml";
static const char KTESTXMLFILE4[]         = "Test2.xml";
static const char KTESTXMLFILE5[]         = "testserviceplugin.xml";

#ifdef __SYMBIAN32__
    static const char KTESTDATAFOLDER[]         = "c:/testdata/";
    static const char RESOLVER_DATABASE_PATH[]  = "c:/private/102829B8/";
#else
    static const char KTESTDATAFOLDER[]         = "testdata/";
#endif


class servicehandlermoduletest: public QObject
{
    Q_OBJECT
private slots:

    void initTestCase();

    void testRegisterService();
    void testGetServiceObject();
    void testGetServiceProviders();
    void testGetServiceVersion();

    void cleanupTestCase();



private:
    ServiceHandler* iHandler;
    void copyFileToImport( const QString aFile,
                           const QString aFolder=KIMPORTFOLDER );
    void deleteFiles( const QString aFolder );
};


void servicehandlermoduletest::initTestCase()
{
    __countTestsHelper__(this);
    // Cleanup database by just deleting the file
#ifndef __SYMBIAN32__
    QDir dir;
#else
    QDir dir(RESOLVER_DATABASE_PATH);
#endif
    // Cleanup files in import and registered folder
    deleteFiles(KIMPORTFOLDER);
    deleteFiles(KREGISTEREDFOLDER);
    if (dir.exists(KDATABASEFILE)){
        dir.remove(KDATABASEFILE);
    }
    QTest::qWait( KWAITINTERVAL *5 );
    iHandler = new ServiceHandler(NULL);
}


void servicehandlermoduletest::cleanupTestCase()
{
    // Cleanup files in import and registered folder
    deleteFiles(KIMPORTFOLDER);
    deleteFiles(KREGISTEREDFOLDER);
    // Cleanup database by just deleting the file
#ifndef __SYMBIAN32__
    QDir dir;
#else
    QDir dir(RESOLVER_DATABASE_PATH);
#endif
    if (dir.exists(KDATABASEFILE)){
        dir.remove(KDATABASEFILE);
    }
    delete iHandler;
    iHandler = NULL;
}

void servicehandlermoduletest::deleteFiles( const QString aFolder )
{

#ifndef __SYMBIAN32__
    QDir dir(QDir::currentPath());
#else
    QDir dir(RESOLVER_DATABASE_PATH);
#endif
    if ( dir.exists( aFolder )) {
        QVERIFY(dir.cd(aFolder));
        QDirIterator it(dir, QDirIterator::Subdirectories);
        while (it.hasNext()) {
            it.next();
            dir.remove(it.filePath());
        }
        QVERIFY(dir.cdUp());
        dir.rmdir(aFolder);
    }
}


void servicehandlermoduletest::copyFileToImport( const QString aFile,
                                      const QString aFolder/*=KIMPORTFOLDER*/ )
{
    // Copy a file into the import folder
#ifndef __SYMBIAN32__
    QDir dir;
#else
    QDir dir(RESOLVER_DATABASE_PATH);
#endif
    QVERIFY( dir.mkpath( aFolder ));
    QVERIFY(dir.cd(aFolder));
    QVERIFY(QFile::copy(KTESTDATAFOLDER + aFile,dir.absoluteFilePath(aFile)));

    // Check not read-only so that it can be deleted
    QFile file( aFolder + aFile );
    QFile::Permissions perm = file.permissions();
    if ( !perm.testFlag( QFile::WriteOwner ))
    {
        perm |= QFile::WriteOwner;
    }
    file.setPermissions( perm );
    file.close();
}


void servicehandlermoduletest::testRegisterService()
{
    //register the service
    copyFileToImport( KTESTXMLFILE );
    qDebug() << "Waiting for registration <--";
    QTest::qWait( KWAITINTERVAL);
    qDebug() << "Waiting for registration -->";
    //get list of providers to check service is registered
    QStringList list = iHandler->getServiceProviders();
    QVERIFY( list.count() == 1 );
    QCOMPARE(list[0],QString("TestService"));

    //try to register the same service again
    copyFileToImport(KTESTXMLFILE);
    QTest::qWait( KWAITINTERVAL );
    list = iHandler->getServiceProviders();
    QVERIFY( list.count() == 1 );
    for ( int i=0; i<list.count(); ++i )
        qDebug() << list[i];

    //register another vesion of the same service
    copyFileToImport(KTESTXMLFILE2);
    QTest::qWait( KWAITINTERVAL );
    list = iHandler->getServiceProviders();
    QVERIFY( list.count() == 2 );
    for ( int i=0; i<list.count(); ++i )
        qDebug() << list[i];
    QCOMPARE(list[1],QString("TestService1"));

    //try to register a service without .dll
    copyFileToImport(KTESTXMLFILE3);
    QTest::qWait( KWAITINTERVAL );
    list = iHandler->getServiceProviders();
    QVERIFY( list.count() == 2 );
    for ( int i=0; i<list.count(); ++i )
        qDebug() << list[i];

    //try to register a service with an invalid xml file
    copyFileToImport(KTESTXMLFILE4);
    QTest::qWait( KWAITINTERVAL );
    list = iHandler->getServiceProviders();
    QVERIFY( list.count() == 2 );
    for ( int i=0; i<list.count(); ++i )
        qDebug() << list[i];
}

void servicehandlermoduletest::testGetServiceObject()
{
    ServiceInterfaceInfo interfaceInfo={"com.nokia.IDownloader/1.0", 0};

    //get an instance of a registered service
    IServiceBase* ptr(NULL);
    //load testserviceplugin
    iHandler->getInterface(&ptr, "TestService2", interfaceInfo, true, NULL);
    QTest::qWait( KWAITINTERVAL);
    // return NULL since testserviceplugin does not implement IDownloader
    QVERIFY(!ptr);

    //try to get an instance of a service, invalid criteria
    ServiceInterfaceInfo interfaceInfo2={"", 0};

    iHandler->getInterface(&ptr, "TestService", interfaceInfo2, true, NULL);
    QVERIFY(!ptr);

    //try to get an instance of a registered service, invalid version
    ServiceInterfaceInfo interfaceInfo4={"com.nokia.ISimpleTypeTest/-1", 0};

    //load testserviceplugin
    iHandler->getInterface(&ptr, "TestServices", interfaceInfo4, true, NULL);
    QTest::qWait( KWAITINTERVAL);
    QVERIFY(!ptr);

    //register the service
    copyFileToImport( KTESTXMLFILE5 );
    qDebug() << "Waiting for registration <--";
    QTest::qWait( KWAITINTERVAL);
    qDebug() << "Waiting for registration -->";
    //get list of providers to check service is registered
    QStringList list = iHandler->getServiceProviders();

    QVERIFY( list.count() == 3 );
    QCOMPARE(list[2],QString("TestServices"));
    for ( int i=0; i<list.count(); ++i )
        qDebug() << list[i];

    ServiceInterfaceInfo interfaceInfo5={"com.nokia.ISimpleTypeTest/1.0", 0};

    //load testserviceplugin
    iHandler->getInterface(&ptr, "TestServices", interfaceInfo5, false, NULL);
    QTest::qWait( KWAITINTERVAL);
    QVERIFY(ptr);
}

void servicehandlermoduletest::testGetServiceProviders()
{
    QList<QString> serviceList = iHandler->getServiceProviders();
    QVERIFY(serviceList.count() == 3);
    for (int i = 0; i < serviceList.count(); i++)
        qDebug() <<  serviceList[i];
    QCOMPARE(serviceList[0],QString("TestService"));
    QCOMPARE(serviceList[1],QString("TestService1"));
}

void servicehandlermoduletest::testGetServiceVersion()
{
    QCOMPARE(iHandler->getServiceVersion(""), QString("ServiceNotSupport"));
    QCOMPARE(iHandler->getServiceVersion("TestService"), QString("1.4.56"));
}


QTEST_MAIN(servicehandlermoduletest)
#include "servicehandlermoduletest.moc"
