/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#ifndef sfwservicelogger_h
#define sfwservicelogger_h

#ifdef __SYMBIAN32__
#include <flogger.h>
#include <f32file.h>

_LIT(KSwrtLogDir, "sfw");
#endif //__SYMBIAN32__

const int KMaxMsgSize = 1024;

/*!
    \brief A logging utility class for service providers

    This utlity class is intended to be intantiated by service providers to
    output messages to debug console(all platform) and to a log file(Symbian only)
    in the file system. The folder containing the log file has been hard-coded to
    "sfw" and it's realtive to c:\logs\ (i.e. the log file will be saved in c:\logs\sfw).
    The user can set the name of the log file when instantiating this utility class,
    however, if no name is specified(i.e. if default constructor is used), messages will
    only be printed to debug console. Logging messages to a file is only supported
    in Symbian platform, but messages will be printed to debug console in
    all platforms.
*/
class SfwServiceLogger
{
public:
    /*!
        Default constructor
     */
    SfwServiceLogger()
    {
#ifdef __SYMBIAN32__
        iLogger.Connect();
#endif
    }


    /*!
        Constructs a utility object, connects to RFileLogger, and creates a log
        file. This constructor is intended to be used in Symiban only.

        @param aFileName name of the log file
     */
    SfwServiceLogger(const char* aFileName)
    {
#ifdef __SYMBIAN32__
        iLogger.Connect();
        setLogFile(aFileName);
#else
        aFileName;
#endif
    }

    /*!
        Creates a log file with specified file name. If a log file already exists,
        old content won't be erased. Only intended to be used in Symbian.

        @param aFileName name of the log file to be created
     */
    bool setLogFile(const char* aFileName)
    {
        bool ret = true;
#ifdef __SYMBIAN32__
        if (iLogger.LogValid() == true)
        {
            iLogger.CloseLog();
        }

        TPtrC8 fileName8((TUint8 *)(aFileName));
        if ((fileName8.Length() >= KMaxFileName) || (fileName8.Length() == 0))
        {
            ret = false;
        }
        else
        {
            TBuf<KMaxFileName> fileName16;
            fileName16.Copy(fileName8);
            iLogger.CreateLog(KSwrtLogDir, fileName16, EFileLoggingModeAppend);
        }
#else
        aFileName;
#endif
        return ret;
    }

    /*!
        Returns the last error status that has been set. Symbian only.

        @return error number of last error
     */
    int lastLogError()
    {
#ifdef __SYMBIAN32__
        return iLogger.LastError();
#else
        return -1;
#endif
    }

    /*!
        Returns the status of the log. Symbian only.

        @return true if log is valid, false otherwise
     */
    bool logValid()
    {
#ifdef __SYMBIAN32__
        return iLogger.LogValid();
#else
        return false;
#endif
    }


    /*!
        Destructor
     */
    ~SfwServiceLogger()
    {
#ifdef __SYMBIAN32__
        // Close the log file and the connection to the server.
        if (iLogger.LogValid() == true)
        {
            iLogger.CloseLog();
        }
        iLogger.Close();
#endif
    }

    /*!
        Outputs messages to debug console for all platforms. Writes messages
        to a log file(Symbian only).

        @param text the format string to be printed
        @param ... A variable number of arguments to be converted as dictated
        by the format string
     */
    void log(const char* text, ...)
    {
        va_list arguments;
        va_start(arguments,text);
        char buf[KMaxMsgSize];
        qvsnprintf( buf, KMaxMsgSize, text, arguments );
        qDebug(buf);
        va_end(arguments);

#ifdef __SYMBIAN32__
        if (iLogger.LogValid() == true)
        {
            VA_LIST arguments_symbian;
            VA_START(arguments_symbian,text);
            TPtrC8 textDescriptor((TUint8 *)(text));
            iLogger.WriteFormat(textDescriptor, arguments_symbian);
            VA_END(arguments_symbian);
        }
#endif //__SYMBIAN32__
    }

private:
#ifdef __SYMBIAN32__
    RFileLogger iLogger;
#endif
};

#endif /* sfwservicelogger_h */
