/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QFile>
#include <QHash>
#include <QXmlStreamReader>
#include "trustparser.h"
#include "storage.h"
#include "secure_defs.h"

namespace WRT
{
/*!
 * \class TrustParser
 * This class parses a trust policy file
 */

/*!
 * Default Constructor
 */
TrustParser::TrustParser() :
    QObject(),
    m_xmlReader(NULL)
{
    createDomainList();
}

/*!
 * Constructor
 * \param[in] policy - Policy file name
 */
TrustParser::TrustParser(const QString &policy) :
    QObject(),
    m_xmlReader(NULL)
{
    setPolicy(policy);
    createDomainList();
}

/*!
 * Constructor
 * \param[in] policy - Policy file in memory
 */
TrustParser::TrustParser(const QByteArray &policy) :
    QObject(),
    m_xmlReader(NULL)
{
    setPolicy(policy);
    createDomainList();
}

/*!
 * Destructor
 */
TrustParser::~TrustParser()
{
    cleanup();
    for (int i = 0; i < m_domainList.count(); i++) {
        m_domainList[i].clear();
    }
    m_domainList.clear();
}

/*!
 * Delete dynamic member variables
 */
void TrustParser::cleanup()
{
    if (m_xmlReader) {
        delete m_xmlReader;
        m_xmlReader = NULL;
    }
}

/*!
 * Sets the policy file name
 * \param[in] policy - Policy file name
 */
void TrustParser::setPolicy(const QString &policy)
{
    cleanup();

    Storage *storage = Storage::createInstance(FILE_STORAGE_NAME,
                                               WRT::Storage::vis_shared,
                                               WRT::Storage::prot_signed);

    //Verify that the file is in storage and add if not
    if (storage->getFileList().contains(policy)) {
        if (!storage->verify(policy)) {
            //File has been altered. bad boy
            //TODO WHAT TO DO HERE IF THE FILE HAS BEEN ALTERED!!!
            delete storage;
            return;
        }
    } else {
        QFile tmp(policy);
        storage->add(tmp);
    }

    //Here we end up only when the file has not been tampered

    //Now get the actual file from the storage
    QIODevice *ptr = storage->getFile(policy);

    if (ptr && ptr->open(QIODevice::ReadOnly)) {
        ptr->setParent(this);
        m_xmlReader = new QXmlStreamReader(ptr);
    }else {
        delete ptr;
    }

    delete storage;
}

/*!
 * Sets the policy file name
 * \param[in] policy - Policy file in memory
 */
void TrustParser::setPolicy(const QByteArray &policy)
{
    cleanup();
    m_xmlReader = new QXmlStreamReader(policy);
}

/*!
 * Parses the policy file
 * \return TRUE if parse succeeds, else FALSE
 */
bool TrustParser::parse()
{
    bool parseSucceeded(false);
    if (m_xmlReader) {
        parseSucceeded = true;
        while (!m_xmlReader->atEnd() && parseSucceeded) {
            QXmlStreamReader::TokenType tokenType = m_xmlReader->readNext();
            switch (tokenType) {
            case QXmlStreamReader::Invalid: {
                parseSucceeded = false;
                break;
            }
            case QXmlStreamReader::StartElement: {
                const QString name = m_xmlReader->name().toString();
                if (KDEFAULTDOMAIN == name) {
                    parseSucceeded = processDefaultDomain();
                } else if (KDOMAIN == name) {
                    parseSucceeded = processDomain();
                }
                break;
            }
            default:
                break;
            } // end switch
        }

        // Handle error
        if (!parseSucceeded) {
            //QXmlStreamReader::Error error = m_xmlReader->error();
            //QString errorString = m_xmlReader->errorString();
            for (int i = 0; i < m_domainList.count(); i++) {
                m_domainList[i].clear();
            }
            m_defaultDomain = QString();
        }
    }
    return parseSucceeded;
}

/*!
 * Gets the default domain
 * \pre The policy file must be specified and parsed
 * \return The default domain
 */
const QString TrustParser::defaultDomain() const
{
    return m_defaultDomain;
}

/*!
 * Gets the default origin domain
 * \pre The policy file must be specified and parsed
 * \return The default domain
 */
const QString TrustParser::defaultOriginDomain() const
{
    return m_defaultDomainHash.value(ETrustCriteriaOrigin, m_defaultDomain);
}

/*!
 * Gets the default apppath domain
 * \pre The policy file must be specified and parsed
 * \return The default domain
 */
const QString TrustParser::defaultAppPathDomain() const
{
    return m_defaultDomainHash.value(ETrustCriteriaAppPath, m_defaultDomain);
}

/*!
 * Gets the domain for a specified value
 * \param[in] type - One of TrustCriteriaTypes,
 *                    specifies what the aValue refers to.
 * \param[in] value - Value to get the domain for
 * \return The requested domain
 */

const QString TrustParser::domainFor(TrustCriteriaType type,
                                     const QString &value) const
{
    QString domain;
    const QHash<QString,QString> *domainList = getHashList(type);
    if (domainList) {
      domain = domainList->value(value, "");
    }

    if (domain == "")
        domain = m_defaultDomainHash.value(type, m_defaultDomain);

    return domain;
}

/*!
 * Gets the list of all values for a specified field
 * \param[in] type - One of TrustCriteriaTypes to get the values for
 * \return List of values
 */
QStringList TrustParser::values(TrustCriteriaType type) const
{
    QStringList valuesList;
    const QHash<QString,QString> *domainList = getHashList(type);
    if (domainList) {
        valuesList = domainList->keys();
    }
    return valuesList;
}

/*!
 * Creates the initial empty domain list
 * \return void
 */
void TrustParser::createDomainList()
{
    QHash<QString,QString> originsHash;
    m_domainList.insert(ETrustCriteriaOrigin, originsHash);
    QHash<QString,QString> processHash;
    m_domainList.insert(ETrustCriteriaProcessId, processHash);
    QHash<QString,QString> certificateHash;
    m_domainList.insert(ETrustCriteriaCertificate, certificateHash);
}

/*!
 * Gets the hash list of the given type
 * \param[in] type - Specifies which list to get
 * \return The hash list for the specified type.  Not owned.  NULL if not found
 */
const QHash<QString,QString> *TrustParser::getHashList(TrustCriteriaType type) const
{
    const QHash<QString,QString> *list = NULL;

    switch (type) {
    case ETrustCriteriaOrigin:
    case ETrustCriteriaAppPath:
        list = &(m_domainList[ETrustCriteriaOrigin]);
        break;
    case ETrustCriteriaProcessId:
        list = &(m_domainList[ETrustCriteriaProcessId]);
        break;
    case ETrustCriteriaCertificate:
        list = &(m_domainList[ETrustCriteriaCertificate]);
        break;
    }
    return list;
}

/*!
 * Handles retrieving the default domain
 * \return TRUE if successful, otherwise FALSE
 */
bool TrustParser::processDefaultDomain()
{
    bool parseSucceeded = true;
    QString domainName;
    QXmlStreamAttributes atts = m_xmlReader->attributes();
    QStringRef domainRef = atts.value(KNAME);
    if (!domainRef.isNull() && !domainRef.isEmpty()) {
        domainName = domainRef.toString();
    } else {
        parseSucceeded = false;
    }

    // Read to the rest of this element
    bool isGlobalDefaultDomain = true;
    if (parseSucceeded) {
        QXmlStreamReader::TokenType tokenType = m_xmlReader->tokenType();
        while (tokenType != QXmlStreamReader::EndElement && parseSucceeded) {
            tokenType = m_xmlReader->readNext();
            if (tokenType == QXmlStreamReader::EndDocument
                || tokenType == QXmlStreamReader::Invalid) {
                parseSucceeded = false;
            } else if (tokenType == QXmlStreamReader::StartElement) {
                // Get subsequent attributes, could be ORIGIN, APPPAHT, PROCESS or CERTIFICATE
                // Check which one it needs to get first
                const QString name = m_xmlReader->name().toString();
                TrustCriteriaType type;
                if (name == KORIGIN) {
                    type = ETrustCriteriaOrigin;
                } else if (name == KPROCESS) {
                    type = ETrustCriteriaProcessId;
                } else if (name == KCERTIFICATE) {
                    type = ETrustCriteriaCertificate;
                } else if (name == KAPPPATH) {
                    type = ETrustCriteriaAppPath;
                } else {
                    parseSucceeded = false;
                }
                if (parseSucceeded) {
                    m_defaultDomainHash.insert(type, domainName);
                    // Read and ignore any other data for this element
                    m_xmlReader->readElementText();
                    tokenType = m_xmlReader->readNext();
                    isGlobalDefaultDomain = false;
                }
            }
        }
    }

    if (isGlobalDefaultDomain)
        m_defaultDomain = domainName;
    return parseSucceeded;
}

/*!
 * Handles retrieving a domain
 * \return TRUE if successful, otherwise FALSE
 */
bool TrustParser::processDomain()
{
    bool parseSucceeded(true);
    QString domainName;
    QStringRef domainRef = m_xmlReader->attributes().value(KNAME);
    if (!domainRef.isNull() && !domainRef.isEmpty()) {
        domainName = domainRef.toString();
    } else {
        parseSucceeded = false;
    }

    // Read to the rest of this element
    if (parseSucceeded) {
        QXmlStreamReader::TokenType tokenType = m_xmlReader->tokenType();
        while (tokenType != QXmlStreamReader::EndElement && parseSucceeded) {
            tokenType = m_xmlReader->readNext();
            if (tokenType == QXmlStreamReader::EndDocument
                || tokenType == QXmlStreamReader::Invalid) {
                parseSucceeded = false;
            } else if (tokenType == QXmlStreamReader::StartElement) {
                // Get subsequent attributes, could be ORIGIN or PROCESS

                // Check which one it needs to get first
                const QString name = m_xmlReader->name().toString();
                TrustCriteriaType type;
                QString attributeKey;
                if (name == KORIGIN) {
                    type = ETrustCriteriaOrigin;
                    attributeKey = KURL;
                } else if (name == KPROCESS) {
                    type = ETrustCriteriaProcessId;
                    attributeKey = KID;
                } else if (name == KCERTIFICATE) {
                    type = ETrustCriteriaCertificate;
                    attributeKey = KAKI;
                } else if (name == KAPPPATH) {
                    type = ETrustCriteriaAppPath;
                    attributeKey = KURL;
                } else {
                    parseSucceeded = false;
                }

                // Get the ORIGIN/PROCESS/CERTIFICATE and add to the list
                if (parseSucceeded) {
                    QHash<QString,QString> *hashList =
                        const_cast<QHash<QString,QString>*>(getHashList(type));
                    QStringRef attRef = m_xmlReader->attributes().value(attributeKey);
                    if (hashList && !attRef.isNull() && !attRef.isEmpty()) {
                        hashList->insert(attRef.toString(), domainName);
                        // Read and ignore any other data for this element
                        m_xmlReader->readElementText();
                        tokenType = m_xmlReader->readNext();
                    } else {
                        parseSucceeded = false;
                    }
                }
            }
        }
    }
    return parseSucceeded;
}

} // end namespace WRT
