/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */



#include <QFile>
#include <QHash>
#include <QtDebug>
#include <QPointer>
#include <QXmlStreamReader>
#include "policyparser.h"
#include "xmlpolicyserializer.h"
#include "secmgrdefs.h"
#include "storage.h"
#include "secure_defs.h"

namespace WRT
{
/*!
 * \class PolicyParser
 * Parser for access policy XML file
 * Note that the policy parser is not forgiving of non-compliant policy files. This is OK since
 * policy files are a highly controlled substance.
 */

/*!
 * Constructor
 * \param[in] policy - Policy file name
 */
PolicyParser::PolicyParser(const QString &policy) :
    QObject(),
    m_xmlReader(NULL),
    m_conditionHandler(NULL),
    m_conditionSerializer(NULL)
{
    createConditionTable();

#ifndef Q_OS_MAEMO6
    Storage *storage = Storage::createInstance(FILE_STORAGE_NAME,
                                               WRT::Storage::vis_shared,
                                               WRT::Storage::prot_signed);

    if (storage->getFileList().contains(policy)) {
        if (!storage->verify(policy)) {
            //File has been altered. bad boy
            //TODO WHAT TO DO HERE IF THE FILE HAS BEEN ALTERED!!!
            delete storage;
            return;
        }
    } else {
        QFile tmp(policy);
        if (tmp.exists()) {
            storage->add(tmp);
        }
    }

    QIODevice *ptr = storage->getFile(policy);

    if (ptr && ptr->open(QIODevice::ReadOnly)) {
        ptr->setParent(this);
        m_xmlReader = new QXmlStreamReader(ptr);
    }
    else {
        // if ptr->open fails, delete allocated object
        delete ptr;
    }

    delete storage;

#else
    //TODO: WE WILL USE AEGIS FILE SYSTEM FOR HARMATTAN

    QFile *ptr = new QFile(policy);

    if (ptr && ptr->open(QIODevice::ReadOnly)) {
        ptr->setParent(this);
        m_xmlReader = new QXmlStreamReader(ptr);
    }
    else {
        // if ptr->open fails, delete allocated object
        delete ptr;
    }

#endif

}

/*!
 * Constructor
 * \param[in] policy - Policy file in memory
 */
PolicyParser::PolicyParser(const QByteArray &policy) :
    QObject(),
    m_xmlReader(NULL),
    m_conditionHandler(NULL),
    m_conditionSerializer(NULL)
{
    createConditionTable();
    m_xmlReader = new QXmlStreamReader(policy);
}

/*!
 * Destructor
 */
PolicyParser::~PolicyParser()
{
    cleanup();
}

/*!
 * Parse the policy file for only the given domain
 * \param[in] domain - Domain to parse the policy file for
 * \return TRUE if successful, else FALSE
 */
bool PolicyParser::parse(const QString &/*domain*/)
{
    //TODO:
    return parse();
}

/*!
 * Parse the policy file for all domains
 * \pre A valid policy file must be set on construction.  If the policy is invalid,
 *      parse will fail and any subsequent parse() calls will do nothing.
 * \return void
 */
bool PolicyParser::parse()
{
    bool parseSucceeded(false);
    if (m_xmlReader) {
        parseSucceeded = true;
        while (!m_xmlReader->atEnd() && parseSucceeded) {
            QXmlStreamReader::TokenType tokenType = m_xmlReader->readNext();
            switch (tokenType) {
            case QXmlStreamReader::Invalid: {
                parseSucceeded = false;
                break;
            }
            case QXmlStreamReader::StartElement: {
                const QString name = m_xmlReader->name().toString();
                if (KALIAS == name) {
                    parseSucceeded = processAlias();
                } else if (KDOMAIN == name) {
                    parseSucceeded = processDomain();
                }
                break;
            }
            default:
                break;
            } // end switch
        }

        // Handle error
        if (!parseSucceeded) {
            //QXmlStreamReader::Error error = m_xmlReader->error();
            //QString errorString = m_xmlReader->errorString();
            cleanup();
        }
    }
    return parseSucceeded;
}

/*!
 * Gets a list of all domains
 * \pre The policy file must be specified and parsed
 * \return List of domains
 */
QStringList PolicyParser::domains() const
{
    return m_domains.keys();
}

/*!
 * Gets the default domain
 * \pre The policy file must be specified and parsed
 * \return The default domain
 */
const QString PolicyParser::defaultDomain() const
{
    QString domain;
    if (!m_domains.isEmpty()) {
        domain = m_domains.keys().at(0);
    }
    return domain;
}

/*!
 * Gets a list of all capabilities for a specified domain
 * \pre The policy file must be specified and parsed
 * \param[in] domain - Domain to get capabilities for
 * \return List of capabilities, else NULL if domain is not found
 */
//TBS: need a version that gets an unmodifiable capability set for public use!
const CapabilitySet * PolicyParser::capabilities(const QString &domain) const
{
    return m_domains.value(domain, NULL);
}

/*!
 * Gets a list of all aliases for a specified capability
 * \pre The policy file must be specified and parsed
 * \param[in] capability - Capability to get aliases for
 * \return List of Aliases
 */
const QStringList PolicyParser::aliasesFor(const QString &capability) const
{
    return m_capAliases.value(capability);
}

/*!
 * Checks whether a capability has an alias
 * \pre The policy file must be specified and parsed
 * \param[in] capability - Capability to check for aliases
 * \return TRUE if aliase exists, else FALSE
 */
bool PolicyParser::hasAlias(const QString &capability) const
{
    return (m_capAliases.contains(capability)
            || m_aliasSet.contains(capability));
}

/*!
 * Gets all aliases
 * \pre The policy file must be specified and parsed
 * \return The set of all aliases
 */
const AliasSet *PolicyParser::aliases() const
{
    return &m_aliasSet;
}

/*!
 * Sets the user condition handler for this policy
 * \param[in] handler - User Condition Handler
 * \return void
 */
void PolicyParser::setUserHandler(const UserConditionHandler *handler)
{
    m_conditionHandler = handler;
}

/*!
 * Sets the serializer for this policy
 * \param[in] serializer - The policy serializer to be used
 * \return void
 */
void PolicyParser::setSerializer(UserConditionSerializer *serializer)
{
    m_conditionSerializer = serializer;
}

/*!
 * Cleans up and deletes dynamic member variables
 */
void PolicyParser::cleanup()
{
    QHash<QString, CapabilitySet*>::iterator domainIterator;
    for (domainIterator = m_domains.begin();
         domainIterator != m_domains.end();
         ++domainIterator) {
        CapabilitySet *capSet = domainIterator.value();
        CapabilitySet::iterator capIterator;
        for (capIterator = capSet->begin();
             capIterator != capSet->end();
             ++capIterator) {
            Condition *cond = capIterator.value().data();
            delete cond;
        }
        capSet->clear();
        delete capSet;
    }
    m_domains.clear();

    if (m_xmlReader) {
        delete m_xmlReader;
        m_xmlReader = NULL;
    }
}

/*!
 * Creates internal condition table, used to map between
 * a string and the bitmask value
 */
void PolicyParser::createConditionTable()
{
    m_conditionTable[KONESHOT] = EUserGrantOneShot;
    m_conditionTable[KSESSION] = EUserGrantSession;
    m_conditionTable[KBLANKET] = EUserGrantBlanket;
}

/*!
 * Process the <alias> tag
 * This tag contains the name of the alias, and any number of capabilities
 * that are contained in this alias.
 * \return TRUE if successful, else FALSE
 */
bool PolicyParser::processAlias()
{
    bool parseSucceeded(true);

    // Get the alias name first
    QString aliasName;
    QStringRef aliasRef = m_xmlReader->attributes().value(KNAME);
    if (!aliasRef.isNull() && !aliasRef.isEmpty()) {
        aliasName = aliasRef.toString();
    } else {
        parseSucceeded = false;
    }

    // Read to the rest of this element
    // Get all the <capability> tags that are associated with this alias
    if (parseSucceeded) {
        CapabilityList capList;
        QXmlStreamReader::TokenType tokenType = m_xmlReader->readNext();
        while (parseSucceeded
               && tokenType != QXmlStreamReader::Invalid
               && tokenType != QXmlStreamReader::EndDocument
               && (tokenType != QXmlStreamReader::EndElement
                   || m_xmlReader->name() != KALIAS)) {

            // Only read <capability> tags
            if (tokenType == QXmlStreamReader::StartElement
                && m_xmlReader->name() == KCAPABILITY) {
                QString capName = processCapability();
                if (!capName.isEmpty() && !capName.isNull()) {
                    capList.append(capName);
                    if (m_capAliases.contains(capName)) {
                        QStringList aliasList = m_capAliases.value(capName);
                        if (!aliasList.contains(aliasName)) {
                            aliasList.append(aliasName);
                            m_capAliases.insert(capName, aliasList);
                        }
                    } else {
                        QStringList aliasList;
                        aliasList.append(aliasName);
                        m_capAliases.insert(capName, aliasList);
                    }
                } else {
                    parseSucceeded = false;
                }
            }
            tokenType = m_xmlReader->readNext();
        }   // end while
        if (parseSucceeded) {
            m_aliasSet.insert(aliasName, capList);
        }
    }
    return parseSucceeded;
}

/*!
 * Process <capability> tag
 * \return The capability as a string if successful, else an empty string
 */
QString PolicyParser::processCapability()
{
    QString capName;
    QStringRef capRef = m_xmlReader->attributes().value(KNAME);
    if (!capRef.isNull() && !capRef.isEmpty()) {
        capName = capRef.toString();

        // Read to the rest of this element
        m_xmlReader->readElementText();
    }
    return capName;
}

/*
 * Process <domain> tag
 * \return TRUE if successful, else FALSE
 */
bool PolicyParser::processDomain()
{
    bool parseSucceeded(true);

    // Get the domain name
    QString domainName;
    QStringRef domainRef = m_xmlReader->attributes().value(KNAME);
    if (!domainRef.isNull() && !domainRef.isEmpty()) {
        domainName = domainRef.toString();
    } else {
        parseSucceeded = false;
    }

    // Read the rest of the element
    // A domain consists of a number of capabilities, along with
    // an optional user section for user prompted capabilities.
    if (parseSucceeded) {
        CapabilitySet *capSet = m_domains.value(domainName, NULL);
        if (!capSet) {
            capSet = new CapabilitySet();
        }
        m_domains.insert(domainName, capSet);    //ownership passed
        QXmlStreamReader::TokenType tokenType = m_xmlReader->readNext();
        while (parseSucceeded
               && tokenType != QXmlStreamReader::Invalid
               && tokenType != QXmlStreamReader::EndDocument
               && (tokenType != QXmlStreamReader::EndElement
                   || m_xmlReader->name() != KDOMAIN)) {
            if (tokenType == QXmlStreamReader::StartElement) {
                if (m_xmlReader->name() == KCAPABILITY) {
                    QString capName = processCapability();
                    if (!capName.isEmpty() && !capName.isNull()) {
                        capSet = m_domains.value(domainName);
                        capSet->insert(capName, NULL);
                        m_domains.insert(domainName, capSet);
                    } else {
                        parseSucceeded = false;
                    }
                } else if (m_xmlReader->name() == KUSER) {
                    parseSucceeded = processUserSection(domainName);
                }
            }
            tokenType = m_xmlReader->readNext();
        }   // end while
        parseSucceeded = !m_xmlReader->hasError();
    }
    return parseSucceeded;
}

/*!
 * Process <user> tag
 * The user area contains a list of the scopes and a list of capabilities.
 * These capabilities are the ones that need to be requested (e.g. prompting) to
 * the user before it is granted.  The user section also contains a <grant> tag
 * that contains the current grant status (if restoring a session).
 * \return TRUE if successful, else FALSE
 */
bool PolicyParser::processUserSection(const QString &domainName)
{
    bool parseSucceeded(true);
    QXmlStreamReader::TokenType tokenType = m_xmlReader->readNext();
    unsigned int flags(0);
    unsigned int grant(0);
    QString capName;
    QStringList capNameList;
    while (parseSucceeded
           && tokenType != QXmlStreamReader::Invalid
           && tokenType != QXmlStreamReader::EndDocument
           && (tokenType != QXmlStreamReader::EndElement
               || m_xmlReader->name() != KUSER)) {
        if (tokenType == QXmlStreamReader::StartElement) {
            if (m_xmlReader->name() == KDEFAULTSCOPE) {
                QString defaultScope = processDefaultScope();
                if (!defaultScope.isEmpty() && !defaultScope.isNull()) {
                    flags = (flags & EUserGrantCapsMask) | ((m_conditionTable.value(defaultScope))<<3);
                    flags |= m_conditionTable.value(defaultScope);
                } else {
                    parseSucceeded = false;
                }
            } else if (m_xmlReader->name() == KSCOPE) {
                QString scope = processScope();
                if (!scope.isEmpty() && !scope.isNull()) {
                    flags |= m_conditionTable.value(scope);
                } else {
                    parseSucceeded = false;
                }
            } else if (m_xmlReader->name() == KCAPABILITY) {
                capName = processCapability();
                if (capName.isEmpty() || capName.isNull())
                    parseSucceeded = false;
                else
                    capNameList.append(capName);
            } else if (m_xmlReader->name() == KGRANT) {
                grant = processGrant();
            }
        }
        tokenType = m_xmlReader->readNext();
    }   // end while
    parseSucceeded = !m_xmlReader->hasError();

    // FIXME: need adding 'denied all' support in the future
    // Create new condition and insert into domains hash
    if (parseSucceeded) {
        // capName now is the last capibility in the capNameList
        UserCondition *cond = new UserCondition(capName, flags, m_conditionHandler, m_conditionSerializer);
        if (grant != 0) {
            cond->update(0, grant);
        }
        CapabilitySet *capSet = m_domains.value(domainName, NULL);
        if (!capSet) {
            capSet = new CapabilitySet;
        }
        capSet->insert(capName, cond);
        int size = capNameList.size() - 1; // no need the last capibility, added before
        for (int i = 0; i < size; i++) {
            capName = capNameList.at(i);
            cond->add(capName);
            capSet->insert(capName, cond);
        }
        m_domains.insert(domainName, capSet);
    }
    return parseSucceeded;
}

/*!
 * Process DefaultScope tag
 * \return The default scope as a string if successful, else an empty string
 */
QString PolicyParser::processDefaultScope()
{
    return processScope();
}

/*!
 * Process Scope tag
 * \return The scope as a string if successful, else an empty string
 */
QString PolicyParser::processScope()
{
    QString scopeName;
    QStringRef scopeRef = m_xmlReader->attributes().value(KTYPE);
    if (!scopeRef.isNull() && !scopeRef.isEmpty()) {
        scopeName = scopeRef.toString();

        // Read to the rest of this element
        m_xmlReader->readElementText();
    }
    return scopeName;
}

/*!
 * Process Grant tag
 * \return The grant as an integer if successful, else an empty string
 */
unsigned int PolicyParser::processGrant()
{
    unsigned int grant(0);
    QStringRef grantRef = m_xmlReader->attributes().value(KTYPE);
    if (!grantRef.isNull() && !grantRef.isEmpty()) {
        grant = grantRef.toString().toUInt();

        // Read to the rest of this element
        m_xmlReader->readElementText();
    }
    return grant;
}

}
