/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <clientinfo.h>
#include "securestoragedefs.h"
#include "securestorageserver.h"
#include "securestorageserversession.h"
#include "serviceipcdefs.h"
#include "serviceipcserver.h"
#include "serviceipcrequest.h"
#include "sslog.h"

#ifdef __SYMBIAN32__
#include "securestoragesymbian.h"
#include "defaultcaps.hrh"
#endif  // __SYMBIAN32__

namespace WRT
{
// CONSTANTS

/*!
 * \class WRT::SecureStorageServer
 * Server class, handles requests from clients
 */

/*!
 * Constructor
 * \param[in] aParent - QObject owning this object
 */
SecureStorageServer::SecureStorageServer(QObject *aParent) :
    QObject(aParent),
    m_server(NULL)
{
    m_server = new ServiceFwIPCServer(this, this);
    if (!m_server->listen(KSECURESTORAGESERVER)){
        //Close server if it is already started
        QTimer::singleShot(1, this, SLOT( closeServer()));
        return;
    }

    connect( m_server,
           SIGNAL( handleExit()),
           this,
           SLOT( closeServer()));
}

/*!
 * Destructor
 */
SecureStorageServer::~SecureStorageServer()
{
    if (m_server) {
        m_server->disconnect();
    }
    // Don't need to delete the actual session objects since their parent is this
    // qDeleteAll(m_storageSessions.begin(), m_storageSessions.end());
    foreach (SecureStorageServerSession *sess, m_storageSessions) {
        sess->uninit();
        delete sess;
    }
    m_storageSessions.clear();
}

/*!
 * From MServiceIPCObserver
 * \see MServiceIPCObserver::handleRequest(ServiceIPCRequest *aRequest)
 */
bool SecureStorageServer::handleRequest(ServiceIPCRequest *aRequest)
{
    SS_FUNC("SecureStorageServer::handleRequest()");
    // TODO: currently result is ignored in IPC code.  Update if that behaviour changes
    bool result(false);
    bool isVIDValid(true);  // Default to true for platforms that don't have VID

    // check if the calling process's vendor ID is Nokia VID
    ClientInfo* client(const_cast<ClientInfo*>(aRequest->clientInfo()));
    if (client) {
#ifdef __SYMBIAN32__
        isVIDValid = (client->vendorId() == VID_DEFAULT);
#endif
    }

    if (isVIDValid) {
        bool isOK(false);
        int operation(aRequest->getOperation().toInt(&isOK));
        if (isOK) {
            QByteArray data(aRequest->getData());
            switch (operation) {
            case KSECURESTORAGEOPINIT: {
                SS_DEBUG("SecureStorageServer::handleRequest(): KSECURESTORAGEOPINIT");
                QList<QByteArray> list(data.split(KSECURESTORAGEDELIMITER));
                if (list.count() == 3) {
                    SecureStorageServerSession *sess(findOrCreateSession(list[0]));
                    sess->setClientInfo(client);
                    Storage::visibility vis =
                        static_cast<Storage::visibility>(QString(list[1]).toInt());
                    Storage::protection prot =
                        static_cast<Storage::protection>(QString(list[2]).toInt());
                    // TODO: Should check if valid enum value
                    sess->setMode(vis, prot);
                    sess->init();
                }
                break;
            }
            case KSECURESTORAGEOPGETFILELIST: {
                SS_DEBUG("SecureStorageServer::handleRequest(): KSECURESTORAGEOPGETFILELIST");
                QString name(data);
                if (!name.isEmpty()) {
                    SecureStorageServerSession *sess(findSession(name));
                    if (sess) {
                        sess->setClientInfo(client);
                        aRequest->write(sess->getFileList());
                    }
                }
                break;
            }
            case KSECURESTORAGEOPSAVEFILELIST: {
                SS_DEBUG("SecureStorageServer::handleRequest(): KSECURESTORAGEOPSAVEFILELIST");
                int index(data.indexOf(KSECURESTORAGEDELIMITER));
                if (index != -1) {
                    QString name(data.left(index));
                    SecureStorageServerSession *sess(findSession(name));
                    if (sess) {
                        sess->setClientInfo(client);
                        sess->saveFileList(data.mid(index+1));
                    }
                }
                break;
            }
            case KSECURESTORAGEOPSAVEMETADATA: {
                SS_DEBUG("SecureStorageServer::handleRequest(): KSECURESTORAGEOPSAVEMETADATA");
                int index(data.indexOf(KSECURESTORAGEDELIMITER));
                if (index != -1) {
                    QString name(data.left(index));
                    SecureStorageServerSession *sess(findSession(name));
                    if (sess) {
                        sess->setClientInfo(client);
                        sess->saveMetadata(data.mid(index+1));
                    }
                }
                break;
            }
            case KSECURESTORAGEOPREMOVEMETADATA: {
                SS_DEBUG("SecureStorageServer::handleRequest(): KSECURESTORAGEOPREMOVEMETADATA");
                QList<QByteArray> list(data.split(KSECURESTORAGEDELIMITER));
                if (list.count() == 2) {
                    SecureStorageServerSession *sess(findSession(list[0]));
                    if (sess) {
                        sess->setClientInfo(client);
                        sess->removeMetadata(list[1]);
                    }
                }
                break;
            }
            case KSECURESTORAGEOPGETMETADATA: {
                SS_DEBUG("SecureStorageServer::handleRequest(): KSECURESTORAGEOPGETMETADATA");
                QList<QByteArray> list(data.split(KSECURESTORAGEDELIMITER));
                if (list.count() == 2) {
                    SecureStorageServerSession *sess(findSession(list[0]));
                    if (sess) {
                        sess->setClientInfo(client);
                        QByteArray response(sess->getMetadata(list[1]));
                        aRequest->write(response);
                    }
                }
                break;
            }
            case KSECURESTORAGEOPSAVEKEY: {
                SS_DEBUG("SecureStorageServer::handleRequest(): KSECURESTORAGEOPSAVEKEY");
                int index(data.indexOf(KSECURESTORAGEDELIMITER));
                if (index != -1) {
                    QString name(data.left(index));
                    SecureStorageServerSession *sess(findSession(name));
                    if (sess) {
                        sess->setClientInfo(client);
                        data = data.mid(index+1);
                        index = data.indexOf(KSECURESTORAGEDELIMITER);
                        if (index != -1) {
                            QString fileName(data.left(index));
                            sess->saveKey(fileName, data.mid(index+1));
                        }
                    }
                }
                break;
            }
            case KSECURESTORAGEOPGETKEY: {
                SS_DEBUG("SecureStorageServer::handleRequest(): KSECURESTORAGEOPGETKEY");
                QList<QByteArray> list(data.split(KSECURESTORAGEDELIMITER));
                if (list.count() == 2) {
                    SecureStorageServerSession *sess(findSession(list[0]));
                    if (sess) {
                        sess->setClientInfo(client);
                        QByteArray response(sess->getKey(list[1]));
                        aRequest->write(response);
                    }
                }
                break;
            }
            case KSECURESTORAGEOPGETFILE: {
                SS_DEBUG("SecureStorageServer::handleRequest(): KSECURESTORAGEOPGETFILE");
                QList<QByteArray> list(data.split(KSECURESTORAGEDELIMITER));
                if (list.count() == 2) {
                    SecureStorageServerSession *sess(findSession(list[0]));
                    if (sess) {
                        sess->setClientInfo(client);
                        QByteArray response(sess->getFile(list[1]));
                        aRequest->write(response);
                    }
                }
                break;
            }
            default: {
                // Unrecognized operation
                // Do nothing
            }
            }   // end switch (operation)
        }   // end if (isOK)
    }   // end if (isVIDValid)
    result = true;
    aRequest->completeRequest();
    return result;
}

/*!
 * From MServiceIPCObserver
 * \see MServiceIPCObserver::handleCancelRequest(ServiceIPCRequest *aRequest)
 */
void SecureStorageServer::handleCancelRequest(ServiceIPCRequest * /*aRequest*/)
{
}

/*!
 * From MServiceIPCObserver
 * \see MServiceIPCObserver::handleClientConnect(ClientInfo *aNewClient)
 */
void SecureStorageServer::handleClientConnect( ClientInfo * /*aNewClient*/ )
{
}

/*!
 * From MServiceIPCObserver
 * \see MServiceIPCObserver::handleClientDisconnect(ClientInfo *aClient)
 */
void SecureStorageServer::handleClientDisconnect( ClientInfo * /*aClient*/ )
{
}

/*!
 * Finds the session object associated with the name.
 */
SecureStorageServerSession *SecureStorageServer::findSession(const QString &aName)
{
    SecureStorageServerSession *sess(NULL);
    foreach(SecureStorageServerSession *s, m_storageSessions) {
        if (s->name() == aName) {
            sess = s;
            break;
        }
    }
    return sess;
}

/*!
 * Creates a new session object associated with the name if one does not exist
 */
SecureStorageServerSession *SecureStorageServer::findOrCreateSession(const QString &aName)
{
    SecureStorageServerSession *sess(findSession(aName));
    if (!sess) {
#ifdef __SYMBIAN32__
        sess = SecureStorageSymbian::NewL(this);
#else
        // TODO: platform specific implementation
        sess = new SecureStorageServerSession(this);
#endif
        sess->setName(aName);
        m_storageSessions.append(sess);
    }
    return sess;
}

/*!
 * Close server
 */
void SecureStorageServer::closeServer()
{
    QCoreApplication::quit();
}

} // end namespace WRT
