/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef __WRT_DEFAULTSTORAGE_HH
#define __WRT_DEFAULTSTORAGE_HH

#include <QObject>
#include <QString>
#include <QStringList>
#include <QSettings>

#include "storage.h"

class QString;
class QByteArray;

namespace WRT {

/** Implements WRT::Storage interface in an unprocted / unsecure way.
  *
 */
class DefaultStorage: public WRT::Storage {
    Q_OBJECT

public:
    explicit DefaultStorage(const QString & aStorageName, QObject* aParent = 0);
    virtual ~DefaultStorage();
    /**
     * Returns a list of files stored in the storage
     * @return a list of file name stored in the storage.
     */
    virtual QStringList getFileList() const;

    /**
     * Adds a file to storage. Calculates checksum for the file based on the file name and size. Basic support based on MD5 hash.
     * @param aFile the file to be added to storage
     */
    virtual void add(const QFile & aFile);

    /**
     * Adds an in memory file to storage. Calculates checksum for the file based on the file name and size. Basic support based on MD5 hash.
     * @param aFilePath the the file path including filename that is to be added to the storage
     */
    virtual void add(const QString & aFilePath, const QBuffer & aData);

    /**
     * Removes a file from storage
     * @param aFile to be removed from storage
     */
    virtual void remove(const QFile & aFile);

    /**
     * Verifies that the checksum of a file matches to the stored file.
     * Checksum is calculated from the file name, size, and last modification date of the file. Basic support based on MD5 hash.
     * @param aFile to be verified
     */
    virtual bool verify(const QFile & aFile) const;

    /**
     * Returns a file stored in the storage. Caller becomes the owner of the QIODevice. Hence, the caller must delete the QIODevice instance.
     * @param aFile the name of file to be returned, including full path.
     * @return a QIODevice instance when file exists in storage, null is returned otherwise. The returned QIODevice is not opened
     */
    virtual QIODevice * getFile(const QString & aPathFile);

    /**
     * Returns the number of files stored in the the storage.
     */
    virtual qint32 count() const;


private: // Member functions
    /**
     * Calculates an MD5 hash value from the file name and size. The output hash is a hex encoded and it uses
     * the numbers 0-9 and the letters a-f. Hash value is returned in output parameter (aOutHash).
     * @param aFileName
     * @param aSize of the file
     * @param aOutHash an output param for hash
     * @return true on success, otherwise false
     */
    bool
    calculateHash(const QString & aFile, const qint64 & aSize,
            QByteArray & aOutHash) const;

private: // Member variables
    QSettings m_storageFiles;
};

}
#endif // __WRT_DEFAULTSTORAGE_HH
