/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "defaultstorage.h"

#include <QCryptographicHash>
#include <QByteArray>
#include <QDataStream>
#include <QtDebug>
#include <QIODevice>
#include <QFile>
#include <QFileInfo>
#include <QDateTime>
#include <QBuffer>

namespace WRT {

DefaultStorage::DefaultStorage(const QString & aStorageName, QObject* aParent) :
    WRT::Storage(aStorageName, aParent), m_storageFiles("cwrt",
            aStorageName, this) {
}

DefaultStorage::~DefaultStorage() {
}

QStringList DefaultStorage::getFileList() const {
    return m_storageFiles.allKeys();
}

void DefaultStorage::add(const QFile & aFile) {
    QString filePath = QFileInfo(aFile).absoluteFilePath();
    QByteArray hashValue;
    qint64 size = aFile.size();
    bool ok = calculateHash(filePath, size, hashValue);
    if (ok) {
        m_storageFiles.setValue(filePath, hashValue);
    }
}

void DefaultStorage::add(const QString & aFilePath, const QBuffer & aData) {
    QFile * file = new QFile(aFilePath);
    file->open(QIODevice::WriteOnly);
    file->resize(0);

    QDataStream stream(file);
    qDebug() << "Data: " << aData.data().constData();
    qDebug() << "Data size: " << aData.data().size();

    int bytesWritten = stream.writeRawData(aData.data().constData(), aData.data().size());

    if (bytesWritten == -1)
       qDebug() << Q_FUNC_INFO << "stream.writeRawData failed";
    else {
       qDebug() << Q_FUNC_INFO << "Bytes written to file: " << bytesWritten;
       add(*file);
     }

    file->close();

    delete file;

}

void DefaultStorage::remove(const QFile & aFile) {
    QByteArray hashValue;
    qint64 size = aFile.size();
    QString filePath = QFileInfo(aFile).absoluteFilePath();
    bool ok = calculateHash(filePath, size, hashValue);
    if (ok) {
        m_storageFiles.remove(filePath);
    }

}

bool DefaultStorage::verify(const QFile & aFile) const {
    QByteArray hash;
    QString filePath = QFileInfo(aFile).absoluteFilePath();
    qint64 size = aFile.size();
    bool ok = calculateHash(filePath, size, hash);
    if (ok) {

        if (!m_storageFiles.contains(filePath)) {
            // File not in storage
            return false;
        }

        QByteArray storedHash = m_storageFiles.value(filePath).toByteArray();
        qDebug() << Q_FUNC_INFO << "Stored hash vs file hash >>>>>>> " << storedHash << "==" << hash;
        if (storedHash == hash) {
            return true;
        } else {
            return false;
        }

    } else {
        return false;
    }
}

QIODevice * DefaultStorage::getFile(const QString & aFileName) {
    QFile placeHolder(aFileName);
    QString absPath=QFileInfo (placeHolder).absoluteFilePath();
    if (!m_storageFiles.contains(absPath)) {
        return 0;
    } else {
        QFile * tmp = new QFile(absPath);
        return tmp;
    }
}

qint32 DefaultStorage::count() const {
    return m_storageFiles.allKeys().count();
}

bool DefaultStorage::calculateHash(const QString & aFileName, const qint64 & size,
        QByteArray & aOutHash) const {

    QByteArray bytes = QByteArray::number(size);
    bytes.append("_");
    bytes.append(aFileName.toUtf8());
    QByteArray hashValue = QCryptographicHash::hash(bytes,
            QCryptographicHash::Md5);

    aOutHash = hashValue.toHex();
    return true;
}

}


