/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "webnavigation.h"
#include <QObject>
#include "qwebframe.h"
#include "qwebpage.h"
#include "webtouchnavigation.h"
#include "webcursornavigation.h"
#include "webhtmltabbednavigation.h"
#include "webdirectionalnavigation.h"

namespace WRT {

/*!
    \class WebNavigation
    \since cwrt 1.0
    \brief cwrt navigation.

    \sa WebTouchNavigation, WebCursorNavigation, WebDirectionalNavigation, WebHtmlTabIndexedNavigation
*/
WebNavigation::WebNavigation(QObject* object, QWebPage* webPage) :
m_viewObject(object)
, m_webPage(webPage)
, m_webTouchNavigation(0)
, m_webCursorNavigation(0)
, m_webDirectionalNavigation(0)
, m_webHtmlTabIndexNavigation(0)
{
    connect(m_webPage->mainFrame(), SIGNAL(initialLayoutCompleted()), this, SLOT(setNavigationMode()));
}

WebNavigation::~WebNavigation()
{
    delete m_webTouchNavigation;
    delete m_webCursorNavigation;
    delete m_webDirectionalNavigation;
    delete m_webHtmlTabIndexNavigation;
}

/*!
    Sets the navigation mode
*/
void WebNavigation::setNavigationMode()
{
#if (defined(Q_OS_MAEMO6) || defined(Q_OS_MAEMO5))
    return;
#endif
    if (!m_webPage)
        return;

    delete m_webTouchNavigation;
    delete m_webCursorNavigation;
    delete m_webDirectionalNavigation;
    delete m_webHtmlTabIndexNavigation;

    m_webTouchNavigation = 0;
    m_webCursorNavigation = 0;
    m_webDirectionalNavigation = 0;
    m_webHtmlTabIndexNavigation = 0;

    QWebFrame* webFrame = m_webPage->currentFrame();
    if (webFrame) {
        QMultiMap<QString, QString> map = webFrame->metaData();
        if (map.value("touchnavigation") != "off" && map.value("navigation") != "none")
            m_webTouchNavigation = new WebTouchNavigation(m_viewObject, m_webPage);

        if (map.value("navigation") == "tabbed" || map.value("keynavigation") == "directional")
            m_webDirectionalNavigation = new WebDirectionalNavigation(m_viewObject, m_webPage);
        else if (map.value("keynavigation") == "tab-index")
            m_webHtmlTabIndexNavigation = new WebHtmlTabIndexedNavigation(m_viewObject, m_webPage);
        else if (map.value("navigation") == "cursor" || map.value("keynavigation") == "cursor")
            m_webCursorNavigation = new WebCursorNavigation(m_viewObject, m_webPage);
    }
}

#ifdef CWRT_BUILDING_TENONE
void WebNavigation::pauseNavigation()
{
    if (m_webTouchNavigation)
        m_viewObject->removeEventFilter(m_webTouchNavigation);

    if (m_webCursorNavigation)
        m_viewObject->removeEventFilter(m_webCursorNavigation);

    if (m_webDirectionalNavigation)
        m_viewObject->removeEventFilter(m_webDirectionalNavigation);

    if (m_webHtmlTabIndexNavigation)
        m_viewObject->removeEventFilter(m_webHtmlTabIndexNavigation);
}

void WebNavigation::resumeNavigation()
{
    if (m_webTouchNavigation)
        m_viewObject->installEventFilter(m_webTouchNavigation);

    if (m_webCursorNavigation)
        m_viewObject->installEventFilter(m_webCursorNavigation);

    if (m_webDirectionalNavigation)
        m_viewObject->installEventFilter(m_webDirectionalNavigation);

    if (m_webHtmlTabIndexNavigation)
        m_viewObject->installEventFilter(m_webHtmlTabIndexNavigation);
}
#endif
}
