/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "webkineticscroller.h"

#include <QAbstractKineticScroller>
#include <QGraphicsSceneMouseEvent>
#include <QGraphicsView>
#include <QGraphicsWebView>
#include <QWebElement>
#include <QWebHitTestResult>

namespace WRT {
namespace Maemo {

WebKineticScroller::WebKineticScroller(QGraphicsWebView* webView)
    : QGraphicsWidget()
    , QAbstractKineticScroller()
    , m_webView(webView)
    , m_ignoreEvents(false)
    , m_draggable(false)
{
    setFlag(QGraphicsItem::ItemHasNoContents, true);
    setFlag(QGraphicsItem::ItemClipsChildrenToShape, true);
    setFlag(QGraphicsItem::ItemClipsToShape, true);
    setAttribute(Qt::WA_OpaquePaintEvent, true);
    setFiltersChildEvents(true);

    activateScrollingForView();
}

WebKineticScroller::~WebKineticScroller()
{
    if (m_webView)
        m_webView->setParentItem(0);
}

void WebKineticScroller::activateScrollingForView()
{
    if (m_webView) {
        m_webView->setParentItem(this);
        m_webView->setAttribute(Qt::WA_OpaquePaintEvent, true);

        QWebFrame* frame = m_webView->page()->mainFrame();
        frame->setScrollBarPolicy(Qt::Vertical, Qt::ScrollBarAlwaysOff);
        frame->setScrollBarPolicy(Qt::Horizontal, Qt::ScrollBarAlwaysOff);
    }
}

bool WebKineticScroller::sceneEventFilter(QGraphicsItem* i, QEvent* e)
{
    bool res = false;
    if (i && (i == m_webView) && !m_ignoreEvents && m_webView->isEnabled()) {
        if (e->type() == QEvent::GraphicsSceneMouseRelease)
            m_draggable = false;

        switch (e->type()) {
        case QEvent::GraphicsSceneMousePress: {
            QPoint mousePressPosition = static_cast<QGraphicsSceneMouseEvent *>(e)->pos().toPoint();
            QWebHitTestResult hitResult = m_webView->page()->mainFrame()->hitTestContent(mousePressPosition);
            m_frame = scrollingFrameAt(hitResult);
            m_draggable = isDraggableElement(hitResult);
        }
        case QEvent::GraphicsSceneMouseRelease:
        case QEvent::GraphicsSceneMouseMove:
        case QEvent::GraphicsSceneMouseDoubleClick: {
            if (!m_draggable)
                res = handleMouseEvent(static_cast<QGraphicsSceneMouseEvent*>(e));
            break;
        }
        default:
            break;
        }
    }

    // prevent text selection
    if (!m_draggable && e->type() == QEvent::GraphicsSceneMouseMove)
        return true;

    return res ? true : QGraphicsWidget::sceneEventFilter(i, e);
}

QSize WebKineticScroller::viewportSize() const
{
    return size().toSize();
}

QPoint WebKineticScroller::maximumScrollPosition() const
{
    QWebFrame* frame = currentFrame();

    if (!frame)
        return QPoint(0, 0);

    QSize result = frame->contentsSize() - frame->geometry().size();
    return QPoint(qMax(0, result.width()), qMax(0, result.height()));
}

QPoint WebKineticScroller::scrollPosition() const
{
    QWebFrame* frame = currentFrame();
    return frame ? frame->scrollPosition() : QPoint(0, 0);
}

void WebKineticScroller::setScrollPosition(const QPoint& p, const QPoint& overShoot)
{
    QWebFrame* frame = currentFrame();
    if (frame)
        frame->setScrollPosition(p);
}

void WebKineticScroller::sendEvent(QGraphicsItem* i, QEvent* e)
{
    m_ignoreEvents = true;
    scene()->sendEvent(i, e);
    m_ignoreEvents = false;
}

QWebFrame* WebKineticScroller::currentFrame() const
{
    if (m_frame)
        return m_frame;

    if (m_webView)
        return m_webView->page()->mainFrame();

    return 0;
}

QWebFrame* WebKineticScroller::scrollingFrameAt(const QWebHitTestResult& hitResult) const
{
    QWebFrame* hitFrame = hitResult.frame();

    QSize range = hitFrame->contentsSize() - hitFrame->geometry().size();

    while (hitFrame && range.width() <= 1 && range.height() <= 1)
        hitFrame = hitFrame->parentFrame();

    return hitFrame;
}

bool WebKineticScroller::isDraggableElement(const QWebHitTestResult& hitResult) const
{
    return (hitResult.element().attribute("draggable", "false") == "true");
}

}
}
