/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "homescreenview.h"

#include "wrtpage.h"

#include <QPainter>
#include <QPaintEvent>
#include <QWebFrame>
#include <QX11Info>
#include <X11/X.h>
#include <X11/Xatom.h>
#include <X11/Xlib.h>

namespace WRT {
namespace Maemo {

HomeScreenView::HomeScreenView(const QString& id, QWidget* parent)
    : QWidget(parent)
    , m_page(0)
    , m_snapShot(0)
    , m_disabledColor(QColor(0, 0, 0, 63))
    , m_visible(false)
{
    setAttribute(Qt::WA_TranslucentBackground, true);

    setHildonAppletWindowType();
    setHildonAppletId(id);
}

HomeScreenView::~HomeScreenView()
{
}

void HomeScreenView::freeze()
{
    if (m_snapShot || !m_page)
        return;

    QRect rect(QPoint(0, 0), geometry().size());

    m_snapShot = new QPixmap(rect.size());
    m_snapShot->fill(Qt::transparent);
    QPainter painter(m_snapShot);

    m_page->mainFrame()->render(&painter, rect);

    // Setting composition mode to avoid darkening transparent areas.
    painter.setCompositionMode(QPainter::CompositionMode_SourceAtop);

    painter.setPen(Qt::NoPen);
    painter.fillRect(rect, m_disabledColor);
    m_page = 0;

    update();
}

void HomeScreenView::restore()
{
    if (m_snapShot) {
        delete m_snapShot;
        m_snapShot = 0;
    }
}

void HomeScreenView::setHildonAppletWindowType()
{
    Atom windowType = XInternAtom(QX11Info::display(), "_NET_WM_WINDOW_TYPE", False);
    Atom hildonwinType = XInternAtom(QX11Info::display(), "_HILDON_WM_WINDOW_TYPE_HOME_APPLET", False);
    XChangeProperty(QX11Info::display(), winId(), windowType, XA_ATOM, 32,
                    PropModeReplace, (unsigned char *) &hildonwinType, 1);
}

void HomeScreenView::setHildonAppletId(const QString& id)
{
    const char* idStr = id.toLatin1().constData();

    Atom appletId = XInternAtom(QX11Info::display(), "_HILDON_APPLET_ID", False);
    Atom utf8String = XInternAtom(QX11Info::display(), "UTF8_STRING", False);
    XChangeProperty(QX11Info::display(), winId(), appletId, utf8String, 8,
                    PropModeReplace, (unsigned char*)idStr, strlen(idStr));
}

void HomeScreenView::setPage(WrtPage* page)
{
    m_page = page;

    if (m_page) {
        m_page->mainFrame()->setScrollBarPolicy(Qt::Horizontal, Qt::ScrollBarAlwaysOff);
        m_page->mainFrame()->setScrollBarPolicy(Qt::Vertical, Qt::ScrollBarAlwaysOff);
        m_page->setViewportSize(geometry().size());
    }

    QPalette palette = m_page->palette();
    palette.setBrush(QPalette::Base, Qt::transparent);
    m_page->setPalette(palette);

    update();
}

void HomeScreenView::showEvent(QShowEvent* event)
{
    notifyVisibilityChanged(true);

    QWidget::showEvent(event);
}

void HomeScreenView::hideEvent(QHideEvent* event)
{
    notifyVisibilityChanged(false);

    QWidget::hideEvent(event);
}

bool HomeScreenView::x11Event(XEvent* event)
{
    Atom onCurrentDesktopAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_ON_CURRENT_DESKTOP", True);

    if (onCurrentDesktopAtom && (event->type == PropertyNotify)) {
        if (event->xproperty.atom == onCurrentDesktopAtom) {
            if (event->xproperty.state == PropertyNewValue)
                notifyVisibilityChanged(true);
            else if (event->xproperty.state == PropertyDelete)
                notifyVisibilityChanged(false);
        }
    }

    QWidget::x11Event(event);
    return false;
}

void HomeScreenView::notifyVisibilityChanged(bool visible)
{
    if (m_visible == visible)
        return;

    m_visible = visible;
    emit visibilityChanged(m_visible);
}

bool HomeScreenView::isVisible()
{
    return m_visible;
}

void HomeScreenView::paintEvent(QPaintEvent* event)
{
    if (m_snapShot) {
        QPainter painter(this);
        painter.drawPixmap(0, 0, *m_snapShot);
        return;
    }

    QWidget::paintEvent(event);
}

}
}
