/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef _SUPER_WIDGET_H_
#define _SUPER_WIDGET_H_


/*****************************************************************************
 * Super class for all Widget types
 *
 * *************************************************************************/

#include <QString>
#include <QDebug>
#include "widgetmanagerconstants.h"
#include "private/WidgetInfo.h"
#include "WidgetUtils.h"

typedef QMap<QString, QString> W3c2PlistMap;
typedef QMap<QString, QString> WidgetFeatures;

// Widget types
enum WidgetType
    {
        WidgetTypeWgz = 0,
        WidgetTypeW3c,
        WidgetTypeJIL,
        WidgetTypeUnknown
    };
typedef WidgetInfo WidgetManifestInfo;


enum WidgetInstallError
    {
        WidgetInstallSuccess = 0,
        WidgetValidSignature,
        WidgetUnZipBundleFailed,
        WidgetFindSignatureFailed,
        WidgetSignatureParsingFailed,
        WidgetSignatureOrSignedInfoMissing,
        WidgetSignatureRefExistFailed,
        WidgetSignatureRefValidationFailed,
        WidgetCertValidationFailed,
        WidgetSignatureValidationFailed,
        WidgetParseManifestFailed,
        WidgetRegistrationFailed,
        WidgetReplaceFailed,
        WidgetRmDirFailed,
        WidgetCapabilityNotAllowed,
        WidgetPlatformSpecificInstallFailed,
        WidgetCorrupted,
        WidgetSharedLibraryNotSigned,
        WidgetDriveLetterValidationFailed,
        WidgetTypeValidationFailed,
        WidgetSystemError,
        WidgetInstallPermissionFailed,
        WidgetUpdateFailed,
        WidgetUpdateVersionCheckFailed,
        WidgetUserConfirmFailed,
        WidgetInsufficientDiskSpace,
        WidgetInstallFailed,
        WidgetStartFileNotFound
    };
Q_DECLARE_METATYPE(WidgetInstallError)

enum WidgetUninstallError
    {
        WidgetUninstallSuccess = 0,
        WidgetUninstallFailed,
        WidgetUninstallPermissionFailed,
        WidgetUninstallCancelled,
        WidgetUninstallSuccessButIncomplete

    };

#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
namespace WRT {
     class Storage;
}
#endif

class WidgetInstaller;

class SuperWidget : public QObject
{
    Q_OBJECT
public:
    SuperWidget() {}
    SuperWidget( QString& widgetPath ) { m_widgetBundlePath = widgetPath; }
    virtual ~SuperWidget();

public:
    virtual QString launcherPath(const QString&) {return "";}
    WidgetInstallError install(const QString&){return WidgetInstallFailed;}
    static WidgetUninstallError uninstall(const QString &uniqueIdentifier,
                                          const bool removeData);
    static QString resourcePath(const QString& installPath);
    WidgetProperties* getProperties(const QString &pkgPath);
    virtual WidgetProperties* getProperties() {return m_widgetProperties;};

    virtual WidgetInstallError install( const QString&,
                                        const bool silent=false );

    /**
     * Installs a web widget.
     *
     * Emits following signals:
     * @emit installationError(WidgetInstallError)
     * @emit aboutToReplaceExistingWidget(QString)
     * @emit aboutToInstallWidgetWithFeatures(QList<QString>)
     * @emit installationSucceed()
     *
     * @param update is installation update
     * @return true if installation is successful; otherwise false
     */
    virtual WidgetInstallError install( const bool update=false ) = 0;

    virtual bool isValidWidgetType();
#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
    virtual SuperWidget* toWgz() { return this; }
#endif
    virtual SuperWidget* toWgt() { return this; }
    virtual void writeManifest( const QString& path="" );
    virtual bool unZipBundle( const QString& path="" );
    virtual bool parseManifest( const QString& path="",
                                const bool force=false );

    /**
     * Returns a metadata value by key and attribute
     * @param key a metadata element key
     * @attribute an attribute is an optional argument. With attribute you can get a value
     * of given attribute within specified key.
     */
    virtual QString value( const QString & key, const QString & attribute = QString("")) = 0;

    /**
     * Returns true if the key with the attribute exists in widget metadata; otherwise false.
     * @param key the metadata element key
     * @attribute the attribute is an optional argument. With attribute you can test whether a value
     * of given attribute within specified key exists in widget metadata.
     */
    virtual bool contains( const QString & key, const QString & attribute = QString("")) = 0;

public:
    static WidgetType getWidgetType(const QString&, const QString&);
    WIDGETUTILS_EXPORT static WidgetType getWidgetType(const QString&);
    static bool isValidWidgetType(const QString&, const QString&);
    static bool rmDir(const QString& );
    WIDGETUTILS_EXPORT static QString getLocalizedFile(const QString&, QString&, const bool=false, const QString& path="");
    WIDGETUTILS_EXPORT static bool getWidgetPathParts(QStringList& , const QString&);
    static bool getUniqueNameFromPath(const QString pkgPath, QString &uniqueName);

public:
    virtual void setWidgetBundlePath(const QString& path) {m_widgetBundlePath = path;};
    virtual void setWidgetUnZipPath(const QString& path) {m_widgetUnZipPath = path;};
    virtual void setWidgetInstallPath(const QString& path) {m_widgetInstallPath = path;};
    virtual void setWidgetCertificatePath(const QString& path){m_widgetCertificatePath = path;};
    virtual void setWidgetManifestInfo(WidgetManifestInfo& mi){m_manifest = &mi;};
    virtual QString widgetBundlePath() {return m_widgetBundlePath;};
    virtual QString widgetUnZipPath() {return m_widgetUnZipPath;};
    virtual QString widgetInstallPath(){return m_widgetInstallPath;};
    virtual QString widgetCertificatePath(){return m_widgetCertificatePath;};
    virtual QString& getContentType() {return m_contentType;};
    virtual QString& getSecureSessionString() {return m_secSessionStr;};
    virtual unsigned long getSize() {return m_size;};
    virtual void disableBackupRestoreValidation(bool /*disableUnsignWidgetSignCheck*/){};

public Q_SLOTS:

    /**
     * Try to continue installation process when and installation is about to do something.
     *
     * @see aboutToReplaceExistingWidget(QString)
     */
    void tryContinueInstallation();


Q_SIGNALS:
    void installProgress(int);

    /**
     * This signal is emitted whenever an error occurs during installation phase.
     *
     * If you intend to continue installation when error is signaled, you should
     * tryContinueInstallation() method when you're handling installation error.
     *
     */
    void installationError(WidgetInstallError error);

    /**
     * This signal is emitted whenever an existing widget is about to be replaced during installation
     * phase.
     *
     * If you intend to replace the widget when this message is signaled, you should invoke
     * tryContinueInstallation() method when you're handling this signal.
     */
    void aboutToReplaceExistingWidget(QString widgetTitle);

     /**
     * This signal is emitted whenever a widget with features is about to be installed.
     *
     *
     * If you intend to install the widget when this message is signaled, you should
     * tryContinueInstallation() method when you're handling this signal.
     */
    void aboutToInstallWidgetWithFeatures(QList<QString> capList);

    /**
     * This signal is emitted when a not signed widget is being installed.
     *
     * If you intend to install the widget when this message is signaled, you should invoke
     * tryContinueInstallation() method when you're handling this signal.
     *
     * @param errorCode tells reasons for installed widget being untrusted
     */
    void aboutToInstallUntrustedWidget(WidgetInstallError errorCode);

    /**
     * This signal is emitted whenever an existing widget is about to be replaced during installation
     * phase.
     */
    void installationSucceed();

    void promptFeatureList(QList<QString> capList);

protected:
    virtual void initialize(QString& rootDirectory) = 0;

    virtual bool findDigitalSignatures( QStringList&, const QString& path="" );

    virtual bool isValidDigitalSignature( WidgetInstallError & error,
                                          const QString& widgetPath="",
                                          bool unzip=false,
                                          bool cleanup=false );

    virtual bool findManifest( QString& ) { return false; }

    virtual WidgetProperties* widgetProperties() { return NULL; }

    virtual void saveStartFile( QString&, const QString& widgetPath="" );

    virtual bool getStartFile( QString&, const QString& widgetPath="" );

    virtual bool findStartFile( QString&, const QString& widgetPath="" );

    virtual bool findIcons( QStringList&, const QString& widgetPath="" );

    virtual bool findThumnail( const QString& ) { return false; }

    virtual bool findFeatures( WidgetFeatures&, const QString& widgetPath="" );

    virtual bool isFeatureAllowed( WidgetFeatures& ) { return false; }

    virtual bool getCertificateCn( QString&, const QString& widgetPath="",
                                   bool unbundle=false, bool cleanup=false );

    bool cleanup( const QString& path="" );
    bool registerWidget(const QString& startFile);

    QString getTempLocation();
    QString getCertificatePath();
    bool createDir(const QString& path);

    bool checkDiskSpaceForInstallation( const QString& widgetPath) const;

protected:
    virtual WidgetManifestInfo* getWidgetManifestInfo(){return m_manifest;};
    virtual WidgetType getWidgetType(){return m_widgetType;};
    virtual W3c2PlistMap* getWidgetMap(){return m_widgetMap;};
    virtual bool setWidgetRootPath(const QString& path=""){m_widgetRootPath = path; return true;};
    virtual QString widgetRootPath(){return m_widgetRootPath;};
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
    virtual WRT::Storage* getStorage() {return m_storage;};
#endif

protected :
    QString m_widgetBundlePath;          // path to the widget bundle.for eg: /home/user/Downloads
    QString m_widgetUnZipPath;           // path where widget can be unzipped for validation
    QString m_widgetInstallPath;         // path to install widget contents
    QString m_widgetCertificatePath;     // path to certificates
    QString m_widgetRootPath;            // path to widget root
    QString m_secSessionStr;
    bool m_continueInstallation;         // on user prompting continuation status
    unsigned long m_size;


    WidgetType m_widgetType;
    QString m_contentType;
    WidgetManifestInfo *m_manifest;
    WidgetProperties *m_widgetProperties;
    W3c2PlistMap *m_widgetMap;
    WidgetInstaller * m_widgetInstaller;

#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
    WRT::Storage *m_storage;
#endif
};

#endif
