/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "wrtwidgetcontainerBase.h"
#include "wrtwidgetcontainerBase_p.h"
#include "qwebframe.h"
#include "qwebpage.h"
#include "webwidgetpreference.h"
#include "WebAppRegistry.h"
#include "wrtcontroller.h"
#include "wrtpage.h"
#include "wrtnetworkaccessmanager.h"
#include "wrtsettings.h"
//#include "dialogsprovider.h"
#include "widgetcorelogs.h"
#include "WidgetProperties.h"
#include "widgetmanagerconstants.h"

#include "wrtapimanager.h"
#include "wrtapiconf.h"
#include "jswidgetinterface.h"
#include "widgetmanagerconstants.h"
#include "qwebsecurityorigin.h"

#include <Qt>
#include <QtCore>
#include <QtCore/qglobal.h>
#include <QVariant>
#include <QStringList>
//#include <QtGui/QDesktopServices>

using namespace WRT;


void Q_DECL_IMPORT qt_drt_whiteListAccessFromOrigin(const QString& sourceOrigin,
                                                 const QString& destinationProtocol,
                                                 const QString& destinationHost,
                                                 bool allowDestinationSubdomains);
// This deletes all whitelists.
void Q_DECL_IMPORT qt_drt_resetOriginAccessWhiteLists();


WidgetContainerBasePrivate::WidgetContainerBasePrivate(QWidget* parent, WebKitView webkitView) :
#ifdef CWRT_BUILDING_TENONE
  m_wrtController(new WrtController(0, webkitView))
#else
  m_wrtController(new WrtController(parent, webkitView))
#endif
  , m_widgetPreference(new WebWidgetPreference())
  , m_wrtApiManager(new  WRT::WrtApiManager(m_wrtController->wrtPage()))
  , m_widgetProperties(0)
  , m_wrtNetworkAccessManager(0)
{
#ifdef CWRT_BUILDING_TENONE
Q_ASSERT(0); // not implemented
#endif
}



WidgetContainerBasePrivate::~WidgetContainerBasePrivate()
{
    delete m_wrtController;
    delete m_widgetPreference;
    delete m_widgetProperties;
    if (m_wrtApiManager) {
        delete m_wrtApiManager;
    }
}

WidgetContainerBase::WidgetContainerBase(QWidget* parent, WebKitView webkitView) :
  d(new WidgetContainerBasePrivate(parent, webkitView))
{

    connect( wrtPage(), SIGNAL(frameCreated(QWebFrame*)), this, SLOT(frameCreated(QWebFrame*)) );
    connect( wrtPage(), SIGNAL(windowCloseRequested()), this, SIGNAL(windowCloseRequested()) );
    // FIXME if a widget is allowed to open a new widget window,
    //they both should have the same page group.
    static int pageGroup = 0;
    d->m_pageGroupName = "nokia_widget_page_group_" + QString::number(pageGroup);
    pageGroup++;
    wrtPage()->setPageGroupName(d->m_pageGroupName);
    d->m_wrtNetworkAccessManager = new WrtNetworkAccessManager(this, parent);
    connect( d->m_wrtNetworkAccessManager, SIGNAL(createConnection()), this, SIGNAL(createConnection()), Qt::QueuedConnection);
    connect( this, SIGNAL(connectionEstablished()), d->m_wrtNetworkAccessManager, SLOT(handleConnection()), Qt::QueuedConnection);
    connect( this, SIGNAL(connectionError()), d->m_wrtNetworkAccessManager, SLOT(connectionError()));
    connect( this, SIGNAL(wrtApisLoaded()), this, SLOT(postPrepareWrtApis()));
    wrtPage()->setNetworkAccessManager(d->m_wrtNetworkAccessManager);
    frameCreated(wrtPage()->mainFrame());

}

WidgetContainerBase::~WidgetContainerBase()
{
    delete d;
}

QUrl WidgetContainerBase::loadedUrl()
{
    if (wrtPage() && wrtPage()->mainFrame())
        return wrtPage()->mainFrame()->url();
    else
        return QUrl();
}

WrtPage* WidgetContainerBase::wrtPage() const
{
    return d->m_wrtController->wrtPage();
}

WrtController* WidgetContainerBase::wrtController() const
{
    return d->m_wrtController;
}

void WidgetContainerBase::setWidgetInfo(const QString& wgzPath, const QString& id)
{
    LOG("WebWidgetPreference::setWidgetInfo()");
    d->m_widgetPreference->setWidgetBundleId(id);
    QString webAppsBase = wgzPath;
    int len = wgzPath.length();
    if (webAppsBase.at(len-1) != QDir::separator())
        webAppsBase.append(QDir::separator());
    LOG("webAppsBase:" << webAppsBase);
    d->m_widgetPreference->setBasePath( webAppsBase );
    d->m_wrtNetworkAccessManager->setBasePath(webAppsBase);

    // Enable persistent storage for widget based on widget root. Persistent storage must be unique per
    // widget.
    wrtPage()->enableOfflineWebApplicationCache(DATA_PATH + id);
#ifndef __SYMBIAN32__
    wrtPage()->enableLocalStorage(DATA_PATH + id);
#else
    WebAppInfo info;
    WebAppRegistry::instance()->isRegistered(id, info);
    wrtPage()->enableLocalStorage(info.dataPath() + id);
#endif
}

void WidgetContainerBase::setWidgetProperties(const WidgetProperties* widgetProperties)
{
    WidgetProperties * oldprops = d->m_widgetProperties;

    if (widgetProperties)
        d->m_widgetProperties = new WidgetProperties(*widgetProperties);
    else
        d->m_widgetProperties = 0;

    delete oldprops;
}

const WidgetProperties* WidgetContainerBase::widgetProperties()
{
    return d->m_widgetProperties;
}

void WidgetContainerBase::loadWrtApis(JSApi::WidgetJavaScriptApi apilist) {
    d->m_wrtApiManager->loadApis(apilist);

    emit wrtApisLoaded();
}

void WidgetContainerBase::postPrepareWrtApis()
{
    WrtApiInterface* iface = d->m_wrtApiManager->apiObject(WidgetInterface);
    if (iface) {
        JSWidgetInterface* widget = static_cast<JSWidgetInterface*>(iface);
        connect(widget, SIGNAL(viewModeChangeRequested(QString)),
                this, SIGNAL(viewModeChangeRequested(QString)));
    }
}

WrtApiInterface* WidgetContainerBase::apiObject(QString name) {
    return d->m_wrtApiManager->apiObject(name);
}

void WidgetContainerBase::setCommandline(QString commandline)
{
    d->m_commandline = commandline;
    WrtApiInterface* iface = d->m_wrtApiManager->apiObject(WidgetInterface);
    if (iface){
        JSWidgetInterface* widget = static_cast<JSWidgetInterface*>(iface);
        widget->setCommandline(commandline);
    }
}

QString WidgetContainerBase::commandline()
{
    return d->m_commandline;
}

void WidgetContainerBase::setMessage(QString message)
{
    d->m_message = message;
    WrtApiInterface* iface = d->m_wrtApiManager->apiObject(WidgetInterface);
    if (iface){
        JSWidgetInterface* widget = static_cast<JSWidgetInterface*>(iface);
        widget->setMessage(message);
    }
}

QString WidgetContainerBase::message()
{
    return d->m_message;
}

void WidgetContainerBase::visibilityChanged(bool visible)
{
    WrtApiInterface* iface = d->m_wrtApiManager->apiObject(WidgetInterface);
    if (iface){
        JSWidgetInterface* widget = static_cast<JSWidgetInterface*>(iface);
        if (visible) {
            widget->onShow();
        } else {
            widget->onHide();
        }
    }
}

void WidgetContainerBase::whitelistAccess(const QUrl & url)
{
    QString widgetOrigin = url.scheme() + "://" + url.host();
    QString domain;
    qt_drt_whiteListAccessFromOrigin(widgetOrigin, "http", domain, true);
    qt_drt_whiteListAccessFromOrigin(widgetOrigin, "https", domain, true);
    qt_drt_whiteListAccessFromOrigin(widgetOrigin, "lib", domain, true);

    QWebSecurityOrigin::addLocalScheme("widget");
}

void WidgetContainerBase::viewModeChanged(QString viewMode)
{
    WrtApiInterface* iface = d->m_wrtApiManager->apiObject(WidgetInterface);
    if (iface) {
        JSWidgetInterface* widget = static_cast<JSWidgetInterface*>(iface);
        widget->onViewModeChange(viewMode);
    }
}

WebWidgetPreference* WidgetContainerBase::widgetPreference() const
{
    return d->m_widgetPreference;
}

void WidgetContainerBase::setWrtNetworkAccessManager(WrtNetworkAccessManager* wrtNetworkAccessManager)
{
    //I'm pretty sure the QWebPage deletes the network access manager
    wrtPage()->setNetworkAccessManager(wrtNetworkAccessManager);
    d->m_wrtNetworkAccessManager = wrtNetworkAccessManager;
}

WrtNetworkAccessManager* WidgetContainerBase::wrtNetworkAccessManager() const
{
    return d->m_wrtNetworkAccessManager;
}



void WidgetContainerBase::saveWidgetManifestUrl(const QString& url)
{
    const QString& widgetId = d->m_widgetPreference->widgetBundleId();
    WebAppRegistry* appRegistry = WebAppRegistry::instance();
    if (appRegistry)
        appRegistry->setWebAppAttribute(widgetId, ManifestUrl, url);

}

void WidgetContainerBase::frameCreated(QWebFrame *frame)
{
    Q_UNUSED(frame);
}

