/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "webwidgetpreference.h"
#include <WebAppRegistry.h>
#include <QString>
#include <QFile>
#include <QDataStream>
#include <QIODevice>
#include <QTemporaryFile>
#include "webjslogs.h"

using namespace WRT;

// CONSTANTS
const int KMaxKeyValueSize = 4096; //4 k
const QString KPrefsFile = "prefs.dat";

// ----------------------------------------------------------------------------
// WebWidgetPreferences::WebWidgetPreferences
// C++ Constructor
//
//
// ----------------------------------------------------------------------------
WebWidgetPreference::WebWidgetPreference() : m_load(false)
{
    m_preferences = new QHash<QString, PrefElement*>();
}

// ----------------------------------------------------------------------------
// WebWidgetPreferences::~WebWidgetPreferences
// Destructor
//
//
// ----------------------------------------------------------------------------
WebWidgetPreference::~WebWidgetPreference()
{
    save();
    if (m_preferences) {
        QList<QString> keys = m_preferences->keys();
        int count = keys.size();
        for ( int i=0; i<count; i++)
            {
            QString key = keys.at(i);
            PrefElement* pref = m_preferences->value(key);
            delete pref;
            }
        m_preferences->clear();
        delete m_preferences;
    }
}

void WebWidgetPreference::setWidgetBundleId(const QString& value)
{
    m_widgetBundleId = value;
}

QString WebWidgetPreference::widgetBundleId() const
{
    return m_widgetBundleId;
}

void WebWidgetPreference::setBasePath(const QString& value)
{
    m_basePath = value;
}

QString WebWidgetPreference::basePath() const
{
    return m_basePath;
}

// ----------------------------------------------------------------------------
// WebWidgetPreference::Preference
// Get preference for a key
//
//
// ----------------------------------------------------------------------------
bool WebWidgetPreference::preference( const QString& key, QString& value)
{
    bool rSuccess = false;

    LOG("WebWidgetPreference::preference()");

    if ( m_basePath.isNull() || (m_basePath.size() <= 0) )
        return rSuccess;
    LOG("m_basePath:" << m_basePath);

    if (!m_load) {
        load();
        m_load = true;
    }

    if ( key.isNull() )
        return rSuccess;

    if ( key.size() <= KMaxKeyValueSize && m_preferences->contains(key) )
        {
        PrefElement* pref = m_preferences->value( key );
        int size = pref->valueSize();

        if ( size > KMaxKeyValueSize )
            {
            // return contents from temp file whose name is stored
            // in the m_value member of m_preferences
            QFile file(pref->value());
            if ( !file.open(QIODevice::ReadOnly) )
                return rSuccess;

            QDataStream readStream( &file );
            readStream >> value;    // codescanner::leave
            rSuccess = true;
            }
        else if ( size >= 0 )
            {
            value = pref->value();
            rSuccess = true;
            }
        }

    return rSuccess;
}

// ----------------------------------------------------------------------------
// WebWidgetPreference::SetPreference
// set Preference for a key
//
//
// ----------------------------------------------------------------------------
void WebWidgetPreference::setPreference( const QString& key, const QString& value )
{
    LOG("WebWidgetPreference::setPreference()");

    if ( m_basePath.isNull() || (m_basePath.size() <= 0) )
        return;
    LOG("m_basePath:" << m_basePath);

    if (!m_load) {
        load();
        m_load = true;
    }

    if ( key.isNull() )
        return;

    if ( key.size() <= KMaxKeyValueSize )
        {
        // if hash has the key and its value
        // delete the old value later when the new value was successfully updated
        PrefElement* prefExisting = NULL;
        prefExisting = m_preferences->value( key );

        // remove the existing key if its value is set to Null
        if ( value.isNull() )
            {
            m_preferences->remove( key );
            }
        else if ( value.size() <= KMaxKeyValueSize )
            {
            PrefElement* pref = new PrefElement();
            pref->setValue( value );
            pref->setValueSize( value.size() );
            m_preferences->insert( key, pref );
            }
        else
            {
            // create a temp file
            QString prefFile(m_basePath);
            QTemporaryFile file(prefFile.append(KPrefsFile));
            if ( !file.open() )
                return;
            // save the value in temp file.
            QDataStream writeStream( &file );
            writeStream << value;    // codescanner::leave

            // successfully writing to file
            if ( file.error() == QFile::NoError)
                {
                // ok to make file permenant
                file.setAutoRemove(false);

                // create new preference element
                PrefElement* pref = new PrefElement();
                // m_value member of PrefElement contains the temp file name.
                pref->setValue( file.fileName() );
                pref->setValueSize( value.size() );
                // update new preference element
                m_preferences->insert( key, pref );
                }
            // else, the temporary file will be removed
            // and the old key/value pair will be kept
            }

        // now the new value is updated, it's safe to delete the old valueg
        if ( prefExisting )
            {
            prefExisting->setCleanFileFlag( true );
            delete prefExisting;
            }
        }
}


// ----------------------------------------------------------------------------
// WebWidgetPreference::save
// SAve preferences to persistent storage
//
//
// ----------------------------------------------------------------------------
void WebWidgetPreference::save()
{
    LOG("WebWidgetPreference::save()");

    if ( m_basePath.isNull() || (m_basePath.size() <= 0) )
        return;
    LOG("m_basePath:" << m_basePath);
    WebAppInfo info;
    WebAppRegistry::instance()->isRegistered(m_widgetBundleId, info);
    QString prefPath =  info.dataPath();
    QString prefFile(prefPath);
    QFile file(prefFile.append(KPrefsFile));
    if ( !file.open(QIODevice::WriteOnly) )
        return;

    QDataStream writeStream( &file );
    QList<QString> keys = m_preferences->keys();
    int count = keys.size();
    writeStream << count;    // codescanner::leave
    for ( int i=0; i<count; i++)
        {
        QString key = keys.at(i);
        PrefElement* pref = m_preferences->value(key);
        writeStream << key << pref->value() << pref->valueSize();    // codescanner::leave
        }
}

// ----------------------------------------------------------------------------
// WebWidgetPreference::load
// Load preferences from persistent storage
//
//
// ----------------------------------------------------------------------------
void WebWidgetPreference::load()
{
    LOG("WebWidgetPreference::load()");

    if ( m_basePath.isNull() || (m_basePath.size() <= 0) )
        return;
    LOG("m_basePath:" << m_basePath);
    WebAppInfo info;
    WebAppRegistry::instance()->isRegistered(m_widgetBundleId, info);
    QString prefPath = info.dataPath();
    QString prefFile(prefPath);
    QFile file(prefFile.append(KPrefsFile));
    if ( !file.open(QIODevice::ReadOnly) )
        return;

    int count = 0;
    QDataStream readStream( &file );
    readStream >> count;    // codescanner::leave
    for ( int i=0; i<count; i++)
        {
        QString key, value;
        int len = 0;
        readStream >> key >> value >> len;    // codescanner::leave
        PrefElement* pref = new PrefElement();
        pref->setValue( value );
        pref->setValueSize( len );
        m_preferences->insert( key, pref );
        }
}


// ----------------------------------------------------------------------------
// PrefElement::PrefElement
// C++ constructor
//
//
// ----------------------------------------------------------------------------
PrefElement::PrefElement() : m_valueSize (0), m_cleanFileFlag (false)
{
}


// ----------------------------------------------------------------------------
// PrefElement::~PrefElement
// Destructor
//
//
// ----------------------------------------------------------------------------
PrefElement::~PrefElement()
{
    // When we update the existing key with newer value:
    // if the value > 4k, which means a file was created to store the value;
    // we need to delete the old file.
    // If called from ~Preferences(), we shouldn't do this -- cleanFileFlag is false
    if ( m_cleanFileFlag && m_valueSize > KMaxKeyValueSize )
        {
        QFile::remove( value() );
        }
}


// ----------------------------------------------------------------------------
// PrefElement::setValue
// set value for a preference
//
//
// ----------------------------------------------------------------------------
void PrefElement::setValue( const QString& value )
{
    m_value  = value;
}
