/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QVariant>
#include <QtCore>
#include <QtGui>
#ifdef CWRT_BUILDING_TENONE
#include <HbMenu>
#include <HbAction>
#endif
#include "webjsmenuitem.h"
#include "webjsmenu.h"

using namespace WRT;

// LOCAL FUNCTION PROTOTYPES

// ----------------------------------------------------------------------------
// WebJSMenuItemPrivate::
//
//
//
// ----------------------------------------------------------------------------
//
WebJSMenuItemPrivate::WebJSMenuItemPrivate( WebJSMenu* menu,
                                            double id,
                                            const QString& label ) :
                                        m_menu(menu),
                                        m_id(id),
                                        m_label(label),
                                        m_dimmed(false)
{
}

WebJSMenuItemPrivate::~WebJSMenuItemPrivate()
{
    for (int i=0; i<m_children.count(); i++) {
        WebJSMenuItem* menuItem = m_children.at(i);
        delete menuItem;
    }
    m_children.clear();
}

// ----------------------------------------------------------------------------
// WebJSMenuItem::
//
//
//
// ----------------------------------------------------------------------------
//
WebJSMenuItem::WebJSMenuItem(WebJSMenu* menu, double id, const QString& label)
{
    d = new WebJSMenuItemPrivate(menu, id, label);
}

WebJSMenuItem::~WebJSMenuItem()
{
    delete d;
}

void WebJSMenuItem::append(WebJSMenuItem* child)
{
    d->m_children.append(child);
}

void WebJSMenuItem::remove(double id)
{
    int count = d->m_children.count();
    for (int i=0; i<count; i++)
    {
        WebJSMenuItem* menuItem = d->m_children.at(i);
        if (menuItem->id() == id)
        {
            d->m_children.removeAt(i);
            delete menuItem;
            break;
        }
    }
}

void WebJSMenuItem::setDimmed(bool dimmed)
{
    d->m_dimmed = dimmed;
}

WebJSMenuItem* WebJSMenuItem::getMenuItemById(double id)
{
    WebJSMenuItem* item = NULL;
    if (d->m_id == id)
        item = this;
    else {
        int count = d->m_children.count();
        for (int i=0; i<count; i++)
        {
            item = d->m_children.at(i)->getMenuItemById(id);
            if (item != NULL)
                break;
        }
    }
    return item;
}

WebJSMenuItem* WebJSMenuItem::getMenuItemByName(const QString& label)
{
    WebJSMenuItem* item = NULL;
    if (d->m_label == label)
        item = this;
    else {
        int count = d->m_children.count();
        for (int i=0; i<count; i++)
        {
            item = d->m_children.at(i)->getMenuItemByName(label);
            if (item != NULL)
                break;
        }
    }
    return item;
}

void WebJSMenuItem::clear()
{
    int count = d->m_children.count();
    for (int i=0; i<count; i++)
    {
        WebJSMenuItem* menuItem = d->m_children.at(i);
        menuItem->clear();
        delete menuItem;
    }
    d->m_children.clear();
}

bool WebJSMenuItem::hasChildren()
{
    bool has = false;
    int count = d->m_children.count();
    if (count>0)
        has = true;

    return has;
}

void WebJSMenuItem::constructUIMenu(QMenu* menu)
{
    if (d->m_dimmed)
        return;

    int count = d->m_children.count();
    if (count == 0 )
    {
        QAction* action = menu->addAction(d->m_label);
       // connect onselect callback with QAction's trigger() signal
       connect(action, SIGNAL(triggered()), this, SLOT(onSelectMenuItem()));
    }
    else
    {
        QMenu* uiItem = menu->addMenu(d->m_label);
       // connect onselect callback with QMenu's trigger() signal
        connect(uiItem, SIGNAL(triggered(QAction*)), this, SLOT(onSelectMenuItem()));

        for (int i=0; i<count; i++)
        {
            WebJSMenuItem* menuItem = d->m_children.at(i);
            menuItem->constructUIMenu(uiItem);
        }
    }
}

#ifdef CWRT_BUILDING_TENONE
void WebJSMenuItem::constructUIMenu(HbMenu * menu)
    {
        if (d->m_dimmed)
            return;

        int count = d->m_children.count();
        if (count == 0 )
        {
           HbAction* action = menu->addAction(d->m_label);
           // connect onselect callback with QAction's trigger() signal
           connect(action, SIGNAL(triggered()), this, SLOT(onSelectMenuItem()));
        }
        else
        {
        HbMenu* uiItem = menu->addMenu(d->m_label);
           // connect onselect callback with QMenu's trigger() signal
            connect(uiItem, SIGNAL(triggered(HbAction*)), this, SLOT(onSelectMenuItem()));

            for (int i=0; i<count; i++)
            {
                WebJSMenuItem* menuItem = d->m_children.at(i);
                menuItem->constructUIMenu(uiItem);
            }
        }
    }
#endif

void WebJSMenuItem::onSelectMenuItem()
{
    d->m_menu->onSelectMenuItem(d->m_id);
}

double WebJSMenuItem::id()
{
    return d->m_id;
}

QString& WebJSMenuItem::label()
{
    return d->m_label;
}

void WebJSMenuItem::printArray(int level)
{
    QString tab;
    switch (level)
    {
        case 1:
        default:
            tab = "\t";
            break;
        case 2:
            tab = "\t\t";
            break;
        case 3:
            tab = "\t\t\t";
            break;
        case 4:
            tab = "\t\t\t\t";
            break;
    }
    QString print = tab + d->m_label;
    qDebug() << print;
    WebJSMenuItem* item = NULL;
    int count = d->m_children.count();
    for (int i=0; i<count; i++)
    {
        item = d->m_children.at(i);
        item->printArray(level+1);
    }
}
