/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
*/

#include <QDebug>
#include <QUrl>
#include <QStringList>

#include "WgtWidget.h"
#include "SuperWidget.h"
#include "webwidget.h"
#include "w3csettingskeys.h"
#include "utils.h"
#include "wrtsettings.h"
#include "desktopfilereader.h"

namespace WRT {
    namespace Maemo {

        const QString WebWidget::W3CWIDGET = QString("W3CWidget");
        const QString WebWidget::PLAINWIDGET = QString("Plain");
        const QString TrustedDomain("TrustedWidgets");
        const uint WEBAPP_DEFAULT_WIDTH = 860;
        const uint WEBAPP_DEFAULT_HEIGHT = 410;

        // TODO : Internal keys should be generated automatically from X-DUIApplet-WRT keys and
        // from W3C Web Widget keys
        // Proposal: X-DUIApplet-WRT-Url should be modified to X-DUIApplet-WRT-Content which would
        // return content source similarly as content element does in W3C Widget.
        // 1) Store stripped "X-DUIApplet-WRT-xyz" keys' xyz parts as lowercase
        // 2) On top of desktop keys write W3C Widget keys.

        WebWidget::WebWidget(const QString path, QObject* parent) :
            QObject(parent), m_metadataPath(path), m_valid(true),
            m_minLandscape(-1, -1), m_maxLandscape(-1, -1),
            m_preferredLandscape(-1, -1), m_minPortrait(-1, -1),
            m_maxPortrait(-1, -1), m_preferredPortrait(-1, -1),
            m_widget(0), m_mode(WINDOWED), m_errorCode(NoError) {

            DesktopFileReader tmpMetadata(path);
            m_valid = tmpMetadata.isValid();
            if (m_valid) {
                init(tmpMetadata);
            } else {
                m_errorCode = MetadataError;
            }
        }

        WebWidget::~WebWidget() {
            if (m_widget) {
                delete m_widget;
                m_widget = 0;
            }
        }

        bool WebWidget::isValid() {
            return m_valid;
        }

        WebWidget::WebWidgetErrorCode WebWidget::errorCode() {
            return m_errorCode;
        }

        bool WebWidget::reloadWidget(QString language) {
            if (m_properties.value("type") == WebWidget::PLAINWIDGET
                    || !m_widget) {
                qDebug() << "Widget type is other";
                return false;
            }

            // TODO : Reading of config file should not contain any reference to DesktopFileReader
            DesktopFileReader tmpMetadata(m_metadataPath);
            m_locale = language;
            readConfigfile(tmpMetadata);
            return true;
        }

        //FIXME!!!! implement once dui has the ability
        //-1 for dimensions means keep original
        void WebWidget::setIcon(QString uri, int width, int height) const {
            qDebug() << "Setting icon " << uri << "width:height " << width
                    << ":" << height;
        }

        QVariant WebWidget::getPropertyValue(QString property) const {

            return m_properties.value(property);

        }

        // getters

        QString WebWidget::getName() const {
            return m_properties.value("name").toString();
        }

        QString WebWidget::getDescription() const {
            return m_properties.value("description").toString();
        }

        QString WebWidget::getVersion() const {
            return m_properties.value("version").toString();
        }

        QString WebWidget::getLocale() const {
            return m_locale;
        }

        QString WebWidget::getLicense() const {
            return m_properties.value("license").toString();
        }

        QString WebWidget::getUrl() const {
            return m_properties.value("path").toString();
        }

        QString WebWidget::getBrowsingCapacity() const {
            return m_properties.value("browsingcapacity").toString();
        }

        // TODO : Check and remove
        QString WebWidget::getWidgetResource() const {
            m_properties.value("widgetresource").toString();
            return m_properties.value("widgetresource").toString();
        }


        QString WebWidget::webAppId() const {
            return m_properties.value("webappid").toString();
        }

         QString WebWidget::splashScreen() const {
             if (type() == WebWidget::W3CWIDGET ) {
                 QString path = m_properties.value("icon").toString();
                 if (!path.isEmpty()) {
                     return path;
                 }
             }
             return QString("/usr/share/images/wrt_splashscreen.jpg");
        }

        QString WebWidget::protectedStorageName() const {
            return m_properties.value("protected-store").toString();
        }


        QString WebWidget::viewmode() const {
            switch (m_mode) {
            case MINIMIZED: return "minimized";
            case WINDOWED: return "windowed";
            case FULLSCREEN: return "fullscreen";
            }
            return "";
        }

        QStringList WebWidget::supportedViewModes() const {
            QString viewmodes = m_widget->value(W3CSettingsKey::WIDGET_VIEWMODES);
            return viewmodes.split(" ");
        }

        QString WebWidget::type() const {
            return m_properties.value("type").toString();
        }

        QSize WebWidget::size() const {
            return m_properties.value("size").toSize();
        }

        // FIXME: Remove this function and use widget registry to get this info
        WidgetProperties* WebWidget::widgetProperties() {
            return ((SuperWidget*)m_widget)->getProperties();
        }


        /**
         *  private for friend classes (allowed to be used from friend classes)
         */
        QString WebWidget::fileName() {
            return m_metadataPath;
        }

        QSize WebWidget::minSize(const WRT::Maemo::WebWidget::Orientation & orientation) const {
            if (type() == WebWidget::W3CWIDGET) {
                QSize land = size();
                return widgetSize(land, QSize(land.height(), land.width()), orientation);
            } else {
                return widgetSize(m_minLandscape, m_minPortrait, orientation);
            }
        }
        QSize WebWidget::maxSize(const WRT::Maemo::WebWidget::Orientation & orientation) const {
            if (type() == WebWidget::W3CWIDGET) {
                QSize land = size();
                return widgetSize(land, QSize(land.height(), land.width()), orientation);
            } else {
                return widgetSize(m_maxLandscape, m_maxPortrait, orientation);
            }
        }

        QSize WebWidget::preferredSize(const WRT::Maemo::WebWidget::Orientation & orientation) const {
            if (type() == WebWidget::W3CWIDGET) {
                QSize land = size();
                return widgetSize(land, QSize(land.height(), land.width()), orientation);
            } else {
                return widgetSize(m_preferredLandscape, m_preferredPortrait, orientation);
            }
        }

        QSize WebWidget::widgetSize(const QSize & landspace, const QSize & portrait, const WRT::Maemo::WebWidget::Orientation & orientation) const {
            // Landscape mode
             if (orientation == Landscape) {
                 if (m_mode == WINDOWED) {
                     return QSize(WEBAPP_DEFAULT_WIDTH, WEBAPP_DEFAULT_HEIGHT);
                 }

                 else if (landspace.isValid()) {
                     return landspace;
                 }
                 else {
                     return size();
                 }
             }
             // Portrait mode
             else {
                 if (m_mode == WINDOWED) {
                     return QSize(WEBAPP_DEFAULT_HEIGHT, WEBAPP_DEFAULT_WIDTH);
                 }
                 else if (portrait.isValid()) {
                     return portrait;
                 }
                 else {
                     QSize portSize = size();
                     portSize.transpose();
                     return portSize;
                 }
             }
        }

        /**
         *  private for this class (should not be called / used with friend classes)
         */
        void WebWidget::init(const DesktopFileReader & metadata) {
            // W3C widget is defined with 'WRT/Type' value W3CWidget
            // TODO: add another field 'type' in the desktop file for resolving this
            setViewMode(metadata);

            // parses the name of the protected store
            // in case it is empty we assume non-signed widget
            m_properties.insert("protected-store", metadata.value("WRT/Protected-Store"));
#ifdef Q_OS_MAEMO5
            m_properties.insert("type", metadata.value("Desktop Entry/Type"));
#else
            m_properties.insert("type", metadata.value("WRT/Type"));
#endif
            initSizeAttbutes(metadata);

            // W3C widget is invalid if WebAppId is not availabe
            if (!metadata.contains("WRT/WebAppId")) {
                m_valid = false;
                m_errorCode = DesktopFileError;
                return;
            }

            m_webAppId = metadata.value(
                    "WRT/WebAppId");

            m_properties.insert("webappid", m_webAppId);

            if (!metadata.contains("WRT/StartHtml")) {
                m_valid = false;
                m_errorCode = DesktopFileError;
                return;
            }

            QString path =
                    metadata.value("WRT/StartHtml");

            m_widgetRoot = WIDGET_INSTALL_PATH + m_webAppId;

            m_widget = new WgtWidget(m_widgetRoot);
            m_widget->setWidgetUnZipPath(m_widgetRoot);

            if (!m_widget->parseManifest() && type() == WebWidget::W3CWIDGET) {
                m_valid = false;
                m_errorCode = DesktopFileError;
                return;
            }

            m_widget->widgetProperties()->setId(m_webAppId);
            //parse manifest adds widget id to the end of install
            //so we need to set it here again. otherwise the installpath is /path/to/widgetA/widgetA
            //RAINE FIXME! should the WgtWidget be fixed !?
            m_widget->widgetProperties()->setInstallPath(m_widgetRoot);

            m_properties.insert("browsingcapacity", metadata.value(
                    "WRT/BrowsingCapacity"));

            if (metadata.contains("WRT/SecureStore")) {
                m_properties.insert("secstore", metadata.value(
                        "WRT/SecureStore"));
            }

            // name of the .desktop file without path
            m_properties.insert("desktopfilename",
                                metadata.fileName().section('/', -1));

            // TODO fixme: This a fix for widgetresource
            // Raine: widgetresource should be decrecated same key between desktop-file and W3C should be used
            // instead (X-DUIApplet-WRT-Url -> Url part of this key). This applies to all keys.
            m_properties.insert("widgetresource", QUrl(m_widgetRoot).path());

            WRT::WrtSettings* m_settings = WRT::WrtSettings::createWrtSettings();
            m_locale = m_settings->valueAsString("UserAgentLanguage");
            if (m_locale.isEmpty() || m_locale.compare(" ") == 0)
                m_locale = QLocale::system().name().toLower().replace(QString("_"),QString("-"));

            readConfigfile(metadata);
            //check Capabilities
            WebAppRegistry * registry = WebAppRegistry::instance();
            WebAppInfo webInfo;
            QString appId = this->widgetProperties()->id();
            if (!registry->isRegistered(appId, webInfo)) {
                if (type() == WebWidget::W3CWIDGET){
                    m_valid = false;
                    m_errorCode = RegistrationError;
                    return;
                }
            }
            //Check from registry if policy files have been updated
            if (webInfo.capabilityCheck()) {
                WidgetFeatures features;
                WRT::SecSession* secureSession;
                QString trustDn;
                QString sigId;
                bool res = m_widget->findFeatures(features);
                if (m_widget->createSecuritySession(&secureSession,trustDn,sigId,true)) {
                    if (res || (trustDn.compare(TrustedDomain,Qt::CaseInsensitive) == 0)) {
                        qDebug() << "Found features at runtime capability check";
                        if (!m_widget->isFeatureAllowed(features,secureSession,true)) {
                            qDebug()<<"Feature NOT ALLOWED";
                            m_valid = false;
                            m_errorCode = CapabilityValidationError;

                        } else {
                            //capability check ok then save secsession file
                            QString resourcesDir = this->widgetProperties()->resourcePath();
                            QString secureSessionString = this->widgetProperties()->secureSessionString();
                            if (!resourcesDir.isNull() && !resourcesDir.isEmpty())
                               secureSession->saveSecsession(resourcesDir,secureSessionString);
                        }
                    }
                }
                //update the capabilityCheck Flag to false in WebappRegistry
                registry->setCapabilityCheck(appId,false);
            }

            // TODO : Check that widget types are working with JS test suites
            // IS THIS WRT/TestSuite still needed.
            if (metadata.contains("WRT/TestSuite")) {
                m_properties.insert("path", metadata.value(
                        "WRT/TestSuite"));
            }
            else {
                QString startFile = "";
                m_widget->findStartFile(startFile, m_widgetRoot);
                startFile = QString("widget://" + m_webAppId + "/" + startFile);
                m_properties.insert("path", startFile);
             }
        }

        void WebWidget::initSizeAttbutes(const DesktopFileReader & metadata) {
            if (metadata.contains("DUI/X-DUIApplet-MinimumSize-Landscape")) {
                qDebug() << "contains X-DUIApplet-MinimumSize";
                m_minLandscape = Utils::parseSize( metadata.value(
                        "DUI/X-DUIApplet-MinimumSize-Landscape"));
            }

            if (metadata.contains("DUI/X-DUIApplet-MinimumSize-Portrait")) {
                m_minPortrait = Utils::parseSize( metadata.value(
                        "DUI/X-DUIApplet-MinimumSize-Portrait"));
            }

            if (metadata.contains("DUI/X-DUIApplet-PreferredSize-Landscape")) {
                m_preferredLandscape = Utils::parseSize( metadata.value(
                        "DUI/X-DUIApplet-PreferredSize-Landscape"));
            }

            if (metadata.contains("DUI/X-DUIApplet-PreferredSize-Portrait")) {
                m_preferredPortrait = Utils::parseSize( metadata.value(
                        "DUI/X-DUIApplet-PreferredSize-Portrait"));
            }

            if (metadata.contains("DUI/X-DUIApplet-MaximumSize-Landscape")) {
                m_maxLandscape = Utils::parseSize( metadata.value(
                        "DUI/X-DUIApplet-MaximumSize-Landscape"));
            }

            if (metadata.contains("DUI/X-DUIApplet-MaximumSize-Portrait")) {
                m_maxPortrait = Utils::parseSize( metadata.value(
                        "DUI/X-DUIApplet-MaximumSize-Portrait"));
            }

        }

        bool WebWidget::readConfigfile(const DesktopFileReader & metadata) {

            // FIXME : What if set language is invalid. We could try all the languages that we
            // get from languages list.

            // size
            int width = 150;
            int height = 300;
            if (m_widget->contains(W3CSettingsKey::WIDGET_WIDTH))
                width = m_widget->value(W3CSettingsKey::WIDGET_WIDTH).toInt();
            if (m_widget->contains(W3CSettingsKey::WIDGET_HEIGHT))
                height = m_widget->value(W3CSettingsKey::WIDGET_HEIGHT).toInt();
            m_properties.insert("size", QSize(width, height));

            // name
            if (m_widget->contains(W3CSettingsKey::WIDGET_NAME)) {
                m_properties.insert("name", m_widget->value(
                        W3CSettingsKey::WIDGET_NAME));
            }
            else if (metadata.contains("Desktop Entry/Name")) {
                m_properties.insert("name",
                        metadata.value("Desktop Entry/Name"));
            }

            //description
            if (m_widget->contains(W3CSettingsKey::WIDGET_DESCRIPTION)) {
                m_properties.insert("description", m_widget->value(
                        W3CSettingsKey::WIDGET_DESCRIPTION));
            }

            // version
            if (m_widget->contains(
                    W3CSettingsKey::WIDGET_VERSION)) {
                m_properties.insert("version", m_widget->value(
                        W3CSettingsKey::WIDGET_VERSION));
            }

            // license text
            if (m_widget->contains(
                    W3CSettingsKey::WIDGET_LICENSE)) {
                m_properties.insert("license", m_widget->value(
                        W3CSettingsKey::WIDGET_LICENSE));
            }

            // id
            if (m_widget->contains(W3CSettingsKey::WIDGET_ID)) {
                m_properties.insert("id", m_widget->value(
                        W3CSettingsKey::WIDGET_ID));
            }

            // viewmodes

            if (m_widget->contains(W3CSettingsKey::
                                           WIDGET_VIEWMODES)) {
                m_properties.insert("viewmodes", m_widget->value(
                                        W3CSettingsKey::WIDGET_VIEWMODES));
            }

            // author name
            if (m_widget->contains(
                    W3CSettingsKey::WIDGET_AUTHOR)) {
                m_properties.insert("authorname", m_widget->value(
                        W3CSettingsKey::WIDGET_AUTHOR));
            }

            // author email
            if (m_widget->contains(
                    W3CSettingsKey::WIDGET_AUTHOR, "email")) {
                m_properties.insert("authoremail", m_widget->value(
                        W3CSettingsKey::WIDGET_AUTHOR, "email"));
            }

            // author uri
            if (m_widget->contains(
                    W3CSettingsKey::WIDGET_AUTHOR, "uri")) {
                m_properties.insert("authoruri", m_widget->value(
                        W3CSettingsKey::WIDGET_AUTHOR, "uri"));
            }

            // icon


            if (this->widgetProperties() && !this->widgetProperties()->iconPath().isEmpty()) {
                QString icon = this->widgetProperties()->iconPath();
                m_properties.insert("icon", icon);
            } else if (metadata.contains("Desktop Entry/Icon")) {
                m_properties.insert("icon",
                        metadata.value("Desktop Entry/Icon"));
            }
            return true;
        }

        void WebWidget::setViewMode(QString mode){
            if (mode == "minimized") {
                m_mode = MINIMIZED;
            } else if (mode == "fullscreen") {
                m_mode = FULLSCREEN;
            } else {
                // if mode is "windowed" or "maximized"; also a fallback for unknown modes
                m_mode = WINDOWED;
            }
        }

        void WebWidget::setViewMode(const DesktopFileReader & metadata) {
            QString viewmode = metadata.value("WRT/ViewMode", "windowed");
            setViewMode(viewmode);
        }

        bool WebWidget::contains(const QString & key, const QString & attribute) const {
            return m_widget->contains(key, attribute);
        }

        QString WebWidget::value(const QString & key, const QString & attribute) const {
            return m_widget->value(key, attribute);
        }

    }

}
