/*
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Author:  Jonas Gastal <jgastal@profusion.mobi>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libhildondesktop/libhildondesktop.h>
#include "hd-plugin-loader-w3c.h"
#include <string.h>
#include <stdlib.h>
#include <gio/gio.h>

#define HD_PLUGIN_LOADER_W3C_GET_PRIVATE(obj) \
        (G_TYPE_INSTANCE_GET_PRIVATE((obj), HD_TYPE_PLUGIN_LOADER_W3C, HDPluginLoaderW3CPrivate))

G_DEFINE_TYPE(HDPluginLoaderW3C, hd_plugin_loader_w3c, HD_TYPE_PLUGIN_LOADER);


struct webwidget_ {
  GFileMonitor *monitor;
  GFile *fd;
  GPid pid;
  gchar *desktop_file;
};

typedef struct webwidget_ webwidget;

static void delete_webwidget(webwidget *widget);

static GList* list = NULL;

static void desktop_file_deleted(GFileMonitor *monitor, GFile *file, GFile *other, GFileMonitorEvent event, gpointer data)
{
    if (event != G_FILE_MONITOR_EVENT_DELETED)
        return;

    webwidget *widget = (webwidget*)data;
    kill(widget->pid, SIGTERM);
    delete_webwidget(widget);
}

static webwidget* create_webwidget(const gchar *name, GPid pid)
{
    webwidget* widget = (webwidget *)malloc(sizeof(webwidget));
    if (widget == NULL)
        return NULL;

    widget->fd = g_file_new_for_path(name);
    widget->desktop_file= g_strdup(name);
    widget->monitor = g_file_monitor_file(widget->fd, G_FILE_MONITOR_NONE, NULL, NULL);
    widget->pid = pid;
    g_signal_connect(widget->monitor, "changed", G_CALLBACK(desktop_file_deleted), (gpointer)(widget));
    return widget;
}

static void delete_webwidget(webwidget *widget)
{
    GList *element = g_list_find(list, widget);

    if (element == NULL)
        return;

    list = g_list_remove(list, widget);
    g_file_monitor_cancel(widget->monitor);
    g_free(widget->desktop_file);
    g_free(widget);
}

static void hd_plugin_loader_w3c_destroy_plugin_id(gpointer data)
{
    gchar *plugin_id = data;
    g_free(plugin_id);
}

static void hd_plugin_loader_w3c_destroy_child_watch(gpointer data)
{
    /* object died, but not the plugin process, stop listening to it */
    guint source = (long)data;
    g_source_remove(source);
}

static void hd_plugin_loader_w3c_destroy_webwidget(gpointer data)
{
    webwidget *widget = (webwidget*)data;
    delete_webwidget(widget);
}

static void hd_plugin_loader_w3c_destroy_plugin(GPid pid, gint status, gpointer data) {
    GObject *object = G_OBJECT(data);
    const gchar *plugin_id = g_object_get_data(object, "plugin-id");
    /* don't let the source be removed, we are removing it right now */
    g_source_remove((long)g_object_steal_data(object, "pid-watch"));
    g_spawn_close_pid(pid);
    delete_webwidget(g_object_steal_data(object, "webwidget"));


    /** BIG FAT DISCLAIMER:
      *
      * Although hd_plugin_manager expects loaders to return a simple
      * GObject and hildon-desktop/hildon-home does handle any X11 window
      * with correct properties (_HILDON_*) set, the hd_plugin_manager
      * relies on GObjects actually being hd_home_plugin_item, which
      * inherit from GtkWindow/GtkWidget, to listen for its "delete-event"
      * to emit plugin was actually closed.
      *
      * The correct way would be to fix hildon-home to provide a way to let
      * plugins inform they were removed, not just those that are
      * hd_home_plugin_item, but it is not possible to fix this,
      * thus we're writing directly to configuration file.
      *
      * Although this is a bit hackish, it will always work since
      * plugin manager will monitor the directory/file for changes,
      * acting properly.
      */
    HDConfigFile *cfg = hd_config_file_new_with_defaults("home.plugins");
    GKeyFile *file = hd_config_file_load_file(cfg, FALSE);
    if (!file) {
        g_warning("w3c plugin destroy: could not load configuration file.");
        g_object_unref(cfg);
    }
    GError *error = NULL;
    g_key_file_remove_group(file, plugin_id, &error);
    if (error) {
        g_warning("could not remove group %s from config file: %s",
                  plugin_id, error->message);
        g_error_free(error);
        goto end;
    }
    g_print("successfully removed plugin group %s from config file\n", plugin_id);
    if (!hd_config_file_save_file(cfg, file)) {
        g_warning("could not save config file.");
        goto end;
    }
    g_print("successfully saved config file\n");

    end:
    g_object_unref(cfg);
    g_key_file_free(file);
}

static gboolean is_loaded(gchar *filename)
{
    gint list_size = g_list_length(list);
    GList *list_pointer = list;
    gpointer data = NULL;
    int cont;
    for (cont = 0; cont < list_size; cont++) {
        data = list_pointer->data;
        if (!g_strcmp0(((webwidget *) data)->desktop_file, filename))
            return TRUE;
        list_pointer = list_pointer->next;
    }

    return FALSE;
}

static GObject *hd_plugin_loader_w3c_load(HDPluginLoader *loader, const gchar *plugin_id, GKeyFile *keyfile, GError **error)
{
    GObject *object = g_object_new(G_TYPE_OBJECT, NULL);
    if(!object)
        return NULL;

    GPid pid;
    gchar *desktop_file = g_strdup_printf("/usr/share/applications/hildon-home/%s", plugin_id);
    if (!desktop_file) {
        g_object_unref(object);
        return NULL;
    }
    desktop_file[strlen(desktop_file) - 2] = '\0';
    if (is_loaded(desktop_file)) {
        g_free(desktop_file);
        g_object_unref(object);
        return NULL;
    }

    gchar *argv[3] = {"/usr/bin/webwidgetrunner", desktop_file, NULL};
    if(g_spawn_async("/usr/share/applications/hildon-home", argv, NULL, G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_DO_NOT_REAP_CHILD, NULL, NULL, &pid, error)) {
        guint source = g_child_watch_add
                       (pid, hd_plugin_loader_w3c_destroy_plugin, object);
        webwidget *new_widget = create_webwidget(desktop_file, pid);
        if (!new_widget) {
            kill(pid, SIGTERM);
            g_free(desktop_file);
            g_object_unref(object);
            return NULL;
        }
        list = g_list_prepend(list, (gpointer) new_widget);
        g_object_set_data_full
                (object, "pid-watch", (void *)(long)source,
                 hd_plugin_loader_w3c_destroy_child_watch);
        g_object_set_data_full
                (object, "plugin-id", g_strdup(plugin_id),
                 hd_plugin_loader_w3c_destroy_plugin_id);
        g_object_set_data_full
                 (object, "webwidget", (gpointer *)new_widget,
                 hd_plugin_loader_w3c_destroy_webwidget);
    } else {
        g_object_unref(object);
        object =  NULL;
    }
    g_free(desktop_file);
    return object;
}

struct _HDPluginLoaderW3CPrivate
{
    gboolean initialised;
};

static void hd_plugin_loader_w3c_finalize(GObject *loader)
{
    HDPluginLoaderW3CPrivate *priv;

    g_return_if_fail(loader != NULL);
    g_return_if_fail(HD_IS_PLUGIN_LOADER_W3C(loader));

    priv = HD_PLUGIN_LOADER_W3C(loader)->priv;

    G_OBJECT_CLASS(hd_plugin_loader_w3c_parent_class)->finalize(loader);
}

static void hd_plugin_loader_w3c_init(HDPluginLoaderW3C *loader)
{
    loader->priv = HD_PLUGIN_LOADER_W3C_GET_PRIVATE(loader);

    loader->priv->initialised = TRUE;
}

static void hd_plugin_loader_w3c_class_init(HDPluginLoaderW3CClass *c)
{
    GObjectClass *object_class;
    HDPluginLoaderClass *loader_class;

    object_class = G_OBJECT_CLASS(c);
    loader_class = HD_PLUGIN_LOADER_CLASS(c);

    object_class->finalize = hd_plugin_loader_w3c_finalize;

    loader_class->load = hd_plugin_loader_w3c_load;

    g_type_class_add_private(object_class, sizeof(HDPluginLoaderW3CPrivate));
}

G_MODULE_EXPORT gchar *hd_plugin_loader_module_type(void)
{
    return "W3CWidget";
}

G_MODULE_EXPORT HDPluginLoader *hd_plugin_loader_module_get_instance(void)
{
    return g_object_new(HD_TYPE_PLUGIN_LOADER_W3C, NULL);
}
