/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QHash>
#include <QList>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QAuthenticator>
#include <QSslError>
#include <QDesktopServices>
#include <QDir>
#include <QCoreApplication>

#include "wrtpage.h"
#include "wrtsettings.h"
#include "wrtcookiejar.h"
#include "SuperWidget.h"
#include "wrtnetworkaccessmanager.h"
#include "wrtnetworkaccessmanager_p.h"
#include "wrtnetworkfilterlistener.h"
#include "wrtnetworkerrorreply.h"
#include "wrtnetworkreply.h"
#include "wrtwidgetreply.h"
#include "w3celement.h"
#include "secsession.h"
#include "SchemeHandler.h"
#include "dialogsprovider.h"

#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
#include "storage.h"
#endif


QString qt_websettings_localStorageDatabasePath(QWebSettings* settings);

//Methods not available in the current Maemo release of QtWebKit
#if !defined(Q_OS_MAEMO5) && !defined(Q_OS_MAEMO6) && !defined(CWRT_BUILDING_VASCO) || defined(CWRT_BUILDING_QTUNSTABLE)
// This allows you to whitelist access for a widget.
void Q_DECL_IMPORT qt_drt_whiteListAccessFromOrigin(const QString& sourceOrigin,
                                                 const QString& destinationProtocol,
                                                 const QString& destinationHost,
                                                 bool allowDestinationSubdomains);
#endif

namespace WRT {

const int maxHostRequestHeaderSize = 4096;

#if !defined(Q_OS_MAEMO5) && !defined(Q_OS_MAEMO6) && !defined(CWRT_BUILDING_VASCO) || defined(CWRT_BUILDING_QTUNSTABLE)
    // HACK: this is a temp solution until webkit provides better API to keep track of
    // the content of current white list.  This white list might be out of sync with the
    // effective white list in webkit resulting in multiple adds to the webkit's white list
    //
    // TODO update this implementation when webkit provides API to retrieve the current
    // white list content
static QSet<QString> whiteList;
#endif

/*!
 *
 */
WrtNetworkAccessManagerPrivate::WrtNetworkAccessManagerPrivate(WidgetContainerBase* widgetContainerBase, QWidget* wrtView) :
 m_schemeHandler(new SchemeHandler())
, m_secSession(0)
, m_userAgent(0)
, m_cookieJar(new CookieJar())
, m_qDiskCache(0)
, m_widgetContainerBase(widgetContainerBase)
, m_wrtPage(widgetContainerBase->wrtPage())
, m_wrtView(wrtView)
, m_widgetId(0)
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
, m_storage(Storage::createInstance(WIDGET_STORAGE))
#endif
{
}

/*!
 *
 */
WrtNetworkAccessManagerPrivate::~WrtNetworkAccessManagerPrivate()
{
    delete m_schemeHandler;
    delete m_cookieJar;
    delete m_qDiskCache;
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
    delete m_storage;
#endif
}

/*!
 *
 */
WrtNetworkAccessManager::WrtNetworkAccessManager(WidgetContainerBase* widgetContainerBase, QWidget* wrtView) : QNetworkAccessManager(widgetContainerBase->wrtPage()),
d(new WrtNetworkAccessManagerPrivate(widgetContainerBase, wrtView))
{
    QNetworkAccessManager::setCookieJar(d->m_cookieJar);

    setupCache();

    m_connected = false;
    connect(this, SIGNAL(authenticationRequired(QNetworkReply *, QAuthenticator *)), this, SLOT(slotAuthenticationRequired(QNetworkReply *, QAuthenticator *)));

    connect(this, SIGNAL(proxyAuthenticationRequired(const QNetworkProxy & , QAuthenticator * )), this, SLOT(slotProxyAuthenticationRequired(const QNetworkProxy &, QAuthenticator *)));

    // handles SSL errors such as expired certificates
    connect(this, SIGNAL(sslErrors(QNetworkReply*,QList<QSslError>)), this, SLOT(slotSslError(QNetworkReply*,QList<QSslError>)));
}

/*!
 *
 */
WrtNetworkAccessManager::~WrtNetworkAccessManager()
{
    delete d;

}

/*!
 * clears cookies
 */
void WrtNetworkAccessManager::clearCookies()
{
    ((CookieJar*)d->m_wrtPage->networkAccessManager()->cookieJar())->clear();
}

/*!
 *
 */
void WrtNetworkAccessManager::setSecuritySession(SecSession* secSession)
{
    d->m_secSession = secSession;
}

/*!
 *
 */
SecSession* WrtNetworkAccessManager::securitySession() const
{
    return d->m_secSession;
}

/*!
 * sets the host header for Http request
 * currently supports setting of accept headers only
 * @param name  :  Http request header name
 * @param value :  Http request header value to be set
 */
void WrtNetworkAccessManager::setHostRequestHeader(const QString& name, const QString& value)
{
    if (name.isNull()
        || name.isEmpty()
        || name.length() > maxHostRequestHeaderSize
        || value.length() > maxHostRequestHeaderSize)
        return;

    QString nameLowerCase = name.toLower();
    if (nameLowerCase == "accept" && d->m_hostRequestHeaders.contains(nameLowerCase)) {
        // accept header adds new values. Any other header replaces existing values
        if (!(value.isNull() || value.isEmpty())) {
            // rfc 2616 4.2 :
            // Multiple message-header fields with the same field-name MAY be
            // present in a message if and only if the entire field-value for that
            // header field is defined as a comma-separated list [i.e., #(values)].

            QString header = d->m_hostRequestHeaders.find(nameLowerCase).value() + " , " + value;
            if (header.length() <= maxHostRequestHeaderSize)
                d->m_hostRequestHeaders.insert(nameLowerCase, header);
        }
    } else if (nameLowerCase == "accept"
            || nameLowerCase == "user-agent"
            || nameLowerCase.startsWith("x-")) {
            d->m_hostRequestHeaders.insert(nameLowerCase, value);
    }
}

/*!
Public Slots:
    void slotAuthenticationRequired(QNetworkReply *, QAuthenticator *);
    void slotProxyAuthenticationRequired(const QNetworkProxy &, QAuthenticator *);
*/
/*!
 *
 */
void WrtNetworkAccessManager::slotAuthenticationRequired(QNetworkReply *reply, QAuthenticator *athenticator)
{
    QString username("");
    QString password("");

    if (d->m_wrtView != NULL) {
            if (DialogsProvider::getUsernamePassword(d->m_wrtView, username, password)) {
                    athenticator->setUser(username);
                    athenticator->setPassword(password);
        }

    }
}
/*!
 *
 */
void WrtNetworkAccessManager::slotProxyAuthenticationRequired(const QNetworkProxy & networkProxy, QAuthenticator* authenticator)
{
    QString username("");
    QString password("");

    if (d->m_wrtView != NULL) {
            if (DialogsProvider::getUsernamePassword(d->m_wrtView, username, password)) {
                    authenticator->setUser(username);
                    authenticator->setPassword(password);
        }

    }
}

void WrtNetworkAccessManager::slotSslError(QNetworkReply* reply, const QList<QSslError>& errors)
{
    qWarning() << Q_FUNC_INFO << errors;
    // ignore SSL errors for now
    // TODO: define common policy for SSL errors
    reply->ignoreSslErrors();
}


/*!
 *
 */
QNetworkReply* WrtNetworkAccessManager::createRequest(Operation op, const QNetworkRequest &request, QIODevice *outgoingData)
{
    QNetworkRequest networkRequest = request;

    WRT_CAPTURE_TRACE((d->m_wrtPage), "WrtNetworkAccessManager::createRequest", networkRequest.url().toEncoded());

    if (!networkRequest.url().scheme().compare("http", Qt::CaseInsensitive) ||
        !networkRequest.url().scheme().compare("https", Qt::CaseInsensitive)){
        //Connection request.
        if (!m_connected)
            emit createConnection();
        if (d->m_secSession) {
            QStringList capability("Network");
            if (!d->m_secSession->isAllowed(capability)) {
                QNetworkReply* reply = new WrtNetworkErrorReply(QNetworkReply::ContentAccessDenied, "Not allowed", networkRequest.url());
                //a finished signal will be sent and the member is invoked as soon as the application enters the main event loop
                QMetaObject::invokeMethod(reply, "finished", Qt::QueuedConnection);
                return reply;
            }
        }
    }

    QHash<QString, QString>::iterator it;
    for (it = d->m_hostRequestHeaders.begin(); it != d->m_hostRequestHeaders.end(); it++) {
        if (it.key().toLower().compare("accept")==0) {
                QString accept = networkRequest.rawHeader("accept");
                accept += " , ";
                accept += QString(it.value());
                networkRequest.setRawHeader("accept", accept.toUtf8());
        }
        else
            networkRequest.setRawHeader(it.key().toUtf8(), it.value().toUtf8());
    }
    // prevent access to other widget's data
    if (((!networkRequest.url().scheme().compare("file", Qt::CaseInsensitive) ||
         !networkRequest.url().scheme().compare("widget", Qt::CaseInsensitive) ||
         !networkRequest.url().scheme().compare("sandbox", Qt::CaseInsensitive))
        && fileAccessBlocked(networkRequest.url().path())) ||
        (networkRequest.url().toString().startsWith(d->m_appPath, Qt::CaseInsensitive)) &&
        fileAccessBlocked(networkRequest.url().toString())) {
        QNetworkReply* reply = new WrtNetworkErrorReply(QNetworkReply::ContentAccessDenied, "Not allowed", networkRequest.url());
        QMetaObject::invokeMethod(reply, "finished", Qt::QueuedConnection);
        return reply;
    }

    // handle file scheme too for now
    if (!networkRequest.url().scheme().compare("file", Qt::CaseInsensitive)){
        // always assume get request
        WrtWidgetReply* reply = new WrtWidgetReply(networkRequest,
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
                                                   d->m_storage
#else
                                                   0
#endif
                                                   ,"file",getWidgetRoot());
        return reply;
    }

    if (!networkRequest.url().scheme().compare("widget", Qt::CaseInsensitive)){
        enableCrossOriginXHR(networkRequest.url());
        // always assume get request
        WrtWidgetReply* reply = new WrtWidgetReply(networkRequest,
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
                                                   d->m_storage
#else
                                                   0
#endif
        ,"widget", getWidgetRoot());
        return reply;
    }
    if (!networkRequest.url().scheme().compare("sandbox", Qt::CaseInsensitive)){
        // always assume get request
        WrtWidgetReply* reply = new WrtWidgetReply(networkRequest,
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
                                                   d->m_storage,
#else
                                                   0,
#endif
                                                   "sandbox");
        return reply;
    }

    if (!networkRequest.url().scheme().compare("lib", Qt::CaseInsensitive)
#ifdef Q_OS_SYMBIAN //only nokia signed widgets have the access to shared library.
            && (networkRequest.url().toString().contains("lib://Nokia",Qt::CaseInsensitive) || d->m_appPath.contains( QDir::fromNativeSeparators(DEFAULT_ROOT_DIRECTORY), Qt::CaseInsensitive) )
#endif
            ){
        enableCrossOriginXHR(networkRequest.url());
        // always assume get request
        WrtWidgetReply* reply = new WrtWidgetReply(request,
#ifdef CWRT_WIDGET_FILES_IN_SECURE_STORAGE
                                                   d->m_storage,
#else
                                                   0,
#endif
                                                   "lib", d->m_appPath);
        return reply;
    }

    // Requests with "data" scheme can/must be served right away without queueing the request (see code below) even if no
    // network connection is available. Also no access policy checks are required as the request is not pointing to a resource
    // (that it might not be allowed to access, but instead contains the actual encoded data.
    if (!networkRequest.url().scheme().compare("data", Qt::CaseInsensitive))
        return QNetworkAccessManager::createRequest(op, request);

    // access policy check
    if (!accessAllowed(networkRequest)) {
        QNetworkReply* reply = new WrtNetworkErrorReply(QNetworkReply::ContentAccessDenied, "Not allowed", networkRequest.url());
        //a finished signal will be sent and the member is invoked as soon as the application enters the main event loop
        QMetaObject::invokeMethod(reply, "finished", Qt::QueuedConnection);
        return reply;
    }

    if (d->m_schemeHandler->HandleSpecialScheme(networkRequest.url())) {
        // handled in scheme handler - block network access
        QNetworkReply* reply = new WrtNetworkErrorReply(QNetworkReply::OperationCanceledError, "Scheme Handled", networkRequest.url());
        QMetaObject::invokeMethod(reply, "finished", Qt::QueuedConnection);
        return reply;
    }

    //custom network request preprocessing, if error return.
    if (preprocessNetworkRequest(op, networkRequest)) {
        QNetworkReply* reply = new WrtNetworkErrorReply(QNetworkReply::OperationCanceledError, "Pre-process failure", networkRequest.url());
        //a finished signal will be sent and the member is invoked as soon as the application enters the main event loop
        QMetaObject::invokeMethod(reply, "finished", Qt::QueuedConnection);
        return reply;
    }

    // Enable HTTP pipelining  on outbound request
    networkRequest.setAttribute(QNetworkRequest::HttpPipeliningAllowedAttribute, true);

    // To improve content loading speed, enable unexpired HTTP cache items to be returned immediately if present
    // Prevents communication with server to check validity
    networkRequest.setAttribute(QNetworkRequest::CacheLoadControlAttribute, QNetworkRequest::PreferCache);

    QNetworkReply* reply;
    if (!m_connected && (!networkRequest.url().scheme().compare("http", Qt::CaseInsensitive) ||
                         !networkRequest.url().scheme().compare("https", Qt::CaseInsensitive))){
        //if waiting for connection, create a dummy networkreply for now, and bridge the real reply later
        reply = new WrtNetworkReply(op, networkRequest);
        connect(reply, SIGNAL(aborted(WrtNetworkReply*)), this, SLOT(abortPendingRequest(WrtNetworkReply*)));
        //buffer the requests, so it can be send after connection is established
        PendingRequest req = {op, networkRequest, outgoingData, reply};
        m_pendingRequests.append(req);
    }
    else { //create real reply if connection is already established
        reply = QNetworkAccessManager::createRequest(op, networkRequest, outgoingData);
    }

    return reply;
}

/*
 * hanndle Connection established signal
 * once connection is established, create real network replys from requests buffered in the request queue
 * setup the dummy reply created earlier to bridge the real network reply
 */
void WrtNetworkAccessManager::handleConnection()
{
    m_connected = true;

    while (!m_pendingRequests.isEmpty()){
        const PendingRequest &req = m_pendingRequests.takeFirst();
        //create real network reply
        QNetworkReply* reply = QNetworkAccessManager::createRequest(req.op, req.request, req.outgoingdata);
        //set up dummy reply to bridge the real one
        ((WrtNetworkReply*)req.reply)->setReply(reply);
    }
}

/*
 * network disconnected
 */
void WrtNetworkAccessManager::connectionError()
{
    m_connected = false;
}

void WrtNetworkAccessManager::abortPendingRequest(WrtNetworkReply* reply)
{
    for (int i=0; i < m_pendingRequests.count(); i++){
        if (m_pendingRequests[i].reply == reply) {
            m_pendingRequests.removeAt(i);
            i--;
        }
    }
}

/*!
 * Setup cache
 * Need to use WrtSettingsUI to setup Disk Cache Directory Path
 */
void WrtNetworkAccessManager::setupCache()
{

    if ( WrtSettings::createWrtSettings()->value ( "HTTPCacheEnabled" ).toBool() ) {

        QString path(WrtSettings::createWrtSettings()->value ( "HTTPCacheDirectory" ).toString());
        QDir diskCacheDir(path);

        //If fully-qualified path wasn't specified already via Settings, do platform-specific search & replace.
        if ( !diskCacheDir.isAbsolute() ) {
            diskCacheDir.setPath(fixupHTTPcachePath(path));
        }

        Q_ASSERT(diskCacheDir.isAbsolute());

        // NOTE: Each QNetworkAccessManager/WRTNetworkAccessManager object MUST get its own HTTP cache object ..
        // .. and dedicated cache directory! These are not shareable!

        delete d->m_qDiskCache;
#if defined(USE_PRIVATEHTTPCACHE)
        d->m_qDiskCache = new FeatherWeightCache(this);
#else
        d->m_qDiskCache = new QNetworkDiskCache(this);
#endif
        d->m_qDiskCache->setCacheDirectory(diskCacheDir.absolutePath());
        d->m_qDiskCache->setMaximumCacheSize(DISKCACHESIZE);

        setCache(d->m_qDiskCache);
    }

}

/*!
 *
 */
int WrtNetworkAccessManager::preprocessNetworkRequest(const QNetworkAccessManager::Operation &op, QNetworkRequest &request)
{
    int error = 0;
    QList<WRTNetworkFilterListener*>::iterator it;
    for (it = d->m_wrtNetworkFilterListener.begin(); it != d->m_wrtNetworkFilterListener.end(); it++) {
        error = (*it)->customNetworkRequestPreprocess(op, request);
        if (error)
          return error;
    }

    return error;
}

void WrtNetworkAccessManager::setBasePath(const QString& basePath)
{
    QString widgetPath = basePath;
    widgetPath.replace("\\","/");
    if (widgetPath.endsWith("/"))
        widgetPath.chop(1);
    int l = widgetPath.lastIndexOf("/");
    d->m_appPath = widgetPath.left(l + 1).toLower();// always at least one, +1 to include '/'
    bool ok;
    d->m_widgetId = widgetPath.right(widgetPath.length() - l - 1).toInt(&ok);
}
/* prevent access to other widget's data
 ie. file:///C:/private/200267c0/widgets_21D_4C7/157925442/flip.bmp
 widget path = C:/private/200267c0/widgets_21D_4C7/157925442/
 app path = C:/private/200267c0/widgets_21D_4C7/
 because the file is under C:/private/200267c0/widgets_21D_4C7 it must be under 157925442 or a common dir
 ie.
 C:/data - ok
 C:/private/200267c0/widgets_21D_4C7/ - ok
 C:/private/200267c0/widgets_21D_4C7/157925442/ - ok
 C:/private/200267c0/widgets_21D_4C7/lib/ - ok
 C:/private/200267c0/widgets_21D_4C7/123456789/ - BLOCKED
*/
bool WrtNetworkAccessManager::fileAccessBlocked(const QString& urlPath)
{
    bool blocked(false);
    QString url = urlPath.toLower();

    int l = (d->m_appPath.isEmpty()) ? -1 : url.indexOf(d->m_appPath);
    if (l > -1) {
        bool appOk;
        int appIdStart = d->m_appPath.length() + l;
        int appIdEnd = url.indexOf("/",appIdStart);
        int appId = url.mid(appIdStart, appIdEnd - appIdStart).toInt(&appOk);
        if (appOk && appId != d->m_widgetId)
            blocked = true;
    }

    return blocked;
}


bool WrtNetworkAccessManager::accessAllowed(const QNetworkRequest& request)
{
// Allowing everything for Maemo5 while the bug #3690 is not closed.
// http://bugs.nokia-boston.com/bugzilla/show_bug.cgi?id=3690
#ifdef Q_OS_MAEMO5
   return true;
#endif

#ifdef Q_OS_MAEMO6
    //Maemo6 has support for 'plain' widgets that have no properties
    //and now other rights at the moment. Let them access everything
    if (!d->m_widgetContainerBase->widgetProperties()){
        return true;
    }
#endif

    QString widgetId(d->m_widgetContainerBase->widgetProperties()->id());
    bool jilWidget = false;

    WebAppInfo webAppInfo;
    if (!WebAppRegistry::instance()->isRegistered(widgetId, webAppInfo))
        return false;

    // FIXME: if not "JIL" widget, just allow all requests for now!
    if (webAppInfo.widgetType() != WebAppInfo::WidgetJIL)
        return true;

    QList<W3CElement*> accessElems = webAppInfo.getElement("widget/access");

    if (accessElems.empty())
        return false;

    // JIL 1.2 cases
    if (webAppInfo.widgetType() == WebAppInfo::WidgetJIL) {
        for (int index=0; index < accessElems.size(); ++index) {
            QString networkValue = accessElems.at(index)->attributeValue("network");
            if (!networkValue.isEmpty()) {
                jilWidget = true;
                if (networkValue.compare("true", Qt::CaseInsensitive) != 0) {
                   //in this case no access is  allowed.
                   return false;
                }
                break;
            }
        }

        // no jil:access tags found or network attr not found. Do not allow access.
        if (!jilWidget)
            return false;

        // if JIL, and no w3c access tag present, just allow all URL access.
        if (accessElems.size() == 1)
            return true;
    }

    QString origin;
    QString subDomain;

    for (int index=0; index < accessElems.size(); ++index) {
        origin = accessElems.at(index)->attributeValue("origin");
        subDomain = accessElems.at(index)->attributeValue("subdomains");

        QUrl url(origin);

        //if a origin has value as "*", then allow all urls.
        if (origin == "*")
            return true;

        //VALIDATE SCHEME.
        //FIXME ... what are the schemes that should be supported?
        if (url.scheme()!= request.url().scheme())
            continue;

        bool isHttpProtocol;
        if (!request.url().scheme().compare("http", Qt::CaseInsensitive) ||
            !request.url().scheme().compare("https", Qt::CaseInsensitive))
            isHttpProtocol = true;

        //VALIDATE HOST. As per w3c spec, if the scheme is http or https , a case-insensitive
        //comparison of host has to be done.
        if (subDomain == "false") {
           if (request.url().host().compare(url.host(), (isHttpProtocol)?(Qt::CaseInsensitive):(Qt::CaseSensitive)) != 0)
              continue;
        }
        else {
           if (!(request.url().host().endsWith("." + url.host(), (isHttpProtocol)?(Qt::CaseInsensitive):(Qt::CaseSensitive)) ||
                   request.url().host().compare(url.host(), (isHttpProtocol)?(Qt::CaseInsensitive):(Qt::CaseSensitive)) == 0))
              continue;
        }

        //VALIDATE PATH. There is no details present for path in specification
        //comparison of path is done if access origin has one.
        if (!request.url().path().contains(url.path(), Qt::CaseInsensitive))
            continue;

        //VALIDATE PORT. check for the following cases!
        //case 1: no port in access element, port mentioned in requested url
        //case 2: port mentioned in access element, no port in requested url
        //case 3: both the urls have port mentioned in it.
        //case 4: neither of the urls have port mentioned in it
        int port = request.url().port();

        //case 1
        if ((url.port() == -1) && (request.url().port() != -1)) {
            if (!request.url().scheme().compare("http", Qt::CaseInsensitive)) {
                if (port != 80)
                    continue;
            }
            else if (!request.url().scheme().compare("https", Qt::CaseInsensitive)) {
                if (port != 443)
                    continue;
            }
        }
        //case 2
        else if ((url.port() != -1) && (request.url().port() == -1)) {
             if (!request.url().scheme().compare("http", Qt::CaseInsensitive)) {
                if (url.port() != 80)
                    continue;
             }
             else if (!request.url().scheme().compare("https", Qt::CaseInsensitive)) {
                if (url.port() != 443)
                    continue;
             }
        }
        //case 3
        else if ((url.port() != -1) && (request.url().port() != -1)) {
                if (url.port() != request.url().port())
                    continue;
        }

        //the flow reahces this point when all the conditions are valid.
        return true;
    }
    return false;
}
QString WrtNetworkAccessManager::getWidgetRoot(){
    if (!d->m_widgetContainerBase)
       return "";
    if (!d->m_widgetContainerBase->widgetProperties())
       return "";

    return d->m_widgetContainerBase->widgetProperties()->installPath();
}

/* Apply platform-specific rules to covert HTTP cache location into a fully
 * qualified directory path.
 */
QString WrtNetworkAccessManager::fixupHTTPcachePath(const QString& templateFormPath)
{
    QString fullPath(templateFormPath);

    QString baseDir (QDesktopServices::storageLocation(QDesktopServices::CacheLocation));

#if defined(Q_OS_SYMBIAN)
    // On Symbian, use C:\private\UID3\ of the application running this code.
    // Allows OS-enforced security + automatic cleanup when application is uninstalled
    baseDir = QCoreApplication::applicationDirPath();
#elif defined(Q_OS_MAEMO5) ||  defined(Q_OS_MAEMO6)
    //Maemo (oops, Meego) friends should free to tweak this!
    baseDir = QDesktopServices::storageLocation(QDesktopServices::CacheLocation);
#endif

    //perform tag replacement
    fullPath.replace(QLatin1String("<toplevel>"), baseDir,  Qt::CaseInsensitive);

    return fullPath;
}

void WrtNetworkAccessManager::enableCrossOriginXHR(QUrl requestUrl)
{
#if !defined(Q_OS_MAEMO5) && !defined(Q_OS_MAEMO6) && !defined(CWRT_BUILDING_VASCO) || defined(CWRT_BUILDING_QTUNSTABLE)
    QString origin = (requestUrl.scheme() + "://" + requestUrl.host()).toLower();
    // TODO should retrieve the actual white list from webkit when the API
    // becomes available.  See header file for more details
    if (!(whiteList.contains(origin)))
    {
        whiteList << origin;
        addWhiteListEntriesForOrigin(origin);
    }
#endif
}

/*! Adds entries necessary to allow cross origin XHRs from origin to the webkit maintained whiteList.
 *  Right now the access is pretty open, but in the future should be restricted based on what the origin is.
 *  This should be done here if it is easy and not to many dependencies are introduced, in a derived class,
 *  or in an unrelated class employing the signal-slot mechanisms of Qt or a callback interface.
 *  If the derived class mechanism is not used this method should become private non-virtual maybe even inlined in
 *  enableCrossOriginXHR.
 */
void WrtNetworkAccessManager::addWhiteListEntriesForOrigin(QString origin)
{
#if !defined(Q_OS_MAEMO5) && !defined(Q_OS_MAEMO6) && !defined(CWRT_BUILDING_VASCO) || defined(CWRT_BUILDING_QTUNSTABLE)
    QString domain;
    qt_drt_whiteListAccessFromOrigin(origin, "http", domain, true);
    qt_drt_whiteListAccessFromOrigin(origin, "https", domain, true);
    qt_drt_whiteListAccessFromOrigin(origin, "lib", domain, true);
#endif
}

}

