/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef DIALOGSPROVIDER_H
#define DIALOGSPROVIDER_H
#include <QtGui/QSpinBox>
#include <QtGui/QValidator>
#include <QtGui/QKeyEvent>
#include <QtGui/QDialog>
#include <QtCore/QString>
#include <QtGui/QLineEdit>
#include "wrtdialog.h"

QT_BEGIN_NAMESPACE
class QLabel;
class QDialogButtonBox;
class QAbstractButton;
QT_END_NAMESPACE

#if defined(BUILDING_DIALOGS_PROVIDER)
#      define DIALOGS_PROV_EXPORT Q_DECL_EXPORT
#else
#      define DIALOGS_PROV_EXPORT Q_DECL_IMPORT
#endif
// This internal class adds extra validation to a QSpinBox by emitting textChanged(bool)
// after events that may potentially change the visible text. Return or Enter key presses
// are not propagated if the visible text is invalid. Instead, the visible text is modified
// to the last valid value.
class DialogsProviderValidatedSpinBox : public QSpinBox
{
    Q_OBJECT
public:
    DialogsProviderValidatedSpinBox(int minValue, int maxValue, int step, int value)
        : QSpinBox(0)
    {
        setRange(minValue, maxValue);
        setSingleStep(step);
        setValue(value);
        selectAll();
        validator = new QIntValidator(minValue, maxValue, this);
        QObject::connect(
            lineEdit(), SIGNAL(textChanged(const QString &)), this, SLOT(notifyTextChanged()));
        QObject::connect(this, SIGNAL(editingFinished()), this, SLOT(notifyTextChanged()));
    }

private:
    QIntValidator *validator;
    void keyPressEvent(QKeyEvent *event)
    {
        if ((event->key() == Qt::Key_Return || event->key() == Qt::Key_Enter) && !textValid()) {
#ifndef QT_NO_PROPERTIES
            setProperty("value", property("value"));
#endif
        } else {
            QSpinBox::keyPressEvent(event);
        }
        notifyTextChanged();
    }

    void mousePressEvent(QMouseEvent *event)
    {
        QSpinBox::mousePressEvent(event);
        notifyTextChanged();
    }

    bool textValid() const
    {
        QString t = text();
        int pos = 0;
        return validate(t, pos) == QValidator::Acceptable;
    }

private slots:
    void notifyTextChanged() { emit textChanged(textValid()); }

signals:
    void textChanged(bool);
};


// This internal class adds extra validation to a QDoubleSpinBox by emitting textChanged(bool)
// after events that may potentially change the visible text. Return or Enter key presses
// are not propagated if the visible text is invalid. Instead, the visible text is modified
// to the last valid value.
class DialogsProviderValidatedDoubleSpinBox : public QDoubleSpinBox
{
    Q_OBJECT
public:
    DialogsProviderValidatedDoubleSpinBox(
        double minValue, double maxValue, int decimals, double value)
        : QDoubleSpinBox(0)
    {
        setDecimals(decimals);
        setRange(minValue, maxValue);
        setValue(value);
        selectAll();
        validator = new QDoubleValidator(minValue, maxValue, decimals, this);
        QObject::connect(
            lineEdit(), SIGNAL(textChanged(const QString &)), this, SLOT(notifyTextChanged()));
        QObject::connect(this, SIGNAL(editingFinished()), this, SLOT(notifyTextChanged()));
    }
private:
    QDoubleValidator *validator;
    void keyPressEvent(QKeyEvent *event)
    {
        if ((event->key() == Qt::Key_Return || event->key() == Qt::Key_Enter) && !textValid()) {
#ifndef QT_NO_PROPERTIES
            setProperty("value", property("value"));
#endif
        } else {
            QDoubleSpinBox::keyPressEvent(event);
        }
        notifyTextChanged();
    }

    void mousePressEvent(QMouseEvent *event)
    {
        QDoubleSpinBox::mousePressEvent(event);
        notifyTextChanged();
    }

    bool textValid() const
    {
        QString t = text();
        int pos = 0;
        return validate(t, pos) == QValidator::Acceptable;
    }

private slots:
    void notifyTextChanged() { emit textChanged(textValid()); }

signals:
    void textChanged(bool);
};


class DialogsProvider : public WrtDialog
{
    Q_OBJECT

private:
    // ### Qt 5: remove
    enum Type { LineEdit, SpinBox, DoubleSpinBox, ComboBox, EditableComboBox };

    // ### Qt 5: remove
    DialogsProvider(const QString &label, QWidget* parent, Type type, Qt::WindowFlags f);

    DialogsProvider(
        const QString &title, const QString &label, QWidget *parent, QWidget *input,
        Qt::WindowFlags f);

    DialogsProvider(const QString &label, QWidget* parent, const QStringList &buttons, int *buttonIndex);
    DialogsProvider(const QString &label, QWidget* parent);
    DialogsProvider(QWidget* parent, QWidget* inputWidget);

    ~DialogsProvider();

public slots:
    void clicked(QAbstractButton *button);

public:
    DIALOGS_PROV_EXPORT static QString getText(QWidget *parent, const QString &title, const QString &label,
                           QLineEdit::EchoMode echo = QLineEdit::Normal,
                           const QString &text = QString(), bool *ok = 0, Qt::WindowFlags f = 0);
    DIALOGS_PROV_EXPORT static int getInteger(QWidget *parent, const QString &title, const QString &label, int value = 0,
                          int minValue = -2147483647, int maxValue = 2147483647,
                          int step = 1, bool *ok = 0, Qt::WindowFlags f = 0);
    DIALOGS_PROV_EXPORT static double getDouble(QWidget *parent, const QString &title, const QString &label, double value = 0,
                            double minValue = -2147483647, double maxValue = 2147483647,
                            int decimals = 1, bool *ok = 0, Qt::WindowFlags f = 0);
    DIALOGS_PROV_EXPORT static QString getItem(QWidget *parent, const QString &title, const QString &label, const QStringList &list,
                           int current = 0, bool editable = true, bool *ok = 0,Qt::WindowFlags f = 0);
    DIALOGS_PROV_EXPORT static int getButtonResponse(QWidget *parent, const QString &label, const QStringList &buttons);
    DIALOGS_PROV_EXPORT  static bool getUsernamePassword(QWidget *parent, QString &username, QString &password);

    DIALOGS_PROV_EXPORT static void showInfoDialog(QWidget *parent, const QString &label);
    DIALOGS_PROV_EXPORT static void showTimedDialog(QWidget *parent, const QString &label);
    DIALOGS_PROV_EXPORT static void showTimedMessage(QWidget *parent, const QString &label, int msec);


private:

    QLabel *label;
    QPushButton *okButton;
    QWidget *input;
    int *m_buttonIndex;
    QDialogButtonBox *m_buttonBox;

    void init(const QString &label, DialogsProvider::Type);
    void init(const QString &title, const QString &label, QWidget *input);
    void init(const QString &lbl, const QStringList &buttons);
    void init(const QString &lbl);
    void init(QWidget*);

    void tryAccept();

private:
    Q_DISABLE_COPY(DialogsProvider)
};

#endif // DIALOGSPROVIDER_H
