/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef CERTIFICATEMANAGER_H
#define CERTIFICATEMANAGER_H
#include <QByteArray>
#include <QString>
#include <QObject>
#include <QMap>
namespace WRT {

const QString KEY_USAGE("keyusage");
const QString EXT_KEY_USAGE("extendedkeyusage");
const QString CERT_POLICY("certificatepolicies");

class CertificateManager : public QObject {

public:

    CertificateManager();
    virtual ~CertificateManager();
    /**
     * @param certs     all certificates included as part of the signed widget, typically as <X509Certificate> elements in <X509Data>
     * @param endEntity the certificate used to sign the widget (the last cert on the chain)
     * @param topLevel  the certificate whose AKI points to the trusted certificate on the device
     */
    virtual bool validateCertificates(QList<QByteArray> certs, QByteArray endEntity, QByteArray topLevel);

public:

    /**
     * @param topLevel  the certificate whose AKI points to the trusted certificate on the device
     * @param cert      the trusted certificate from the device is returned in this paramter
     */
    virtual bool retrieveDeviceCertificate(QByteArray topLevel, QByteArray& cert) = 0;

    /**
     * @param endEntity the certificate used to sign the widget
     * @param topLevel  the top certificate in the chain (if any) whose AKI points to the trusted certificate on the device
     */
    virtual bool buildCertMapping(QList<QByteArray> certs, QByteArray& endEntityCert, QByteArray& topLevel);

    virtual QString getAuthorityKeyIdentifier(QByteArray certificate);
    virtual QString getSubjectKeyIdentifier(QByteArray certificate);
    virtual bool checkCertificateValidityDates(QByteArray certificate);
    virtual bool checkExtensionValues(QByteArray certificate, QMap<QString,QString>& expectedExtValues);
    virtual bool createDir(const QString& path);

private:

};
} // WRT namespace
#endif // CERTIFICATEMANAGER_H
