/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef _WRT_SETTINGS_H_
#define _WRT_SETTINGS_H_

#include <QtCore/QSettings>
#include <QtCore/QStringList>
#include "runtimeglobal.h"
// responsible for setting WRTSETTINGS_EXPORT correctly
#include "wrtsettings_global.h"


//FIXME: Needs a new UID3 for Symbian!
#define WRT_SETTINGS_UID "10008D39"


#define WRTSETTINGS_VERSION_FORMAT "%1.%2.%3"
#define WRTSETTINGS_WRT_VERSION_KEY "WRTVersion"
#define WRTSETTINGS_SETTINGS_VERSION_KEY "SettingsVersion"
#define WRTSETTINGS_BROWSER_VERSION_KEY "BrowserVersion"
#define WRTSETTINGS_WIDGETS_VERSION_KEY "WidgetsVersion"

#define WRTSETTING_NOT_FOUND -1
#define WRTSETTING_ACCESS_DENIED -2
#define WRTSETTING_BAD_ARGUMENT -3

class QAbstractItemModel;
class SettingsModel;
class QStandardItem;
class QStandardItemModel;

namespace WRT {

    class WrtPage;

class WRTSETTINGS_EXPORT WrtSetting
{

friend class WrtSettings;

public:

    enum WrtSettingType{
        NodeType
       ,StringType
       ,CommandType
       ,ListType
       ,IntType
       ,DblType
    };
    enum WrtSettingCategory{
       RootCategory = -1
       ,Category1
       ,Category2
       ,Category3
       ,Category4
    };
    enum WrtSettingFlag{
        NoFlags = 0
       ,ReadOnly = 1
       ,WebAttribute = 2
       ,Hidden = 4
       //, = 8
       //, = 16
       //, = 32
       //, = 64
    // Componant level settings for filtering setting changes in controller
       ,DatabaseSetting = 128
       ,BookmarksSetting = 256
    };


    WrtSetting(const QString &key);

    QVariant value();
    QString displayValue();
    QString key() const { return m_key; }
    QString desc() const { return m_desc; }
    QStringList valid() const  { return m_validList; }
    QStringList values() const { return m_valuesList; }
    int flags() const { return m_flags; }
    int type() const { return m_type; }
    int category() const { return m_category; }
    int attributeEnum() const { return m_enum; }
    bool isList() { return m_type == WrtSetting::ListType; }
    bool isAction() { return m_type == WrtSetting::CommandType; }
    bool hasRange() { return (m_type == WrtSetting::IntType && valid().size()==2 && values().isEmpty()); }
    int minValue() { return (hasRange() ? valid().at(0).toInt() : 0 ); }
    int maxValue() { return (hasRange() ? valid().at(1).toInt() : 0 ); }
    QString group() const { return m_group; }

private:

    WrtSetting();

protected:

    void setValid(const QString& valid);
    void setValues(const QString& values);
    void setFlags(int flags) { m_flags = flags; }

    QString m_key;
    QString m_desc;
    QString m_group;
    int m_type;
    QVariant m_value;
    QStringList m_validList;
    QStringList m_valuesList;
    int m_flags;
    int m_enum;
    int m_category;

};

class WRTSETTINGS_EXPORT WrtSettings : public QSettings
{
    Q_OBJECT

 signals:
    void settingChanged(const QString &key);
    void settingActivated(const int);

public:
    static WrtSettings* createWrtSettings();

    QList<WrtSetting> settings(const QString &group = QString());
    WrtSetting setting(const QString &key);

    QString valueAsString(const QString &key, const QVariant &defaultValue = QVariant());
    int valueAsInt(const QString &key, const QVariant &defaultValue = QVariant());
    double valueAsDouble(const QString &key, const QVariant &defaultValue = QVariant());
    QVariant value(const QString &key, const QVariant &defaultValue = QVariant());
    int setValue(const QString &key, const int value);
    int setValue(const QString &key, const double value);
    int setValue(const QString &key, const QString &value);
    int setValue(const QString &key, const QVariant &value);
    int activate(const QString &key);

    void resync();

    static QString getWrtVersion();

private:

    WrtSettings( QObject *parent = 0, QString uid=WRT_SETTINGS_UID  );
    void init(bool clear = false);

private slots:
    void doAction(const int action);
    void settingTrigger(const QString &key);

private:
    static WrtSettings* g_wrtSettings;
    QString m_appuid;
};

}

#endif //_WRT_SETTINGS_H_
