/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QMessage>
#include <QMessageAddress>
#include <QMessageAccount>
#include <QMessageManager>
#include <QMessageAccountId>

#include <QtCore>
#include "sendmessage.h"
#include "qmessagingconstants.h"

QTM_USE_NAMESPACE

QSendMessage::QSendMessage():
    iTransactionId(-1),
    m_service(NULL)
{
    qDebug() << "QSendMessage::QSendMessage(): connecting to doSendSignal";
    connect(this, SIGNAL(doSendSignal()), this, SLOT(doSend()), Qt::QueuedConnection);
    m_service = new QMessageService(this);
}

QSendMessage::~QSendMessage()
{
    qDebug() << "QSendMessage::~QSendMessage()";
    disconnect(this, SIGNAL(doSendSignal()), this, SLOT(doSend()));
}

void QSendMessage::sendMessageAsync(MessageData &messageData, int aTransactionId)
{
    qDebug() << "QSendMessage::sendMessageAsync()";
    iTransactionId = aTransactionId;
    iMessageData = messageData;
    iRecipientsList = iMessageData.recipients();
    qDebug() << "QSendMessage::sendMessageAsync(): iRecipientsList:" << iRecipientsList ;
    if (!iRecipientsList.isEmpty())
    {
        if (m_service)
        {
            qDebug() << "QSendMessage::sendMessageAsync(): connecting to QtMobility stateChanged signal";
            connect(m_service, SIGNAL(stateChanged(QMessageService::State)), this, SLOT(sendFinished(QMessageService::State)), Qt::DirectConnection);
            qDebug() << "QSendMessage::sendMessageAsync(): emiting send signal";
            emit doSendSignal();
        }
        else
        {
            qDebug() << "QSendMessage::sendMessageAsync(): Sending message failed!";
            sendErrorCallback();
        }
    }
}

void QSendMessage::doSend()
{
    qDebug() << "QSendMessage::doSend(): call for sendMessage()";
    sendMessage();
}

void QSendMessage::sendMessage()
{
    qDebug() << "QSendMessage::sendMessage()";
    QMessage message;
    bool isSentSuccess = FALSE;
    message.setType(QMessage::Sms);

    if (!iSmsAccId.isValid())
    {
        QMessageManager manager;
        QMessageAccountIdList accList = manager.queryAccounts();
        qDebug() << "QSendMessage::sendMessage(): manager.countAccounts = " << manager.countAccounts();
        int accountNumber = 0;
        if(!accList.isEmpty())
        {
            qDebug() << "QSendMessage::sendMessage(): Account count " << accList.count();
            foreach(QMessageAccountId accId, accList)
            {
                qDebug() << ++accountNumber << "QSendMessage::sendMessage(): Account name " << manager.account(accId).name();
                qDebug() << accountNumber << "QSendMessage::sendMessage(): Account type " << (QMessage::TypeFlags)manager.account(accId).messageTypes();
                if (manager.account(accId).messageTypes() == QMessage::Sms)
                {
                    iSmsAccId = accId;
                    break;
                }
            }
        }
        else
        {
            qDebug() << "QSendMessage::sendMessage(): Failed to get MessageAccountId!";
            sendErrorCallback();
            return;
        }
        if (!iSmsAccId.isValid())
        {
            qDebug() << "QSendMessage::sendMessage(): Failed to get SMS Account Id!";
            sendErrorCallback();
            return;
        }
    }
    message.setParentAccountId(iSmsAccId);
    QMessageAddressList toList;
    qDebug() << "QSendMessage::sendMessage(): recipients count:" << iRecipientsList.count();
    qDebug() << "QSendMessage::sendMessage(): recipients list:" << iRecipientsList;
    for (int i = 0; i < iRecipientsList.count(); i++)
    {
        qDebug() << "QSendMessage::sendMessage(): appending toList with current recipient:" << iRecipientsList.at(i);
        toList.append(QMessageAddress(QMessageAddress::Phone, iRecipientsList.at(i)));
    }
    message.setTo(toList);
    message.setBody(iMessageData.body());

    qDebug() << "QSendMessage::sendMessage(): call for QMessageService::send";
    isSentSuccess = m_service->send(message);
    qDebug() << "QSendMessage::sendMessage(): result:" << isSentSuccess;
    if (!isSentSuccess)
    {
        qDebug() << "QSendMessage::sendMessage(): failed to send!";
        qDebug() << "QSendMessage::sendMessage(): Sending message failed!";
        sendErrorCallback();
    }
    else
    {
        QStringList recipients = iRecipientsList;
        qDebug() << "QSendMessage::sendMessage(): recipients count:" << iRecipientsList.count();
        for (int i = 0; i < recipients.count(); i++)
        {
            iRecipientsList.removeAt(i);
            qDebug() << "QSendMessage::sendMessage(): emiting completed signal for recipient:" << recipients.at(i);
            //currently QMessageService::send doesn't work correctly (bug#169200), don't know if there will be one
            //state notofication or several. Verifying only first, to avoid segmentation falt.
            if (iServiceState.first() == QMessageService::FinishedState)
            {
                emit completed(this, SUCCESS, iTransactionId, recipients.at(i));
            }
            else
            {
                emit completed(this, SERVICE_BUSY_ERR, iTransactionId, recipients.at(i));
            }
        }
        iRecipientsList.clear();
        iServiceState.clear();
    }
}

void QSendMessage::sendFinished(QMessageService::State error)
{
    qDebug() << "QSendMessage::sendFinished(): received state:" << error;
    if ((error == QMessageService::FinishedState) || error == QMessageService::CanceledState)
    {
        qDebug() << "QSendMessage::sendFinished(): message processed, appending error";
        iServiceState.append(error);
    }
}

void QSendMessage::destroyIfFinished()
{
    qDebug() << "QSendMessage::destroyIfFinished()";
    if (iRecipientsList.isEmpty())
    {
        qDebug() << "QSendMessage::destroyIfFinished(): disconnecting from stateChanged signal";
        disconnect(m_service, SIGNAL(stateChanged(QMessageService::State)), this, SLOT(sendFinished(QMessageService::State)));
        //currently QMessageService::send doesn't work correctly (bug#169200), commented to avoid segmentation falt.
/*        if (m_service)
        {
            qDebug() << "QSendMessage::destroyIfFinished(): deleting m_service";
            delete m_service;
        }
        qDebug() << "QSendMessage::destroyIfFinished(): deleting this";
        delete this;*/
    }
}

void QSendMessage::sendErrorCallback()
{
    qDebug() << "QSendMessage::sendErrorCallback(): iRecipientsList";
    QStringList recipients = iRecipientsList;
    for (int i = 0; i < recipients.count(); i++)
    {
        emit completed(this, SERVICE_BUSY_ERR, iTransactionId, recipients.at(i));
        iRecipientsList.removeAt(i);
    }
    iRecipientsList.clear();
    iServiceState.clear();
}
