/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QMap>
#include <QDebug>
#include <QObject>
#include <QtCore>

#include "qmessagingservice.h"
#include "qmessagingiterator.h"
#include "getmsglisttask.h"
#include "messagedata.h"
#include "messagefilter.h"

#include <hildon-uri.h>

//QtMobility includes
#include <QMessage>
#include <QMessageAccount>
#include <QMessageAccountId>
#include <QMessageAddress>
#include <QMessageService>
#include <QMessageFilter>

const QString KSMS_PREFIX("sms://");
const QString KSMS_TO_SEPARATOR(",");

/**
 * constructor
 * 1) Creates Core class object
 * 2) Initializes transaction ID
 * 3) Creates Transaction Map
 * @since x.y
 * @param
 * @return
 */
QMessagingService::QMessagingService()
{
    qDebug(__FUNCTION__);

    iNotifierSet = false;

    QMessageFilter smsFilter(QMessageFilter::byType(QMessage::Sms, QMessageDataComparator::Equal));
    QMessageFilter inboxFilter(QMessageFilter::byStandardFolder(QMessage::InboxFolder, QMessageDataComparator::Equal));
    m_manager.registerNotificationFilter(smsFilter & inboxFilter);
    connect(&m_manager, SIGNAL(messageAdded(const QMessageId&, const QMessageManager::NotificationFilterIdSet&)),
            this, SLOT(newMessageCallback(const QMessageId&, const QMessageManager::NotificationFilterIdSet&)));

    connect(this, SIGNAL(asyncComplete(int, int)), this, SLOT (notifyResult(int,int)), Qt::QueuedConnection);
    connect(this, SIGNAL(asyncGetListComplete(int, int)), this, SLOT (notifyGetListError(int,int)), Qt::QueuedConnection);

}

/**
 * destructor
 * 1) Delete core class instance
 * @since x.y
 * @param
 * @return
 */
QMessagingService::~QMessagingService()
{
    qDebug(__FUNCTION__);
}

/**
 * QMessagingService::setSecuritySession()
 * Implemented from IServiceSecurity.Needed to handle capabilities.
 *
 * @param WRT::SecSession
 * @return
 */
void QMessagingService::setSecuritySession(WRT::SecSession *aSecSession)
{
    qDebug(__FUNCTION__);
    iSecSession = aSecSession;
}

void QMessagingService::notifyResult(const int aErrorCode, int aTransactionId)
{
    qDebug() << __FUNCTION__ << aErrorCode << aTransactionId;
    if (iGoingSendTransactions.contains(aTransactionId))
    {
        iGoingSendTransactions.removeAt(iGoingSendTransactions.indexOf(aTransactionId));
        qDebug() << "Notifying wrt framework - send";
        emit asyncCallback(aErrorCode, aTransactionId, "");
    }
}

void QMessagingService::notifyGetListError(const int aErrorCode, int aTransactionId)
{
    qDebug() << __FUNCTION__ << aErrorCode << aTransactionId;
    if (iGoingAsyncTransactions.contains(aTransactionId))
    {
        iGoingAsyncTransactions.removeAt(iGoingAsyncTransactions.indexOf(aTransactionId));
        QObject* messIterObj = NULL;
        QVariantList list;
        QMessagingIterator* messIter = new QMessagingIterator(list, aTransactionId);
        messIter->setParent(this);
        messIterObj = static_cast<QObject*>(messIter);

        qDebug() << "Notifying wrt framework";
        emit asyncCallback(aErrorCode, aTransactionId, messIterObj);
    }
}

void QMessagingService::getListCallback(GetMsgListTask* aTask, qint32 aTransactionId, int aErrorCode, QVariantList aMessages)
{
    qDebug() << __FUNCTION__ << "Id=" << aTransactionId << ", Error=" << aErrorCode << "MsgCount=" << aMessages.count();

    if (iGoingAsyncTransactions.contains(aTransactionId))
    {
        iGoingAsyncTransactions.removeAt(iGoingAsyncTransactions.indexOf(aTransactionId));
        QObject* messIterObj = NULL;
        QMessagingIterator* messIter = new QMessagingIterator(aMessages, aTransactionId);
        messIter->setParent(this);
        messIterObj = static_cast<QObject*>(messIter);
        qDebug() << "Notifying wrt framework";
        emit asyncCallback(aErrorCode, aTransactionId, messIterObj);
    }

    aTask->finishTask();
}

void QMessagingService::newMessageCallback(const QMessageId &eventId, const QMessageManager::NotificationFilterIdSet &filterSet)
{
    Q_UNUSED(filterSet)
    qDebug() << __FUNCTION__ << "Notifier set" << iNotifierSet << ", TransactionId=" << iNotifierTransactionId;
    if (iNotifierSet)
    {
        qDebug() << "New SMS - Notifying wrt framework" << eventId.toString();
        //Notify with transaction id = 0 - it is expected by platformservices test cases
        emit asyncCallback(SUCCESS, 0, eventId.toString().toInt());
    }
}

QVariant QMessagingService::startEditor(const QMap<QString, QVariant>& aMessageData)
{
    QVariantMap retVal;
    MessageData msgData;
    int errCode = SUCCESS;
    QString errMsg = "";
    qDebug() << "QMessagingService::startEditor(): started";
    if (!msgData.fillMessageData(aMessageData, errCode, errMsg, true))
    {
        qDebug() << "QMessagingService::startEditor(): error occured:" << errCode << "," << errMsg;
        retVal[KCErrorCode] = errCode;
        retVal[KCErrorMessage] = errMsg;
        return retVal;
    }

    QString type = msgData.type();
    qDebug() << "QMessagingService::startEditor(): type:" << type;
    if (!type.compare(KCSMS, Qt::CaseInsensitive))
    {
        QString uriString = KSMS_PREFIX;
        QStringList recipientsList = msgData.recipients();
        qDebug() << "QMessagingService::startEditor(): recipientsList.count():" << recipientsList.count();
        for (int i = 0; i < recipientsList.count(); i++)
        {\
            uriString += recipientsList.at(i) + KSMS_TO_SEPARATOR;
            qDebug() << "QMessagingService::startEditor(): uriString:" << uriString;
        }
        QString body = msgData.body();
        //hildon_uri_open do not open composer if string is empty
        if (body.isEmpty())
        {
            body = " ";
        }
        uriString += "?body=" + body;
        qDebug() << "QMessagingService::startEditor(): uriString:" << uriString;

        bool result = hildon_uri_open(uriString.toStdString().c_str(), NULL, NULL);
        if (!result)
        {
            qDebug() << "QMessagingService::startEditor(): error open uri!";
            errCode = SERVICE_BUSY_ERR;
            errMsg = "Call for hildon_uri_open failed";
        }
    }
    else if (!type.compare(KCMMS, Qt::CaseInsensitive))
    {
        qDebug() << "QMessagingService::startEditor(): MMS not supported!";
        errCode = NOT_SUPPORTED_ERR;
        errMsg = "MMS not supported";
    }
    else if (!type.compare(KCEMAIL, Qt::CaseInsensitive))
    {
        qDebug() << "QMessagingService::startEditor(): KCEMAIL support will be added under spec 2.2 implementation";
        errCode = NOT_SUPPORTED_ERR;
        errMsg = "EMAIL not supported yet";
//Email support will be added under spec 2.2 implementation.
/*
        QMessage message;
        message.setType(QMessage::Email);
        QStringList recipientsList = msgData.recipients();
        if (recipientsList.count())
        {
            QMessageAddressList toList;
            for (int i = 0; i < recipientsList.count(); i++)
            {
                toList.append(QMessageAddress(QMessageAddress::Email, recipientsList.at(i)));
            }
            message.setTo(toList);
            qDebug() << "QMessagingService::startEditor(): recipientsList:" << recipientsList;
        }
        QStringList ccRecipientsList = msgData.cc();
        if (ccRecipientsList.count())
        {
            QMessageAddressList ccList;
            for (int i = 0; i < ccRecipientsList.count(); i++)
            {
                ccList.append(QMessageAddress(QMessageAddress::Email, ccRecipientsList.at(i)));
            }
            message.setCc(ccList);
            qDebug() << "QMessagingService::startEditor(): ccRecipientsList:" << ccRecipientsList;
        }
        qDebug() << "QMessagingService::startEditor(): body:" << msgData.body();
        if (!msgData.body().isEmpty() && !msgData.body().isNull())
        {
            message.setBody(msgData.body());
        }
        if (!msgData.subject().isEmpty() && !msgData.subject().isNull())
        {
            message.setSubject(msgData.subject());
        }
        switch (msgData.priority())
        {
            case 0:
                message.setPriority(QMessage::HighPriority);
                break;
            case 1:
                message.setPriority(QMessage::NormalPriority);
                break;
            case 2:
                message.setPriority(QMessage::LowPriority);
                break;
            default:
                message.setPriority(QMessage::NormalPriority);
        }
        qDebug() << "QMessagingService::startEditor(): priority:" << msgData.priority();

        QStringList bccRecipientsList = msgData.bcc();
        if (bccRecipientsList.count())
        {
            QMessageAddressList bccList;
            for (int i = 0; i < bccRecipientsList.count(); i++)
            {
                bccList.append(QMessageAddress(QMessageAddress::Email, bccRecipientsList.at(i)));
            }
            message.setCc(bccList);
            qDebug() << "QMessagingService::startEditor(): bccRecipientsList:" << bccRecipientsList;
        }
        QStringList attachments = msgData.attachments();
        if (attachments.count())
        {
            message.appendAttachments(attachments);
            qDebug() << "QMessagingService::startEditor(): attachments:" << attachments;
        }
        qDebug() << "QMessagingService::startEditor(): creating message service..";
        QMessageService service;
        qDebug() << "QMessagingService::startEditor(): opening editor";
        if (!service.compose(message))
        {
            qDebug() << "QMessagingService::startEditor(): error sending message via QMessageService!";
            errCode = SERVICE_BUSY_ERR;
            errMsg = "Call for QMessageService::send failed";
        }
*/
    }
    qDebug() << "QMessagingService::startEditor(): errCode:" << errCode;
    qDebug() << "QMessagingService::startEditor(): errMsg:" << errMsg;
    retVal[KCErrorCode] = errCode;
    retVal[KCErrorMessage] = errMsg;
    return retVal;
}

QVariant QMessagingService::getList(const QMap<QString, QVariant>& aMatch, const int aSortKey,
                                    const int aSortOrder, const int aTransactionId)
{
    qDebug() << __FUNCTION__ << "Sort key:" << aSortKey << "Sort order:" << aSortOrder;
    QVariantMap retVal;
    MessageFilter filter;
    int errCode = SUCCESS;
    QString errMsg = "";
    filter.fillFilterData(aMatch, aSortKey, aSortOrder, errCode, errMsg);
    qDebug() << "Parsing resuls:" << errCode << filter.filterFlags();
    qDebug() << "Filter types [" << filter.filterTypes() << "], senders [" << filter.filterSenders() << "] subject [" << filter.filterSubject() << "]";
    qDebug() << "Filter Start date [" << filter.filterStartTime() << "]";
    qDebug() << "Filter End date [" << filter.filterEndTime() << "]";

    qDebug() << "Running task";

    if (errCode == SUCCESS)
    {
        if (filter.allowSMS() || filter.allowMMS())
        {
            // Run getList in a thread
            GetMsgListTask * task = new GetMsgListTask(aTransactionId, filter, m_manager);
            connect(task, SIGNAL(signalGetMsgListCb(GetMsgListTask*, qint32, int, QVariantList)), this,
                    SLOT(getListCallback(GetMsgListTask*, qint32, int, QVariantList))/*, Qt::QueuedConnection*/);

            iGoingAsyncTransactions.append(aTransactionId);
            qDebug() << "MessagingService::getList -> real starting";
            QThreadPool::globalInstance()->start(task);
        }
    }
    else if (errCode == DATA_OUT_OF_RANGE_ERR)
    {
        //special case - according to specification, we need to invoke callback asynchronously
        iGoingAsyncTransactions.append(aTransactionId);
        retVal[KCErrorCode] = SUCCESS;
        retVal[KCErrorMessage] = "";
        retVal[KCTransactionId] = aTransactionId;
        emit asyncGetListComplete(errCode, aTransactionId); // and complete request with error asynchronously
        return retVal;
    }

    retVal[KCErrorCode] = errCode;
    retVal[KCErrorMessage] = errMsg;
    retVal[KCTransactionId] = aTransactionId;

    qDebug() << "Returning from getList";
    return retVal;
}

QObject* QMessagingService::getListSync(const QMap<QString, QVariant>& aMatch,
                                        const int aSortKey,
                                        const int aSortOrder)
{
    qDebug() << __FUNCTION__ << "Sort key:" << aSortKey << "Sort order:" << aSortOrder;
    MessageFilter filter;
    int errCode = SUCCESS;
    QString errMsg = "";
    filter.fillFilterData(aMatch, aSortKey, aSortOrder, errCode, errMsg);
    qDebug() << "Parsing resuls:" << errCode << filter.filterFlags();
    qDebug() << "Filter types [" << filter.filterTypes() << "], senders [" << filter.filterSenders() << "] subject [" << filter.filterSubject() << "]";
    qDebug() << "Filter Start date [" << filter.filterStartTime() << "]";
    qDebug() << "Filter End date [" << filter.filterEndTime() << "]";


    QObject* messIterObj = NULL;
    if (errCode == SUCCESS)
    {
        // Run getList synchronously
        if (filter.allowSMS() || filter.allowMMS())
        {
            QVariantList results;
            GetMsgListTask * task = new GetMsgListTask(0, filter, m_manager);
            errCode = task->getMessagesNow(results);
            if (errCode == SUCCESS)
            {
                qDebug() << "Get message list success";
                QMessagingIterator* messIter = new QMessagingIterator(results, 0);
                messIter->setParent(this);
                messIterObj = static_cast<QObject*>(messIter);
            }
        }
    }

    qDebug() << "Returning from getList Error=" << errCode;
    return messIterObj;
}


QVariant QMessagingService::getEmailAccounts()
{
    qDebug() << Q_FUNC_INFO;
    QVariantMap ret;
    QVariantMap errMap;
    QVariantList resultList;
    int errCode = SUCCESS;
    QString errMsg = "";

    //Not yet implemented

    errMap.insert(KCErrorCode, errCode);
    errMap.insert(KCErrorMessage, errMsg);
    ret.insert(KCReturnValue, resultList);
    ret.insert(KCErrorMap, errMap);
    qDebug() << Q_FUNC_INFO << ret;
    return ret;
}

QVariant QMessagingService::getCurrentEmailAccount()
{
    qDebug() << Q_FUNC_INFO;
    QVariantMap item;
    QVariantMap ret;
    QVariantMap errMap;
    int errCode = SUCCESS;
    QString errMsg = "";
    QMessageAccountId accId = QMessageAccount::defaultAccount(QMessage::Email);
    if (accId.isValid())
    {
        QMessageAccount acc = m_manager.account(accId);
        item.insert(KCAccountId, accId.toString());
        item.insert(KCAccountName, acc.name());
    }
    qDebug() << Q_FUNC_INFO << item;
    errMap.insert(KCErrorCode, errCode);
    errMap.insert(KCErrorMessage, errMsg);
    ret.insert(KCReturnValue, item);
    ret.insert(KCErrorMap, errMap);
    return ret;
}

QVariant QMessagingService::setCurrentEmailAccount(QString aAccountId)
{
    qDebug() << Q_FUNC_INFO << aAccountId;
    QVariantMap retVal;
    int errorCode = SUCCESS;
    QString errMsg = "";

    //Not supported

    retVal.insert(KCErrorCode, errorCode);
    retVal.insert(KCErrorMessage, errMsg);
    qDebug() << Q_FUNC_INFO << retVal;
    return retVal;
}


QVariant QMessagingService::send(const QMap<QString, QVariant>& aMessageData, const QString& aId,
                                 const int aTransactionId)
{
    qDebug() << "QMessagingService::send(): aTransactionId:" << aTransactionId;

    QVariantMap retVal;
    MessageData msgData;
    int errCode = SUCCESS;
    int errSMS = SUCCESS;
    QString errMsg = "";
    qDebug() << "QMessagingService::send: filling message data";
    msgData.fillMessageData(aMessageData, errCode, errMsg, false);

    if (errCode == MISSING_ARG_ERR ||
        errCode == INVALID_ARG_ERR)
    {
        qDebug() << "QMessagingService::send: fillMessageData failed with err:" << errCode;
        retVal[KCErrorCode] = errCode;
        retVal[KCErrorMessage] == errMsg;
        retVal[KCTransactionId] = aTransactionId;
        qDebug() << "QMessagingService::send: returning retVal:" << retVal;
        return retVal;
    }
    else if (errCode != SUCCESS)
    {   //this type of error should be returned asynchronously
        qDebug() << "QMessagingService::send: fillMessageData failed with err:" << errCode;
        retVal[KCErrorCode] = SUCCESS;
        retVal[KCTransactionId] = aTransactionId;
        iGoingSendTransactions.append(aTransactionId);
        qDebug() << "QMessagingService::send: emiting errCode:" << errCode;
        emit asyncComplete(errCode, aTransactionId);
        qDebug() << "QMessagingService::send: returnint retVal:" << retVal;
        return retVal;
    }

    //check if it is needed to forward message
    if (!aId.isEmpty())
    {
        qDebug() << "QMessagingService::send: need to forward message, passed id:" << aId;
        qDebug() << "QMessagingService::send: searching for message with passed id";
        QMessage message = m_manager.message(QMessageId::QMessageId(aId));
        QVariantMap data = MessageData::fillMessageMap(message);
        QMap<QString, QVariant> msg = data.value(KCMessage).toMap();
        qDebug() << "QMessagingService::send: msg:" << msg;
        if (msg.contains(KCType))
        {
            qDebug() << "QMessagingService::send: message founded, type:" << msg.value(KCType);
            qDebug() << "QMessagingService::send: message to send, type:" << msgData.type();
            if (msgData.type() == KCSMS)
            {
                if (msg.value(KCType) != KCSMS)
                {
                    qDebug() << "QMessagingService::send: Invalid to pass a MMS message id to an SMS message! return DATA_OUT_OF_RANGE_ERR";
                    errCode = DATA_OUT_OF_RANGE_ERR;
                    errMsg = KCBadArgument;
                }
                else
                {
                    QString bodyText = msg.value(KCBody).toString();
                    qDebug() << "QMessagingService::send: message founded, body:" << bodyText;
                    if (!bodyText.isEmpty())
                    {
                        msgData.setBody(bodyText + msgData.body());
                    }
                    qDebug() << "QMessagingService::send: new message body:" << msgData.body();
                }
            }
            if (msgData.type() == KCMMS)
            {
                if ((msg.value(KCType) != KCSMS) && (msg.value(KCType) != KCMMS))
                {
                    qDebug() << "QMessagingService::send: Invalid to pass an EMAIL message id to an MMS message! return DATA_OUT_OF_RANGE_ERR";
                    errCode = DATA_OUT_OF_RANGE_ERR;
                    errMsg = KCBadArgument;
                }
                else
                {
                    qDebug() << "QMessagingService::send: MMS not supported, ignoring";
                }
            }
            if (msgData.type() == KCEMAIL)
            {
                if (msg.value(KCType) != KCEMAIL)
                {
                    qDebug() << "QMessagingService::send: Invalid to pass SMS or MMS message id to an EMAIL message! return DATA_OUT_OF_RANGE_ERR";
                    errCode = DATA_OUT_OF_RANGE_ERR;
                    errMsg = KCBadArgument;
                }
                else
                {
                    qDebug() << "QMessagingService::send: EMAIL is not implemented yet, ignoring";
                }
            }
        }
        else
        {
            qDebug() << "QMessagingService::send: no message founded, msg:" << msg;
            errCode = DATA_NOT_FOUND_ERR;
            errMsg = KCDataNotFound;
        }

        if (errCode != SUCCESS)
        {
            retVal[KCErrorCode] = SUCCESS;
            retVal[KCTransactionId] = aTransactionId;
            iGoingSendTransactions.append(aTransactionId);
            qDebug() << "QMessagingService::send: emiting errCode:" << errCode;
            emit asyncComplete(errCode, aTransactionId);
            qDebug() << "QMessagingService::send: returnint retVal:" << retVal;
            return retVal;
        }
    }

    iGoingSendTransactions.append(aTransactionId);

    qDebug() << "QMessagingService::send: type:" << msgData.type();
    if (msgData.type() == KCSMS)
    {
        qDebug() << "QMessagingService::send: creating QSendMessage";
        QSendMessage* msg = new QSendMessage ();
        qDebug() << "QMessagingService::send: connecting to completed signal";
        connect(msg, SIGNAL(completed(QSendMessage*, int, int, QString)), this, SLOT(sendSMSMessageFinished(QSendMessage*, int, int, QString)));
        qDebug() << "QMessagingService::send: starting async task";
        msg->sendMessageAsync(msgData, aTransactionId);
        qDebug() << "QMessagingService::send: task started";
    }
    else if (msgData.type() == KCMMS)
    {
        qDebug() << "QMessagingService::send: MMS not supported";
        emit asyncComplete(SUCCESS, aTransactionId);
    }
    else if (msgData.type() == KCEMAIL)
    {
        //TODO: support for email will be added under spec 2.2 implementation
        qDebug() << "QMessagingService::send: EMAIL currently not supported";
        emit asyncComplete(SUCCESS, aTransactionId);
    }

    retVal[KCErrorCode] = SUCCESS;
    retVal[KCErrorMessage] = errMsg;
    retVal[KCTransactionId] = aTransactionId;
    qDebug() << "QMessagingService::send: returning retVal:" << retVal;
    qDebug() << __FUNCTION__ << " end" << errCode << errMsg << aTransactionId;
    return retVal;
}

void QMessagingService::sendSMSMessageFinished(QSendMessage* message, int aErrorCode, int aTransactionId, QString recipient)
{
    qDebug() << "QMessagingService::sendSMSMessageFinished(): message:" << message;
    qDebug() << "QMessagingService::sendSMSMessageFinished(): aErrorCode:" << aErrorCode;
    qDebug() << "QMessagingService::sendSMSMessageFinished(): aTransactionId:" << aTransactionId;
    qDebug() << "QMessagingService::sendSMSMessageFinished(): recipientList:" << recipient;
    notifySMSResult(aErrorCode, aTransactionId, recipient, message->moreRecipients());
    message->destroyIfFinished();
}

void QMessagingService::notifySMSResult(const int aErrorCode, int aTransactionId, QString aRecipient, bool aMoreRecipients)
{
    qDebug() << "QMessagingService::notifySMSResult(): aTransactionId" << aTransactionId;
    qDebug() << "QMessagingService::notifySMSResult(): aRecipient" << aRecipient;
    qDebug() << "QMessagingService::notifySMSResult(): aMoreRecipients" << aMoreRecipients;
    if (iGoingSendTransactions.contains(aTransactionId))
    {
        qDebug() << "QMessagingService::notifySMSResult(): iGoingSendTransactions" << iGoingSendTransactions;
        if (!aMoreRecipients)
        {
            iGoingSendTransactions.removeAt(iGoingSendTransactions.indexOf(aTransactionId));
            qDebug() << "QMessagingService::notifySMSResult(): updated iGoingSendTransactions" << iGoingSendTransactions;
        }
        qDebug() << "QMessagingService::notifySMSResult(): Notifying wrt framework - send";
        emit asyncCallback(aErrorCode, aTransactionId, aRecipient);
    }
}

QVariant QMessagingService::setNotifier(const int aTransactionId)
{
    qDebug(__FUNCTION__);
    QVariantMap retVal;
    iNotifierSet = true;
    iNotifierTransactionId = aTransactionId;

    retVal[KCErrorCode] = SUCCESS;
    retVal[KCErrorMessage] = "";
    retVal[KCTransactionId] = aTransactionId;
    return retVal;
}

QVariant QMessagingService::cancelNotifier()
{
    QVariantMap retVal;
    qDebug(__FUNCTION__);
    iNotifierSet = false;
    iNotifierTransactionId = 0;
    retVal[KCErrorCode] = SUCCESS;
    return retVal;
}

QVariant QMessagingService::getMessage(const QString& aMessageId, const QString aFolderName)
{
    qDebug() << __FUNCTION__ << aMessageId << aFolderName;
    QVariantMap retVal;
    retVal[KCErrorCode] = SUCCESS;
    retVal[KCErrorMessage] = "";

    if (aMessageId.isEmpty())
    {
        retVal[KCErrorCode] = MISSING_ARG_ERR;
        retVal[KCErrorMessage] = KCMissingArg;
    }
    else
    {
        QMessageId mesId(aMessageId);
        if (mesId.isValid())
        {
            QMessageFilter filter(QMessageFilter::byId(mesId, QMessageDataComparator::Equal));
            QMessageIdList mesIdList = m_manager.queryMessages(filter);
            if (!mesIdList.isEmpty())
            {
                QMessage msg = m_manager.message(mesId);
                if (msg.type() == QMessage::Sms)
                {
                    QVariantMap msgData = MessageData::fillMessageMap(msg);
                    QString msgFolder = "";
                    if (msgData.contains(KCFolder))
                    {
                        QVariant folderVal = msgData.value(KCFolder);
                        if (folderVal.type() == QVariant::String)
                        {
                            msgFolder = folderVal.toString();
                        }
                    }

                    if (aFolderName.isEmpty() || msgFolder.compare(aFolderName, Qt::CaseInsensitive) == 0)
                    {
                        retVal[KCReturnValue] = msgData;
                    }
                    else
                    {
                        retVal[KCErrorCode] = DATA_NOT_FOUND_ERR;
                        retVal[KCErrorMessage] = KCDataNotFound;
                    }
                }
                else
                {
                    retVal[KCErrorCode] = DATA_NOT_FOUND_ERR;
                    retVal[KCErrorMessage] = KCDataNotFound;
                }
            }
            else
            {
                retVal[KCErrorCode] = DATA_NOT_FOUND_ERR;
                retVal[KCErrorMessage] = KCDataNotFound;
            }
        }
        else
        {
            retVal[KCErrorCode] = INVALID_ARG_ERR;
            retVal[KCErrorMessage] = KCDataNotFound;
        }
    }

    qDebug() << __FUNCTION__ << "end. " << retVal;
    return retVal;
}

QVariant QMessagingService::deleteMessage(const QMap<QString, QVariant>& aMessageData)
{
    QVariantMap retVal;
    int errCode = SUCCESS;
    QString errMessage = "";
    retVal[KCErrorCode] = SUCCESS;
    retVal[KCErrorMessage] = "";
    MessageFilter filter;
    qDebug() << __FUNCTION__ << aMessageData;
    bool res = filter.fillPartialData(aMessageData,
                           MessageFilter::EFilterId | MessageFilter::EFilterType | MessageFilter::EFilterFolderName,
                           MessageFilter::EFilterId,
                           errCode,
                           errMessage);
    if (!res)
    {
        retVal[KCErrorCode] = errCode;
        retVal[KCErrorMessage] = errMessage;
        return retVal;
    }

    QString msgIdStr = filter.filterId();

    qDebug() << msgIdStr;
    if (msgIdStr.isEmpty())
    {
        retVal[KCErrorCode] = MISSING_ARG_ERR;
        retVal[KCErrorMessage] = KCMissingArg;
    }
    else
    {
        QMessageId mesId(msgIdStr);
        if (mesId.isValid())
        {
            QMessageFilter msgFilter(QMessageFilter::byId(mesId, QMessageDataComparator::Equal));
            QMessageIdList mesIdList = m_manager.queryMessages(msgFilter);
            if (!mesIdList.isEmpty())
            {
                QMessage msg = m_manager.message(mesId);
                qDebug() << "event is found, type = " << msg.type();
                //check correct type
                if ((msg.type() == QMessage::Sms && !filter.filterTypes().contains(KCSMS, Qt::CaseInsensitive)) ||
                    (msg.type() == QMessage::Mms && !filter.filterTypes().contains(KCMMS, Qt::CaseInsensitive)) ||
                    (msg.type() == QMessage::Email && !filter.filterTypes().contains(KCEMAIL, Qt::CaseInsensitive)))
                {
                    qDebug() << "Message ID" << mesId.toString() << " doesn't belong to types" << filter.filterTypes();
                    retVal[KCErrorCode] = DATA_NOT_FOUND_ERR;
                    retVal[KCErrorMessage] = KCDataNotFound;
                    return retVal;
                }

                QVariantMap item = MessageData::fillMessageMap(msg);
                if (filter.checkFilter(item))
                {
                    qDebug() << "Deleting...";
                    if (!m_manager.removeMessage(mesId, QMessageManager::RemoveLocalCopyOnly))
                    {
                        qDebug() << "Error on deleting...";
                        retVal[KCErrorCode] = SERVICE_BUSY_ERR;
                        retVal[KCErrorMessage] = KCDataNotFound;
                    }
                }
                else
                {
                    qDebug() << "not matching filter";
                    retVal[KCErrorCode] = DATA_NOT_FOUND_ERR;
                    retVal[KCErrorMessage] = KCDataNotFound;
                }
            }
            else
            {
                qDebug() << "delete - Event not found";
                retVal[KCErrorCode] = DATA_NOT_FOUND_ERR;
                retVal[KCErrorMessage] = KCDataNotFound;
            }
        }
        else
        {
            retVal[KCErrorCode] = INVALID_ARG_ERR;
            retVal[KCErrorMessage] = KCDataNotFound;
        }
    }

    qDebug() << __FUNCTION__ << "end. Error code:" << retVal[KCErrorCode] << retVal[KCErrorMessage];
    return retVal;
}

QVariant QMessagingService::setStatus(const QString& aId, const int aStatus)
{
    QVariantMap retVal;
    int errCode = SUCCESS;
    QString errMessage = "";
    qDebug() << __FUNCTION__ << aId << aStatus;
    retVal[KCErrorCode] = SUCCESS;
    retVal[KCErrorMessage] = "";

    if (aId.isEmpty())
    {
        retVal[KCErrorCode] = MISSING_ARG_ERR;
        retVal[KCErrorMessage] = KCMissingArg;
    }
    else
    {
        QMessageId mesId(aId);
        if (mesId.isValid())
        {
            QMessageFilter msgFilter(QMessageFilter::byId(mesId, QMessageDataComparator::Equal));
            QMessageIdList mesIdList = m_manager.queryMessages(msgFilter);
            if (!mesIdList.isEmpty())
            {
                if (aStatus == MESSAGE_READ || aStatus == MESSAGE_UNREAD)
                {
                    QMessage msg = m_manager.message(mesId);
                    msg.setStatus(QMessage::Read, aStatus == MESSAGE_READ);
                    if (!m_manager.updateMessage(&msg))
                    {
                        retVal[KCErrorCode] = SERVICE_BUSY_ERR;
                        retVal[KCErrorMessage] = KCDataNotFound;
                    }
                }
                else
                {
                    retVal[KCErrorCode] = DATA_OUT_OF_RANGE_ERR;
                    retVal[KCErrorMessage] = KCBadArgument;
                }
            }
            else
            {
                retVal[KCErrorCode] = DATA_NOT_FOUND_ERR;
                retVal[KCErrorMessage] = KCDataNotFound;
            }
        }
        else
        {
            retVal[KCErrorCode] = INVALID_ARG_ERR;
            retVal[KCErrorMessage] = KCDataNotFound;
        }
    }


    qDebug() << __FUNCTION__ << "end. Error " << retVal[KCErrorCode] << retVal[KCErrorMessage];
    return retVal;
}

QVariant QMessagingService::cancel(const int aTransactionId)
{
    QVariantMap retVal;
    qDebug() << "QMessagingService::cancel. Id=" << aTransactionId;
    retVal[KCTransactionId] = aTransactionId;
    retVal[KCErrorCode] = SUCCESS;
    retVal[KCErrorMessage] = "";

    if (iGoingAsyncTransactions.contains(aTransactionId))
    {
        qDebug() << "MsgProvider: removing transaction success";
        // Remove request from the array
        iGoingAsyncTransactions.removeAt(iGoingAsyncTransactions.indexOf(aTransactionId));
    }
    else if (iGoingSendTransactions.contains(aTransactionId))
    {
        iGoingSendTransactions.removeAt(iGoingSendTransactions.indexOf(aTransactionId));

        //There is no API to really stop sending SMS messages
    }
    else
    {
        qDebug() << "MsgProvider: not found: transaction doesn't exist";
        retVal[KCErrorCode] = DATA_NOT_FOUND_ERR;
        retVal[KCErrorMessage] = KCDataNotFound;
    }
    qDebug() << __FUNCTION__ << "end. Error code:" << retVal[KCErrorCode] << retVal[KCErrorMessage];
    return retVal;

}
