/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#ifndef __MESSAGE_FILTER_H__
#define __MESSAGE_FILTER_H__

#include <QObject>
#include <QVariant>
#include <QMap>
#include <QString>
#include <QStringList>
#include <QDateTime>

class MessageFilter
{
public:
    enum FilterFlags
    {
        EFilterNone =       0x0000,
        EFilterType =       0x0001,
        EFilterSenders =    0x0002,
        EFilterSubject =    0x0004,
        EFilterStartDate =  0x0008,
        EFilterEndDate =    0x0010,
        EFilterId =         0x0020,
        EFilterFolderName = 0x0040,
        EFilterPriority =   0x0080,
        EFilterAttachment = 0x0100,
        EFilterTo =         0x0200,
        EFilterCc =         0x0400,
        EFilterBody =       0x0800,
        EFilterUnread =     0x1000,
        EFilterValidity =   0x2000,
        EAll              = 0x7fffffff
    };

    struct FilterAttachmentData
    {
        QString iFileName;
    };

    MessageFilter():
            iFilterFlags(EFilterNone),
            iNeededFilterFlags(EAll),
            iMandatoryFlags(0),
            iSortKey(0),
            iSortMethod(0),
            iFilterPriority(0),
            iFilterValidity(0),
            iMultiTypes(true)
    {}
    MessageFilter (const MessageFilter& a):
            iFilterFlags(a.iFilterFlags),
            iNeededFilterFlags(a.iNeededFilterFlags),
            iMandatoryFlags(a.iMandatoryFlags),
            iFilterTypes(a.iFilterTypes),
            iFilterSenders(a.iFilterSenders),
            iFilterTo(a.iFilterTo),
            iFilterCc(a.iFilterCc),
            iFilterSubject(a.iFilterSubject),
            iFilterBody(a.iFilterBody),
            iFilterUnread(a.iFilterUnread),
            iFilterId(a.iFilterId),
            iSortKey(a.iSortKey),
            iSortMethod(a.iSortMethod),
            iFilterStartTime(a.iFilterStartTime),
            iFilterEndTime(a.iFilterEndTime),
            iFilterFolderName(a.iFilterFolderName),
            iFilterPriority(a.iFilterPriority),
            iFilterValidity(a.iFilterValidity),
            iFilterAttachments(a.iFilterAttachments),
            iMultiTypes(a.iMultiTypes)
    {}

    bool fillFilterData(const QVariantMap& aMatch, const int aSortKey,
            const int aSortOrder, int& aErrorCode, QString& aErrorMessage);

    bool fillPartialData(const QVariantMap& aMatch, int aNeededFilterFlags, int aMandatoryFlags, int& aErrorCode, QString& aErrorMessage);

    bool checkFilter(const QVariantMap& aEvent) const;


    bool allowInbox() const;
    bool allowOutbox() const;
    bool allowSent() const;
    bool allowDrafts() const;

    bool allowSMS() const;
    bool allowMMS() const;
    bool allowEmail() const;


public:
    int filterFlags() const {return iFilterFlags;}
    QStringList filterTypes() const {return iFilterTypes;}
    QStringList filterSenders() const {return iFilterSenders;}
    QStringList filterTo() const {return iFilterTo;}
    QStringList filterCc() const {return iFilterCc;}
    QString filterSubject() const {return iFilterSubject;}
    QString filterBody() const {return iFilterBody;}
    bool filterUnread() const {return iFilterUnread;}
    QString filterId() const {return iFilterId;}
    int sortKey() const {return iSortKey;}
    int sortMethod() const {return iSortMethod;}
    QDateTime filterStartTime() const {return iFilterStartTime;}
    QDateTime filterEndTime() const {return iFilterEndTime;}
    QString filterFolderName() const {return iFilterFolderName;}

    int filterPriority() const {return iFilterPriority;}
    int filterValidity() const {return iFilterValidity;}

    QList<FilterAttachmentData> filterAttachments() const {return iFilterAttachments;}

private:

    bool doFillFilterData(const QVariantMap& aMatch, int& aErrorCode, QString& aErrorMessage);

    //used when iMultiTypes is true
    bool parseFilterTypes(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    //used when iMultiTypes is false
    bool parseFilterType(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterId(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterSenders(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterTo(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterCc(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterSubject(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterBody(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterUnread(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterFolderName(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterPriority(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterValidity(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterAttachments(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterStartDate(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool parseFilterEndDate(const QMap<QString, QVariant>& aMatch, int& aErrorCode, QString& aErrorMessage);

    bool checkMandatoryParamExists(const QMap<QString, QVariant>& aMatch, const QString& aParam, const FilterFlags& aFlag, int& aErrorCode, QString& aErrorMessage) const;

    bool addFilterType(const QVariant& aData, int& aErrorCode, QString& aErrorMessage);

    bool parseStringList(const QVariantMap& aMatch,
                         const QString& aParam,
                         const FilterFlags& aFlag,
                         QStringList& aResult,
                         int& aErrorCode,
                         QString& aErrorMessage);

    bool parseInt(const QVariantMap& aMatch,
                     const QString& aParam,
                     const FilterFlags& aFlag,
                     int& aResult,
                     int& aErrorCode,
                     QString& aErrorMessage);

    bool parseString(const QVariantMap& aMatch,
                     const QString& aParam,
                     const FilterFlags& aFlag,
                     QString& aResult,
                     int& aErrorCode,
                     QString& aErrorMessage);

    bool parseBool(const QVariantMap& aMatch,
                     const QString& aParam,
                     const FilterFlags& aFlag,
                     bool& aResult,
                     int& aErrorCode,
                     QString& aErrorMessage);

    bool parseDateTime(const QVariantMap& aMatch,
                     const QString& aParam,
                     const FilterFlags& aFlag,
                     QDateTime& aResult,
                     int& aErrorCode,
                     QString& aErrorMessage);

    bool checkStringList(const QVariantMap& aEvent, const QString& aParam, const FilterFlags& aFlag, const QStringList& aFilter) const;
    bool checkStringListMulti(const QVariantMap& aEvent, const QString& aParam, const FilterFlags& aFlag, const QStringList& aFilter) const;

    bool checkStringWildCard(const QVariantMap& aEvent, const QString& aParam, const FilterFlags& aFlag, const QString& aFilter) const;

private:
    int iFilterFlags;
    int iNeededFilterFlags;
    int iMandatoryFlags;

    QStringList iFilterTypes; //list of message types to filter (SMS, MMS). MMS is not supported but it will not produce any errors, just return empty list
    QStringList iFilterSenders;
    QStringList iFilterTo;
    QStringList iFilterCc;

    QString iFilterSubject;
    QString iFilterBody;

    bool iFilterUnread;

    QString iFilterId;

    int iSortKey;
    int iSortMethod; //SORT_ASCENDING or SORT_DESCENDING

    QDateTime iFilterStartTime;
    QDateTime iFilterEndTime;

    QString iFilterFolderName;

    int iFilterPriority;
    int iFilterValidity;

    QList<FilterAttachmentData> iFilterAttachments;

    bool iMultiTypes;

};

#endif
