/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#ifndef __MESSAGE_DATA_H__
#define __MESSAGE_DATA_H__

#include <QObject>
#include <QVariant>
#include <QMap>
#include <QHash>
#include <QString>
#include <QStringList>
#include <QDateTime>
//#include <CommHistory/Event>

//QtMobility includes
#include <QMessage>
QTM_USE_NAMESPACE

class MessageData
{
public:
    MessageData();

    static QVariantMap fillMessageMap(QMessage& aEvent);

    bool fillMessageData (const QVariantMap& aMap, int& aErrorCode, QString& aErrorMessage, bool aStartEditor);

    QString type() const {return iType;}
    QString body() const {return iBody;}
    void setBody(const QString& aBody) {iBody = aBody;}
    QString subject() const {return iSubject;}
    QStringList recipients() const {return iRecipients;}
    QStringList cc() const {return iCcRecipients;}
    QStringList bcc() const {return iBccRecipients;}
    QStringList attachments() const {return iAttachments;}
    int validity() const {return iValidityMins;}
    int priority() const {return iPriority;}

    bool isStartEditor() const {return iStartEditor;}
    bool isSend() const {return !iStartEditor;}

private:
    /**
     * MessageData::getMessageType()
     * Fills the Type parameter - SMS and MMS are the allowed types
     * @param aMessageData           message data
     * @param aErrorCode             On return contains error code
     * @param aErrorMessage          On return contains error message
     * @return                       'true' if operation was successful, 'false' otherwise
     */
    bool getMessageType(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage);

    /**
     * MessageData::getMessageToList()
     * Fills the recipients of message
     * @param aMessageData           message data
     * @param aErrorCode             On return contains error code
     * @param aErrorMessage          On return contains error message
     * @return                       'true' if operation was successful, 'false' otherwise
     */
    bool getMessageToList(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage);

    /**
     * MessageData::getMessageCcList()
     * Fills the CC or BCC recipients of MMS message
     * @param aMessageData           message data
     * @param aKey                   key representing recipoients type (KCCc or KCBcc)
     * @param aRecipients            recipients list to fill
     * @param aErrorCode             On return contains error code
     * @param aErrorMessage          On return contains error message
     * @return                       'true' if operation was successful, 'false' otherwise
     */
    bool getMessageCcBccList(const QVariantMap& aMessageData, const QString& aKey, QStringList& aRecipients, int& aErrorCode, QString& aErrorMessage);

    /**
     * MessageData::getMessageBody()
     * Fills the body of message
     * @param aMessageData           message data
     * @param aErrorCode             On return contains error code
     * @param aErrorMessage          On return contains error message
     * @return                       'true' if operation was successful, 'false' otherwise
     */
    bool getMessageBody(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage);

    /**
     * MessageData::getMessageSubject()
     * Fills the subject of MMS message
     * @param aMessageData           message data
     * @param aErrorCode             On return contains error code
     * @param aErrorMessage          On return contains error message
     * @return                       'true' if operation was successful, 'false' otherwise
     */
    bool getMessageSubject(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage);

    /**
     * MessageData::getMessageAttachments()
     * Fills the attachments of MMS message
     * @param aMessageData           message data
     * @param aErrorCode             On return contains error code
     * @param aErrorMessage          On return contains error message
     * @return                       'true' if operation was successful, 'false' otherwise
     */
    bool getMessageAttachments(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage);

    /**
     * MessageData::getMessageValidity()
     * Fills the validity of message
     * @param aMessageData           message data
     * @param aErrorCode             On return contains error code
     * @param aErrorMessage          On return contains error message
     * @return                       'true' if operation was successful, 'false' otherwise
     */
    bool getMessageValidity(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage);

    /**
     * MessageData::getMessagePriority()
     * Fills the validity of message
     * @param aMessageData           message data
     * @param aErrorCode             On return contains error code
     * @param aErrorMessage          On return contains error message
     * @return                       'true' if operation was successful, 'false' otherwise
     */
    bool getMessagePriority(const QVariantMap& aMessageData, int& aErrorCode, QString& aErrorMessage);

    /**
     * MessageData::checkValidNumber()
     * Checks if the number provided for send/startEditor API is valid
     * @param aNumber                string representing the recepient number
     * @return                       true if the number is valid OR false if invalid
     */
    bool checkValidNumber( const QString& aNumber ) const;

    /**
     * MessageData::checkAttachmentValid()
     * Checks if the attachment file is valid
     * @param aUri                   string representing the attachment uri
     * @param aErrorCode             On return contains error code
     * @param aErrorMessage          On return contains error message
     * @return                       'true' if the attachment valid, 'false' otherwise
     */
    bool checkAttachmentValid(const QString& aUri, int& aErrorCode, QString& aErrorMessage) const;

private:
    QString iType;
    QString iBody;
    QStringList iRecipients;
    QStringList iCcRecipients; // for MMS/Email
    QStringList iBccRecipients;// for MMS/Email
    QStringList iAttachments;  // for MMS/Email
    QString iSubject;          // for MMS/Email
    int iValidityMins;         // message validity in minutes
    int iPriority;             // message priority

    bool iStartEditor;
};

#endif
