/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QtDBus/QtDBus>
#include <QDBusConnection>
#include <hildon-thumbnail/hildon-thumbnail-factory.h>

#include "thumbnailerfetch.h"

#define THUMBSERVICE        "org.freedesktop.thumbnailer"
#define THUMBPATH           "/org/freedesktop/thumbnailer/Generic"
#define THUMBIFACE          "org.freedesktop.thumbnailer.Generic"

#define THUMBTASKFINISHED   "Finished"
#define THUMBTASKERROR      "Error"



ThumbnailerFetch::ThumbnailerFetch()
{
    /*We need to connect to thumbnailer interface once only. Reconnect has been
      happened when object is created
      */
    isConnect = true;
    isConnect = isConnect && QDBusConnection::sessionBus().connect(THUMBSERVICE,
                                                                   THUMBPATH,
                                                                   THUMBIFACE,
                                                                   THUMBTASKERROR,
                                                                   this,
                                                                   SLOT(onThumbError(quint32, const QStringList&, qint32, const QString&)));
    isConnect = isConnect && QDBusConnection::sessionBus().connect(THUMBSERVICE,
                                                                   THUMBPATH,
                                                                   THUMBIFACE,
                                                                   THUMBTASKFINISHED,
                                                                   this,
                                                                   SLOT(onThumbFinished(quint32)));
    if(!isConnect)
    {
        qDebug()<<Q_FUNC_INFO<<" Interface error";
        emit error(QString("Unknown Error. Couldn't connect to Thumbnailer iface"), QUrl());
    }
    m_lastThumbTransactId = 0;
}

ThumbnailerFetch::~ThumbnailerFetch()
{
}

void ThumbnailerFetch::request(const QList<QUrl> &urlList,const QStringList &mimetypes, int height, int width, bool needScaling)
{
    qDebug()<<Q_FUNC_INFO<<" START . isConnect = "<<isConnect;

    if(!isConnect)
    {
        qDebug()<<Q_FUNC_INFO<<" Interface error";
        emit error(QString("Unknown Error. Couldn't connect to Thumbnailer iface"), QUrl());
        return;
    }

    if(urlList.isEmpty())
    {
        emit error(QString("Unknown Error. URI list is empty"), QUrl());
        return;
    }

    if( !needScaling )
    {
        height = 124;
        width = 124;
    }

    m_sourcePath=urlList.at(0);
    //Get thumbnail uri
    m_thumbnailPath=QUrl(hildon_thumbnail_get_uri (m_sourcePath.toEncoded().data(), height, width, true));

    QDBusInterface iface(THUMBSERVICE, THUMBPATH, THUMBIFACE,QDBusConnection::sessionBus());
    if( !iface.isValid())
    {
        emit error(QString("Unknown Error, iface is'nt valid"), QUrl());
        return;
    }

    QList<QVariant> arg;
    arg.append(QStringList(m_sourcePath.toString()));
    arg.append(mimetypes);
    arg.append((quint32)0);

    QDBusMessage reply=iface.callWithArgumentList(QDBus::Block,QString("Queue"),arg);
    if( reply.type() != QDBusMessage::ReplyMessage)
    {
        emit error(QString("Unknown Error, reply type is error"), QUrl());
        return;
    }
    if(!reply.arguments().isEmpty() && reply.arguments().at(0).type() == QVariant::UInt)
    {
        m_lastThumbTransactId=reply.arguments().at(0).toUInt();
    }
    else
    {
        qDebug()<<Q_FUNC_INFO<<" reply error";
        emit error(QString("Unknown Error"), QUrl());
        return;
    }

}


void ThumbnailerFetch::onThumbError(quint32 thumbTransactId, const QStringList& /*imageList*/, qint32 /*errorValue*/, const QString& description)
{
    if(thumbTransactId==m_lastThumbTransactId)
    {
        qDebug()<<"ThumbnailerFetch::onThumbError. description = "<< description;
        m_lastThumbTransactId = 0;
        emit error(description, m_sourcePath);
    }
}

void ThumbnailerFetch::onThumbFinished(quint32 thumbTransactId)
{
    if(thumbTransactId==m_lastThumbTransactId)
    {
        qDebug()<<"ThumbnailerFetch::onThumbFinished. m_thumbnailPath = "<<m_thumbnailPath;
        m_lastThumbTransactId = 0;
        emit thumbnail(m_sourcePath, m_thumbnailPath,QString("something"));
    }
}
