/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef GETTHUMBNAILTASK_H
#define GETTHUMBNAILTASK_H

#include <QRunnable>
#include <QObject>
#include <QUrl>
#include <QMutex>
#include <QWaitCondition>
#include "thumbnailerfetch.h"




class ResizeImageTask;

//#define THUMBNAIL_THREAD_ENABLED

/**
 * Class representing get thumbnail task. Provides methods to run task asynchronously in separate thread
 * @lib libmediaserviceplugin.so
 * @since x.y
 */
class GetThumbnailTask : public QObject, public QRunnable
{
    Q_OBJECT

public:
    /**
     * Constructor
     * @param aTransactionId transaction identifier representing asynchronous request
     * @param aUrl URL for the media file for getting thumbnail
     * @param aWidth Required width of resulting thumbnail
     * @param aWidth Required height of resulting thumbnail
     * @since x.y
     */
    GetThumbnailTask(int aTransactionId, QUrl aUrl, int aWidth, int aHeight);

    ~GetThumbnailTask();

    /** method invoked by QThreadPool to perform task in separate thread*/
    void run();

    void asyncRun();
    void finishTask();

public slots:
    /**
     * Slot for DuiThumbnailer::thumbnail signal representing successful getting of thumbnail
     * @param resourceUri Uri representing path to thumbnail file
     * @emits signalGetThumbnailCb with NO_ERROR error code if returned thumbnail Uri is valid
     * @since x.y
     */
    void thumbnailReady(QUrl resourceUri, QUrl thumbnailUri,  QString flavor);

    /**
     * Slot for Thumbnailer::error signal representing error during getting of thumbnail
     * @param resourceUri Uri representing path to original file
     * @param error Error message
     * @emits signalGetThumbnailCb with NOT_SUPPORTED_ERR error code
     * @since x.y
     */
    void handleThumbnailError(QString error, QUrl resourceUri);


    /**Resize thumbnail complete slot*/
    void slotResizeImage(ResizeImageTask* aTask, QString aUri, int aErrorCode, int aTransactionId);

Q_SIGNALS:
    /**
     * Get thumbnail complete signal
     * @param aUri Uri representing path to thumbnail image file
     * @param aErrorCode Error code
     * @param aTransactionId Transaction idetnifier of asynchronous request
     * @since x.y
     */
    void signalGetThumbnailCb(GetThumbnailTask *,QUrl aUri, int aErrorCode, int aTransactionId);

private slots:
    void doRunSlot();

Q_SIGNALS:
    void asyncRunSignal();

private:
    bool needScaling() const;
private:
    int iTransactionId;
    QUrl iSourceUri;
    int iWidth;
    int iHeight;
    ThumbnailerFetch *iThumbnailer;
    QMutex iMutex;
    QMutex iMutex2;
    QWaitCondition iWait;
};
 
#endif // GETTHUMBNAILTASK_H
